package repos

import (
	"context"
	"errors"
	"fmt"
	"time"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
)

const (
	projectTagSnapshotTableName  = "project_tag_snapshots"
	hostProjectSnapshotTableName = "host_project_snapshots"
	hostStateEventTableName      = "host_state_events"
	hostStateSnapshotTableName   = "host_state_snapshots"
)

type ProjectTagSnapshot struct {
	Project   ProjectID
	Tag       string
	Timestamp time.Time
}

type HostProjectSnapshot struct {
	FQDN      HostName
	Project   ProjectID
	Timestamp time.Time
}

type HostStateEvent struct {
	FQDN      HostName
	Timestamp time.Time
	Data      []byte
}

type HostStateSnapshot struct {
	FQDN      HostName
	Timestamp time.Time
	Data      []byte
}

type StatRepo struct {
	client *db.YDBClient
}

func NewStatRepo(client *db.YDBClient) *StatRepo {
	return &StatRepo{client: client}
}

func (repo *StatRepo) InsertProjectTagSnapshots(ctx context.Context, snapshots []*ProjectTagSnapshot) error {
	if len(snapshots) == 0 {
		return nil
	}
	rows := make([]ydb.Value, len(snapshots))
	for i, snapshot := range snapshots {
		rows[i] = ydb.StructValue(
			ydb.StructFieldValue("project", ydb.UTF8Value(string(snapshot.Project))),
			ydb.StructFieldValue("tag", ydb.UTF8Value(snapshot.Tag)),
			ydb.StructFieldValue("time", ydb.TimestampValueFromTime(snapshot.Timestamp)))
	}
	return repo.client.BulkUpsert(ctx, projectTagSnapshotTableName, ydb.ListValue(rows...))
}

func (repo *StatRepo) InsertHostProjectSnapshots(ctx context.Context, snapshots []*HostProjectSnapshot) error {
	if len(snapshots) == 0 {
		return nil
	}
	rows := make([]ydb.Value, len(snapshots))
	for i, snapshot := range snapshots {
		rows[i] = ydb.StructValue(
			ydb.StructFieldValue("fqdn", ydb.UTF8Value(string(snapshot.FQDN))),
			ydb.StructFieldValue("project", ydb.UTF8Value(string(snapshot.Project))),
			ydb.StructFieldValue("time", ydb.TimestampValueFromTime(snapshot.Timestamp)))
	}
	return repo.client.BulkUpsert(ctx, hostProjectSnapshotTableName, ydb.ListValue(rows...))
}

func (repo *StatRepo) FindHostStateEvents(
	ctx context.Context,
	fqdn HostName,
	notEarlier time.Time,
	notLater time.Time,
) ([]*HostStateEvent, error) {

	query := fmt.Sprintf(`--!syntax_v1
		DECLARE $fqdn AS Utf8;
		DECLARE $not_earlier AS Timestamp;
		DECLARE $not_later AS Timestamp;

		SELECT time, data FROM %s
		WHERE fqdn = $fqdn
		AND time >= $not_earlier AND time < $not_later
		ORDER BY time ASC;
	`, hostStateEventTableName)

	res, err := repo.client.ExecuteReadQuery(ctx, query,
		table.ValueParam("$fqdn", ydb.UTF8Value(string(fqdn))),
		table.ValueParam("$not_earlier", ydb.TimestampValueFromTime(notEarlier)),
		table.ValueParam("$not_later", ydb.TimestampValueFromTime(notLater)))
	if err != nil {
		return nil, err
	}
	defer res.Close()

	if !res.NextResultSet(ctx) {
		return nil, errors.New("no data from table")
	}
	var events []*HostStateEvent
	for res.NextRow() {
		event := &HostStateEvent{FQDN: fqdn}
		if err := res.ScanWithDefaults(&event.Timestamp, &event.Data); err != nil {
			return nil, err
		}
		events = append(events, event)
	}
	return events, nil
}

func (repo *StatRepo) InsertHostStateEvents(ctx context.Context, events []*HostStateEvent) error {
	if len(events) == 0 {
		return nil
	}
	rows := make([]ydb.Value, len(events))
	for i, event := range events {
		rows[i] = ydb.StructValue(
			ydb.StructFieldValue("fqdn", ydb.UTF8Value(string(event.FQDN))),
			ydb.StructFieldValue("time", ydb.TimestampValueFromTime(event.Timestamp)),
			ydb.StructFieldValue("data", ydb.JSONDocumentValueFromBytes(event.Data)))
	}
	return repo.client.BulkUpsert(ctx, hostStateEventTableName, ydb.ListValue(rows...))
}

func (repo *StatRepo) FindLastHostStateSnapshot(ctx context.Context, fqdn HostName) (time.Time, error) {
	query := fmt.Sprintf(`--!syntax_v1
		DECLARE $fqdn AS Utf8;

		SELECT time FROM %s
		WHERE fqdn = $fqdn
		ORDER BY time DESC LIMIT 1;
	`, hostStateSnapshotTableName)

	var t time.Time
	res, err := repo.client.ExecuteReadQuery(ctx, query, table.ValueParam("$fqdn", ydb.UTF8Value(string(fqdn))))
	if err != nil {
		return t, err
	}
	defer res.Close()

	if !res.NextResultSet(ctx) {
		return t, errors.New("no data from host state snapshot table")
	}
	if !res.NextRow() {
		return t, nil
	}
	return t, res.ScanWithDefaults(&t) // todo: possible timestamp bug, see juggler/internal/db/scanners
}

func (repo *StatRepo) FindHostStateSnapshots(ctx context.Context,
	fqdn HostName,
	notEarlier time.Time,
	notLater time.Time,
) ([]*HostStateSnapshot, error) {

	query := fmt.Sprintf(`--!syntax_v1
		DECLARE $fqdn AS Utf8;
		DECLARE $not_earlier AS Timestamp;
		DECLARE $not_later AS Timestamp;

		SELECT time, data FROM %s
		WHERE fqdn = $fqdn AND time >= $not_earlier AND time < $not_later
		ORDER BY time ASC;
	`, hostStateSnapshotTableName)

	res, err := repo.client.ExecuteReadQuery(ctx, query,
		table.ValueParam("$fqdn", ydb.UTF8Value(string(fqdn))),
		table.ValueParam("$not_earlier", ydb.TimestampValueFromTime(notEarlier)),
		table.ValueParam("$not_later", ydb.TimestampValueFromTime(notLater)))
	if err != nil {
		return nil, err
	}
	defer res.Close()
	if !res.NextResultSet(ctx) {
		return nil, errors.New("no data from host state snapshot table")
	}
	var snaps []*HostStateSnapshot
	for res.NextRow() {
		snap := &HostStateSnapshot{FQDN: fqdn}
		// todo: possible timestamp bug, see juggler/internal/db/scanners
		if err := res.ScanWithDefaults(&snap.Timestamp, &snap.Data); err != nil {
			return nil, err
		}
		snaps = append(snaps, snap)
	}
	return snaps, nil
}

func (repo *StatRepo) InsertHostStateSnapshots(ctx context.Context, snapshots []*HostStateSnapshot) error {
	if len(snapshots) == 0 {
		return nil
	}
	rows := make([]ydb.Value, len(snapshots))
	for i, snapshot := range snapshots {
		rows[i] = ydb.StructValue(
			ydb.StructFieldValue("fqdn", ydb.UTF8Value(string(snapshot.FQDN))),
			ydb.StructFieldValue("time", ydb.TimestampValueFromTime(snapshot.Timestamp)),
			ydb.StructFieldValue("data", ydb.JSONDocumentValueFromBytes(snapshot.Data)))
	}
	return repo.client.BulkUpsert(ctx, hostStateSnapshotTableName, ydb.ListValue(rows...))
}
