package statistics

import (
	"encoding/json"
	"errors"
	"fmt"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/convert"
)

var ErrUnknownHostStateEventGroup = errors.New("unknown host state event group")

type HostStateEventGroup string
type HostStateEventSource string

const (
	HostChangeStatusEventGroup HostStateEventGroup = "change_status"
	HostHealthProcEventGroup   HostStateEventGroup = "health_proc"

	HostStateEventCronSource HostStateEventSource = "cron"
)

type HostStateEvent struct {
	Group  HostStateEventGroup  `json:"group"`
	Source HostStateEventSource `json:"source"`
	Data   json.RawMessage      `json:"data"`
}

type HostChangeStatusEventData struct {
	State  string `json:"state"`
	Status string `json:"status"`
}

type HostHealthProcEventData struct {
	Step string `json:"step"`
}

func NewHostChangeStatusEvent(state string, status string) HostStateEvent {
	return HostStateEvent{
		Group:  HostChangeStatusEventGroup,
		Source: HostStateEventCronSource,
		Data: convert.MustMarshalJSON(HostChangeStatusEventData{
			State:  state,
			Status: status,
		}),
	}
}

// unused
func NewHostHealthProcEvent(step string) HostStateEvent {
	return HostStateEvent{
		Group:  HostHealthProcEventGroup,
		Source: HostStateEventCronSource,
		Data: convert.MustMarshalJSON(HostHealthProcEventData{
			Step: step,
		}),
	}
}

// todo: just use oneof in protobuf and let it do all the dispatching
func (e *HostStateEvent) AsChangeStatus() (HostChangeStatusEventData, error) {
	var out HostChangeStatusEventData
	if e.Group != HostChangeStatusEventGroup {
		return out, fmt.Errorf("invalid event type: got %s instead of %s", e.Group, HostChangeStatusEventGroup)
	}
	if err := json.Unmarshal(e.Data, &out); err != nil {
		return out, err
	}
	return out, nil
}

func (e *HostStateEvent) AsHealthProc() (HostHealthProcEventData, error) {
	var out HostHealthProcEventData
	if e.Group != HostHealthProcEventGroup {
		return out, fmt.Errorf("invalid event type: got %s instead of %s", e.Group, HostHealthProcEventGroup)
	}
	if err := json.Unmarshal(e.Data, &out); err != nil {
		return out, err
	}
	return out, nil
}
