package statistics

import (
	"encoding/json"
	"time"
)

type HostStateEventStat struct {
	Group HostStateEventGroup `json:"group"`
	Data  json.RawMessage     `json:"data"`
}

type HostStateSnapshot struct {
	Stat []HostStateEventStat `json:"stat,omitempty"`
}

type hostEventStatGroupMaker interface {
	Empty() bool
	Keep(time time.Time)
	getData() interface{}
}

type hostChangeStatusEventStatMaker struct {
	Data map[string]map[string]int64
	prev *HostChangeStatusEventData
	time time.Time
}

type hostHealthProcEventStatMaker struct {
	Data map[string]int64
	prev *HostHealthProcEventData
	time time.Time
}

type HostStateSnapshotMaker struct {
	makers map[HostStateEventGroup]hostEventStatGroupMaker
	time   time.Time
}

func NewHostStateSnapshotMaker() HostStateSnapshotMaker {
	return HostStateSnapshotMaker{
		makers: map[HostStateEventGroup]hostEventStatGroupMaker{
			HostChangeStatusEventGroup: &hostChangeStatusEventStatMaker{
				Data: make(map[string]map[string]int64),
			},
			HostHealthProcEventGroup: &hostHealthProcEventStatMaker{
				Data: make(map[string]int64),
			},
		},
	}
}

func (mk *HostStateSnapshotMaker) SetTime(time time.Time) {
	mk.time = time
}

func (mk *HostStateSnapshotMaker) ProcessChangeStatusEvent(e HostChangeStatusEventData) {
	if maker, ok := mk.makers[HostChangeStatusEventGroup].(*hostChangeStatusEventStatMaker); ok {
		maker.Process(e, mk.time)
	}
}

func (mk *HostStateSnapshotMaker) ProcessHealthProcEvent(e HostHealthProcEventData) {
	if maker, ok := mk.makers[HostHealthProcEventGroup].(*hostHealthProcEventStatMaker); ok {
		maker.Process(e, mk.time)
	}
}

func (mk *HostStateSnapshotMaker) Make() ([]HostStateEventStat, error) {
	var stats []HostStateEventStat

	for group, maker := range mk.makers {
		if maker.Empty() {
			continue
		}
		maker.Keep(mk.time)
		data, err := json.Marshal(maker.getData())
		if err != nil {
			return nil, err
		}
		stats = append(stats, HostStateEventStat{
			Group: group,
			Data:  data,
		})
	}

	return stats, nil
}

func (m *hostChangeStatusEventStatMaker) Empty() bool {
	return m.prev == nil
}

func (m *hostChangeStatusEventStatMaker) Keep(time time.Time) {
	duration := time.Sub(m.time)
	if stateM, ok := m.Data[m.prev.State]; ok {
		stateM[m.prev.Status] += duration.Milliseconds()
	} else {
		m.Data[m.prev.State] = map[string]int64{m.prev.Status: duration.Milliseconds()}
	}
}

func (m *hostChangeStatusEventStatMaker) Process(e HostChangeStatusEventData, time time.Time) {
	if m.prev != nil {
		m.Keep(time)
	}
	m.prev = &e
	m.time = time
}

func (m *hostChangeStatusEventStatMaker) getData() interface{} {
	return m.Data
}

func (m *hostHealthProcEventStatMaker) Empty() bool {
	return m.prev == nil
}

func (m *hostHealthProcEventStatMaker) Keep(time time.Time) {
	duration := time.Sub(m.time)
	m.Data[m.prev.Step] += duration.Milliseconds()
}

func (m *hostHealthProcEventStatMaker) Process(e HostHealthProcEventData, time time.Time) {
	if m.prev != nil {
		m.Keep(time)
	}
	m.prev = &e
	m.time = time
}

func (m *hostHealthProcEventStatMaker) getData() interface{} {
	return m.Data
}
