package job

import (
	"context"
	"encoding/json"
	"errors"
	"fmt"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/convert"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/cron"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
	"a.yandex-team.ru/infra/walle/server/go/internal/statistics"
	"a.yandex-team.ru/infra/walle/server/go/internal/utilities"
	lockrepo "a.yandex-team.ru/infra/walle/server/go/internal/utilities/repository"
	"a.yandex-team.ru/library/go/core/log"
)

const (
	hostEventSnapshotJobLockID       = "cron/host-event-snapshot-job/project"
	hostEventSnapshotJobMaxQueueSize = 100
)

type HostEventSnapshotJob struct {
	locker            utilities.Locker
	hosts             HostRepo
	projects          ProjectRepo
	stat              StatRepo
	relevanceDuration time.Duration
	logger            log.Logger
}

func NewHostEventSnapshotJob(config *cron.JobConfig) (cron.Job, error) {
	lockRepo := lockrepo.NewLockRepository(config.Store.YDB)
	locker := utilities.NewLocker(lockRepo)
	var duration time.Duration
	switch raw := config.Extra["relevance_duration"].(type) {
	case string:
		var err error
		duration, err = time.ParseDuration(raw)
		if err != nil {
			return nil, err
		}
	default:
		return nil, errors.New("invalid 'relevance_duration' field")
	}

	return &HostEventSnapshotJob{
		locker:            locker,
		hosts:             repos.NewHostRepo(config.Store.MongoDB, config.Store.MongoReadPref),
		projects:          repos.NewProjectRepo(config.Store.MongoDB, config.Store.MongoReadPref),
		stat:              repos.NewStatRepo(config.Store.YDB),
		relevanceDuration: duration,
		logger:            config.Logger,
	}, nil
}

func (job *HostEventSnapshotJob) Do(ctx context.Context) (executed bool, err error) {
	executed = true
	now := time.Now()
	lockOwner := uuid.Must(uuid.NewV4()).String()

	projectStrIds, err := job.projects.FindFieldValues(ctx, &repos.ProjectFilter{}, repos.ProjectFieldKeyID)
	var projectIds []repos.ProjectID
	for _, projectStrID := range projectStrIds {
		projectIds = append(projectIds, repos.ProjectID(projectStrID))
	}
	if err != nil {
		return executed, err
	}
	for {
		var notDone []repos.ProjectID
		for _, id := range projectIds {
			success, err := job.handleProject(ctx, id, lockOwner, now)
			if err != nil {
				switch {
				case errors.Is(err, context.Canceled):
					return executed, err
				default:
					job.logger.Errorf("HostEventSnapshotJob: handle project: %v", err)
				}
			} else if !success {
				notDone = append(notDone, id)
			}
		}

		if len(notDone) == 0 {
			break
		}
		projectIds = notDone
		select {
		case <-ctx.Done():
			return executed, ctx.Err()
		case <-time.After(5 * time.Minute):
		}
	}

	return executed, nil
}

func (job *HostEventSnapshotJob) handleProject(
	ctx context.Context,
	project repos.ProjectID,
	lockOwner string,
	now time.Time,
) (bool, error) {
	lock, err := job.locker.Lock(job.logger, fmt.Sprintf("%s/%s", hostEventSnapshotJobLockID, project), lockOwner)
	if err != nil {
		return false, err
	}
	if lock == nil {
		return false, nil
	}
	defer job.locker.Unlock(lock)

	hosts, err := job.hosts.FindCommon(ctx, &repos.HostFilter{Project: project})
	if err != nil {
		return false, err
	}
	events := make([]*repos.HostStateEvent, hostEventSnapshotJobMaxQueueSize)
	snapshots := make([]*repos.HostStateSnapshot, hostEventSnapshotJobMaxQueueSize)
	i := 0

	for _, host := range hosts {
		events[i], snapshots[i] = &repos.HostStateEvent{}, &repos.HostStateSnapshot{}
		skip, err := job.handleHost(ctx, host.Name, host.State, host.Status, now, events[i], snapshots[i])
		if err != nil {
			return false, err
		} else if skip {
			continue
		}

		if i++; i == hostEventSnapshotJobMaxQueueSize {
			if err := job.stat.InsertHostStateEvents(ctx, events); err != nil {
				return false, err
			} else if err := job.stat.InsertHostStateSnapshots(ctx, snapshots); err != nil {
				return false, err
			}
			i = 0
		}
	}

	if err := job.stat.InsertHostStateEvents(ctx, events[:i]); err != nil {
		return false, err
	}
	if err := job.stat.InsertHostStateSnapshots(ctx, snapshots[:i]); err != nil {
		return false, err
	}
	return true, nil
}

func (job *HostEventSnapshotJob) handleHost(
	ctx context.Context,
	fqdn repos.HostName,
	state string,
	status string,
	now time.Time,
	rawEvent *repos.HostStateEvent,
	rawSnapshot *repos.HostStateSnapshot) (bool, error) {

	prevSnapshotTime, err := job.stat.FindLastHostStateSnapshot(ctx, fqdn)

	if err != nil {
		return false, err
	}
	if prevSnapshotTime.After(now.Add(-job.relevanceDuration)) {
		return true, nil
	}
	rawEvent.FQDN = fqdn
	rawEvent.Timestamp = now
	rawEvent.Data = convert.MustMarshalJSON(statistics.NewHostChangeStatusEvent(state, status))

	rawSnapshot.FQDN = fqdn
	rawSnapshot.Timestamp = now
	rawSnapshot.Data = []byte("{}")

	if prevSnapshotTime.IsZero() {
		return false, nil
	}

	maker := statistics.NewHostStateSnapshotMaker()
	maker.SetTime(prevSnapshotTime)
	events, err := job.stat.FindHostStateEvents(ctx, fqdn, prevSnapshotTime, now)
	if err != nil {
		return false, err
	}
	for _, rawEv := range events {
		var event statistics.HostStateEvent
		if err := json.Unmarshal(rawEv.Data, &event); err != nil {
			job.logger.Errorf("HostEventSnapshotJob: unmarshal state event: %v", err)
			continue
		}
		maker.SetTime(rawEv.Timestamp)
		// fixme: protobuf oneof
		switch event.Group {
		case statistics.HostChangeStatusEventGroup:
			if v, err := event.AsChangeStatus(); err != nil {
				job.logger.Errorf("HostEventSnapshotJob: unmarshal status change event group: %v", err)
			} else {
				maker.ProcessChangeStatusEvent(v)
			}
		case statistics.HostHealthProcEventGroup:
			if v, err := event.AsHealthProc(); err != nil {
				job.logger.Errorf("HostEventSnapshotJob: unmarshal health proc event group: %v", err)
			} else {
				maker.ProcessHealthProcEvent(v)
			}
		default:
			job.logger.Errorf("HostEventSnapshotJob: invalid group type %s", event.Group)
		}
	}

	maker.SetTime(now)
	stat, err := maker.Make()
	if err != nil {
		return false, err
	}
	rawSnapshot.Data = convert.MustMarshalJSON(statistics.HostStateSnapshot{Stat: stat})
	return false, nil
}
