package job

import (
	"context"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/collections"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/cron"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
	"a.yandex-team.ru/infra/walle/server/go/internal/utilities"
	lockrepo "a.yandex-team.ru/infra/walle/server/go/internal/utilities/repository"
	"a.yandex-team.ru/library/go/core/log"
)

const (
	hostProjectSnapshotJobLockID = "cron/host-project-snapshot-job"
)

type HostProjectSnapshotJob struct {
	locker   utilities.Locker
	hosts    HostRepo
	projects ProjectRepo
	stat     StatRepo
	logger   log.Logger
}

func NewHostProjectSnapshotJob(config *cron.JobConfig) (cron.Job, error) {
	lockRepo := lockrepo.NewLockRepository(config.Store.YDB)
	locker := utilities.NewLocker(lockRepo)
	return &HostProjectSnapshotJob{
		locker:   locker,
		hosts:    repos.NewHostRepo(config.Store.MongoDB, config.Store.MongoReadPref),
		projects: repos.NewProjectRepo(config.Store.MongoDB, config.Store.MongoReadPref),
		stat:     repos.NewStatRepo(config.Store.YDB),
		logger:   config.Logger,
	}, nil
}

func (job *HostProjectSnapshotJob) Do(ctx context.Context) (executed bool, err error) {
	now := time.Now()
	lockOwner := uuid.Must(uuid.NewV4())

	lock, err := job.locker.Lock(job.logger, hostProjectSnapshotJobLockID, lockOwner.String())
	if err != nil {
		return executed, err
	}
	if lock == nil {
		return executed, nil
	}
	defer job.locker.UnlockAt(lock, cron.NextTimeFromContext(ctx))
	executed = true

	projectIds, err := job.projects.FindFieldValues(ctx, &repos.ProjectFilter{}, repos.ProjectFieldKeyID)
	if err != nil {
		return executed, err
	}

	for _, id := range projectIds {
		if err := job.handleProject(ctx, repos.ProjectID(id), now); err != nil {
			return executed, err
		}
	}

	return executed, nil
}

func (job *HostProjectSnapshotJob) handleProject(ctx context.Context, project repos.ProjectID, now time.Time) error {
	hosts, err := job.hosts.FindCommon(ctx, &repos.HostFilter{Project: project})
	if err != nil {
		return err
	}
	snapshots := make([]*repos.HostProjectSnapshot, len(hosts))
	for i, host := range hosts {
		snapshots[i] = &repos.HostProjectSnapshot{
			FQDN:      host.Name,
			Project:   project,
			Timestamp: now,
		}
	}
	for _, chunk := range collections.SplitIntoChunks(len(snapshots), 1000) {
		if err := job.stat.InsertHostProjectSnapshots(ctx, snapshots[chunk.From:chunk.To]); err != nil {
			return err
		}
	}
	return nil
}
