package job

import (
	"context"
	"time"

	"github.com/gofrs/uuid"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/collections"
	"a.yandex-team.ru/infra/walle/server/go/internal/lib/cron"
	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
	"a.yandex-team.ru/infra/walle/server/go/internal/utilities"
	lockrepo "a.yandex-team.ru/infra/walle/server/go/internal/utilities/repository"
	"a.yandex-team.ru/library/go/core/log"
)

const (
	projectTagSnapshotJobLockID = "cron/project-tag-snapshot-job"
)

type ProjectTagSnapshotJob struct {
	locker   utilities.Locker
	projects ProjectRepo
	stat     StatRepo
	logger   log.Logger
}

func NewProjectTagSnapshotJob(config *cron.JobConfig) (cron.Job, error) {
	lockRepo := lockrepo.NewLockRepository(config.Store.YDB)
	locker := utilities.NewLocker(lockRepo)
	return &ProjectTagSnapshotJob{
		locker:   locker,
		projects: repos.NewProjectRepo(config.Store.MongoDB, config.Store.MongoReadPref),
		stat:     repos.NewStatRepo(config.Store.YDB),
		logger:   config.Logger,
	}, nil
}

func (job *ProjectTagSnapshotJob) Do(ctx context.Context) (executed bool, err error) {
	now := time.Now()
	lockOwner := uuid.Must(uuid.NewV4())

	lock, err := job.locker.Lock(job.logger, projectTagSnapshotJobLockID, lockOwner.String())
	if err != nil {
		return executed, err
	}
	if lock == nil {
		return executed, nil
	}
	defer job.locker.UnlockAt(lock, cron.NextTimeFromContext(ctx))
	executed = true

	projects, err := job.projects.FindCommon(ctx, &repos.ProjectFilter{})
	if err != nil {
		return executed, err
	}
	var objects []*repos.ProjectTagSnapshot
	for _, p := range projects {
		for _, tag := range p.Tags {
			// fixme: this storage format seems very inefficient, why can't we just store the array?
			objects = append(objects, &repos.ProjectTagSnapshot{
				Project:   p.ID,
				Tag:       tag,
				Timestamp: now,
			})
		}
	}

	for _, chunk := range collections.SplitIntoChunks(len(objects), 1000) {
		if err := job.stat.InsertProjectTagSnapshots(ctx, objects[chunk.From:chunk.To]); err != nil {
			return executed, err
		}
	}
	return executed, nil
}
