package job

import (
	"context"
	"encoding/json"
	"fmt"
	"sort"
	"testing"
	"time"

	"github.com/stretchr/testify/mock"
	"github.com/stretchr/testify/suite"

	"a.yandex-team.ru/infra/walle/server/go/internal/repos"
	"a.yandex-team.ru/infra/walle/server/go/internal/statistics"
	mockrepo "a.yandex-team.ru/infra/walle/server/go/internal/statistics/job/mocks"
	"a.yandex-team.ru/infra/walle/server/go/internal/utilities"
	mockutil "a.yandex-team.ru/infra/walle/server/go/internal/utilities/mocks"
)

type SnapshotJobSuite struct {
	suite.Suite
	locker      *mockutil.Locker
	projects    []*repos.Project
	projectRepo *mockrepo.ProjectRepo
	hostRepo    *mockrepo.HostRepo
}

func (suite *SnapshotJobSuite) SetupSuite() {
	for i := 0; i < 10; i++ {
		id := repos.ProjectID(fmt.Sprintf("project-%d", i))
		tags := []string{fmt.Sprintf("tag-%d", i), fmt.Sprintf("tag-%d", i+1)}
		suite.projects = append(suite.projects, &repos.Project{ID: id, Tags: tags})
	}
	suite.locker = &mockutil.Locker{}
	suite.locker.On("Lock", mock.Anything, mock.Anything, mock.Anything).Return(&utilities.LockObject{}, nil)
	suite.locker.On("Unlock", mock.Anything).Return()
	suite.locker.On("UnlockAt", mock.Anything, mock.Anything).Return()
	suite.projectRepo = &mockrepo.ProjectRepo{}
	suite.projectRepo.
		On("FindFieldValues", mock.Anything, mock.Anything, repos.ProjectFieldKeyID).
		Return(
			func(context.Context, *repos.ProjectFilter, string) []string {
				var values []string
				for _, item := range suite.projects {
					values = append(values, string(item.ID))
				}
				return values
			},
			func(context.Context, *repos.ProjectFilter, string) error {
				return nil
			},
		)
	var hosts []*repos.Host
	for i := 0; i < 20; i++ {
		hosts = append(hosts, &repos.Host{
			Name:    repos.HostName(fmt.Sprintf("host-%d", i)),
			State:   fmt.Sprintf("state-%d", i%2),
			Status:  fmt.Sprintf("status-%d", i%2),
			Project: repos.ProjectID(fmt.Sprintf("project-%d", i/2)),
		})
	}
	suite.hostRepo = &mockrepo.HostRepo{}
	suite.hostRepo.
		On("FindCommon", mock.Anything, mock.Anything).
		Return(
			func(ctx context.Context, filter *repos.HostFilter) []*repos.HostCommon {
				var res []*repos.HostCommon
				for _, item := range hosts {
					if item.Project == filter.Project {
						res = append(res, &repos.HostCommon{
							Name:   item.Name,
							Status: item.Status,
							State:  item.State,
						})
					}
				}
				return res
			},
			func(context.Context, *repos.HostFilter) error {
				return nil
			},
		)
}

func (suite *SnapshotJobSuite) TestProjectTagSnapshotJob_Do() {
	projectRepo := &mockrepo.ProjectRepo{}
	projectRepo.
		On("FindCommon", mock.Anything, mock.Anything).
		Return(
			func(context.Context, *repos.ProjectFilter) []*repos.ProjectCommon {
				var projects []*repos.ProjectCommon
				for _, item := range suite.projects {
					projects = append(projects, &repos.ProjectCommon{ID: item.ID, Tags: item.Tags})
				}
				return projects
			},
			func(context.Context, *repos.ProjectFilter) error {
				return nil
			},
		)
	statRepo := &mockrepo.StatRepo{}
	var inserted []*repos.ProjectTagSnapshot
	statRepo.
		On("InsertProjectTagSnapshots", mock.Anything, mock.Anything).
		Return(func(ctx context.Context, snapshots []*repos.ProjectTagSnapshot) error {
			inserted = append(inserted, snapshots...)
			return nil
		})
	job := &ProjectTagSnapshotJob{
		locker:   suite.locker,
		projects: projectRepo,
		stat:     statRepo,
	}
	_, err := job.Do(context.Background())
	suite.Require().NoError(err)
	var snaps []*repos.ProjectTagSnapshot
	for i := 0; i < 10; i++ {
		id := repos.ProjectID(fmt.Sprintf("project-%d", i))
		tags := []string{fmt.Sprintf("tag-%d", i), fmt.Sprintf("tag-%d", i+1)}
		snaps = append(snaps, &repos.ProjectTagSnapshot{Project: id, Tag: tags[0]})
		snaps = append(snaps, &repos.ProjectTagSnapshot{Project: id, Tag: tags[1]})
	}
	suite.Equal(len(snaps), len(inserted))
	newSortFunc := func(slice []*repos.ProjectTagSnapshot) func(i, j int) bool {
		return func(i, j int) bool {
			if slice[i].Project > slice[j].Project {
				return true
			}
			if slice[i].Project == slice[j].Project {
				return slice[i].Tag < slice[j].Tag
			}
			return false
		}
	}
	sort.Slice(snaps, newSortFunc(snaps))
	sort.Slice(inserted, newSortFunc(inserted))

	for i, snap := range snaps {
		suite.Equal(snap.Project, inserted[i].Project)
		suite.Equal(snap.Tag, inserted[i].Tag)
	}
}

func (suite *SnapshotJobSuite) TestHostProjectSnapshotJob_Do() {
	statRepo := &mockrepo.StatRepo{}
	var inserted []*repos.HostProjectSnapshot
	statRepo.
		On("InsertHostProjectSnapshots", mock.Anything, mock.Anything).
		Return(func(ctx context.Context, snapshots []*repos.HostProjectSnapshot) error {
			inserted = append(inserted, snapshots...)
			return nil
		})
	job := &HostProjectSnapshotJob{
		projects: suite.projectRepo,
		stat:     statRepo,
		locker:   suite.locker,
		hosts:    suite.hostRepo,
	}
	_, err := job.Do(context.Background())
	suite.Require().NoError(err)
	var snaps []*repos.HostProjectSnapshot
	for i := 0; i < 20; i++ {
		snaps = append(
			snaps,
			&repos.HostProjectSnapshot{
				FQDN:    repos.HostName(fmt.Sprintf("host-%d", i)),
				Project: repos.ProjectID(fmt.Sprintf("project-%d", i/2)),
			},
		)
	}
	suite.Equal(len(snaps), len(inserted))
	newSortFunc := func(slice []*repos.HostProjectSnapshot) func(i, j int) bool {
		return func(i, j int) bool {
			return slice[i].FQDN < slice[j].FQDN
		}
	}
	sort.Slice(snaps, newSortFunc(snaps))
	sort.Slice(inserted, newSortFunc(inserted))

	for i, snap := range snaps {
		suite.Equal(snap.Project, inserted[i].Project)
		suite.Equal(snap.FQDN, inserted[i].FQDN)
	}
}

func (suite *SnapshotJobSuite) TestHostEventSnapshotJob_Do() {

	statRepo := &mockrepo.StatRepo{}
	var insertedEvents []*repos.HostStateEvent
	statRepo.
		On("InsertHostStateEvents", mock.Anything, mock.Anything).
		Return(func(ctx context.Context, events []*repos.HostStateEvent) error {
			insertedEvents = append(insertedEvents, events...)
			return nil
		})
	var insertedSnapshots []*repos.HostStateSnapshot
	statRepo.
		On("InsertHostStateSnapshots", mock.Anything, mock.Anything).
		Return(func(ctx context.Context, snapshots []*repos.HostStateSnapshot) error {
			insertedSnapshots = append(insertedSnapshots, snapshots...)
			return nil
		})
	statRepo.
		On("FindLastHostStateSnapshot", mock.Anything, mock.Anything).
		Return(
			func(ctx context.Context, fqdn repos.HostName) time.Time {
				for _, item := range insertedSnapshots {
					if item.FQDN == fqdn {
						return item.Timestamp
					}
				}
				return time.Time{}
			},
			func(ctx context.Context, fqdn repos.HostName) error {
				return nil
			},
		)
	statRepo.
		On("FindHostStateEvents", mock.Anything, mock.Anything, mock.Anything, mock.Anything).
		Return(
			func(ctx context.Context, fqdn repos.HostName, notEarlier, notLater time.Time) []*repos.HostStateEvent {
				var res []*repos.HostStateEvent
				for _, item := range insertedEvents {
					if item.FQDN == fqdn &&
						(item.Timestamp.After(notEarlier) || item.Timestamp.Equal(notEarlier)) &&
						item.Timestamp.Before(notLater) {
						res = append(res, item)
					}
				}
				return res
			},
			func(ctx context.Context, fqdn repos.HostName, notEarlier, notLater time.Time) error {
				return nil
			},
		)
	statRepo.
		On("FindHostStateSnapshots", mock.Anything, mock.Anything, mock.Anything, mock.Anything).
		Return(
			func(ctx context.Context, fqdn repos.HostName, notEarlier, notLater time.Time) []*repos.HostStateSnapshot {
				var res []*repos.HostStateSnapshot
				for _, item := range insertedSnapshots {
					if item.FQDN == fqdn &&
						(item.Timestamp.After(notEarlier) || item.Timestamp.Equal(notEarlier)) &&
						item.Timestamp.Before(notLater) {
						res = append(res, item)
					}
				}
				return res
			},
			func(ctx context.Context, fqdn repos.HostName, notEarlier, notLater time.Time) error {
				return nil
			},
		)

	job := &HostEventSnapshotJob{
		projects:          suite.projectRepo,
		stat:              statRepo,
		locker:            suite.locker,
		hosts:             suite.hostRepo,
		relevanceDuration: 0,
	}
	time0 := time.Now()
	_, err := job.Do(context.Background())
	suite.Require().NoError(err)

	time1 := time.Now()
	snaps, _ := job.stat.FindHostStateSnapshots(context.Background(), "host-1", time.Time{}, time1)
	suite.Equal(1, len(snaps))
	var stat1 statistics.HostStateSnapshot
	suite.Require().NoError(json.Unmarshal(snaps[0].Data, &stat1))
	suite.Equal(statistics.HostStateSnapshot{}, stat1)

	events, _ := job.stat.FindHostStateEvents(context.Background(), "host-1", time.Time{}, time1)
	suite.Equal(1, len(events))
	var event statistics.HostStateEvent
	suite.Require().NoError(json.Unmarshal(events[0].Data, &event))
	var eventData statistics.HostChangeStatusEventData
	suite.Require().NoError(json.Unmarshal(event.Data, &eventData))
	suite.Equal(statistics.HostChangeStatusEventData{State: "state-1", Status: "status-1"}, eventData)

	_, err = job.Do(context.Background())
	suite.Require().NoError(err)

	time2 := time.Now()
	snaps, _ = job.stat.FindHostStateSnapshots(context.Background(), "host-1", time1, time2)
	suite.Equal(1, len(snaps))
	var stat2 statistics.HostStateSnapshot
	suite.Require().NoError(json.Unmarshal(snaps[0].Data, &stat2))
	suite.Equal(1, len(stat2.Stat))
	suite.Equal(statistics.HostChangeStatusEventGroup, stat2.Stat[0].Group)
	snapData := map[string]map[string]int64{
		"state-1": {"status-1": 0},
	}
	suite.Require().NoError(json.Unmarshal(stat2.Stat[0].Data, &snapData))
	elapsed := time.Duration(snapData["state-1"]["status-1"]) * time.Millisecond
	time3 := time.Now()
	suite.True(elapsed < time3.Sub(time0) && elapsed > 0)

	events, _ = job.stat.FindHostStateEvents(context.Background(), "host-1", time1, time2)
	suite.Equal(1, len(events))
	suite.Require().NoError(json.Unmarshal(events[0].Data, &event))
	suite.Require().NoError(json.Unmarshal(event.Data, &eventData))
	suite.Equal(statistics.HostChangeStatusEventData{State: "state-1", Status: "status-1"}, eventData)
}

func TestSnapshotJobs(t *testing.T) {
	suite.Run(t, new(SnapshotJobSuite))
}
