package utilities

import (
	"context"
	"sync"
	"time"

	"a.yandex-team.ru/infra/walle/server/go/internal/utilities/repository"
	"a.yandex-team.ru/library/go/core/log"
)

const (
	LockDuration        = 5 * time.Minute
	LockRefreshDuration = 30 * time.Second
)

type Locker interface {
	Lock(logger log.Logger, id, owner string) (*LockObject, error)
	Unlock(obj *LockObject)
	UnlockAt(obj *LockObject, t *time.Time)
}

type dbLocker struct {
	repo repository.LockRepo
}

func NewLocker(repo repository.LockRepo) Locker {
	return &dbLocker{
		repo: repo,
	}
}

type LockObject struct {
	lock *lock
}

type lock struct {
	id        string
	owner     string
	logger    log.Logger
	shutdown  chan *time.Time
	stopped   chan struct{}
	state     sync.Mutex
	unlocking bool
}

func (lock *lock) handleSetLockResult(success bool, err error) {
	if err != nil {
		lock.logger.Errorf("lock: %s: %s: %v", lock.id, lock.owner, err)
	} else if !success {
		lock.logger.Errorf("lock: %s: %s: fail to refresh lock: locked by other", lock.id, lock.owner)
	}
}

func (l *dbLocker) Lock(logger log.Logger, id, owner string) (*LockObject, error) {
	var success bool
	var err error
	logger.Debugf("try set lock on %s", id)
	defer func() {
		logger.Debugf("setting lock on %s: %v", id, success)
	}()
	if success, err = l.setDBLock(id, owner); err != nil {
		return nil, err
	} else if !success {
		return nil, nil
	}
	lock := &lock{
		id:       id,
		owner:    owner,
		logger:   logger,
		shutdown: make(chan *time.Time),
		stopped:  make(chan struct{}),
	}
	go l.refresh(lock)
	return &LockObject{lock}, nil
}

func (l *dbLocker) Unlock(obj *LockObject) {
	l.unlock(obj, nil)
}

func (l *dbLocker) UnlockAt(obj *LockObject, t *time.Time) {
	l.unlock(obj, t)
}

func (l *dbLocker) unlock(obj *LockObject, t *time.Time) {
	if obj != nil && obj.lock != nil {
		defer obj.lock.logger.Debugf("lock from %s removed", obj.lock.id)
		obj.lock.state.Lock()
		defer obj.lock.state.Unlock()
		if !obj.lock.unlocking {
			obj.lock.unlocking = true
			obj.lock.shutdown <- t
			<-obj.lock.stopped
		}
	}
}

func (l *dbLocker) refresh(lock *lock) {
	defer func() {
		lock.stopped <- struct{}{}
	}()
	for {
		select {
		case <-time.After(LockRefreshDuration):
			lock.handleSetLockResult(l.setDBLock(lock.id, lock.owner))
		case t := <-lock.shutdown:
			if t != nil {
				lock.handleSetLockResult(l.setDBLockUntil(lock.id, lock.owner, t))
				return
			}
			if err := l.removeDBLock(lock.id, lock.owner); err != nil {
				lock.logger.Errorf("remove lock: %s: %s: %v", lock.id, lock.owner, err)
			}
			return
		}
	}
}

func (l *dbLocker) setDBLock(id, owner string) (bool, error) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*15)
	defer cancel()
	return l.repo.TryLock(ctx, id, owner, time.Now().Add(LockDuration))
}

func (l *dbLocker) removeDBLock(id, owner string) error {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*15)
	defer cancel()
	return l.repo.Unlock(ctx, id, owner)
}

func (l *dbLocker) setDBLockUntil(id, owner string, t *time.Time) (bool, error) {
	ctx, cancel := context.WithTimeout(context.Background(), time.Second*15)
	defer cancel()
	return l.repo.TryLock(ctx, id, owner, *t)
}
