package repository

import (
	"context"
	"errors"
	"fmt"
	"time"

	"a.yandex-team.ru/infra/walle/server/go/internal/lib/db"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb"
	"a.yandex-team.ru/kikimr/public/sdk/go/ydb/table"
)

const locksTableName = "locks"

type LockRepo interface {
	TryLock(ctx context.Context, id string, owner string, until time.Time) (bool, error)
	Unlock(ctx context.Context, id string, owner string) error
	FindOwners(ctx context.Context, filter *LockFilter) ([]string, error)
	Delete(ctx context.Context, id string) error
}

type ydbLockRepo struct {
	client *db.YDBClient
}

func NewLockRepository(client *db.YDBClient) LockRepo {
	return &ydbLockRepo{client: client}
}

func (repo *ydbLockRepo) TryLock(ctx context.Context, id string, owner string, until time.Time) (bool, error) {
	query := fmt.Sprintf(`--!syntax_v1
		DECLARE $id AS Utf8;
		DECLARE $new_owner AS Utf8;
		DECLARE $locked_until AS Timestamp;
		DECLARE $now AS Timestamp;

		$new_data = (
			SELECT
				$id AS id,
				$new_owner AS locked_by,
				$locked_until AS locked_until
		);
		$upsert_data = (
			SELECT
				new.id AS id,
				new.locked_by AS locked_by,
				new.locked_until AS locked_until
			FROM $new_data AS new
			LEFT JOIN %s AS existing
			ON new.id = existing.id
			WHERE existing.id IS NULL
				OR existing.locked_by IS NULL
				OR existing.locked_by = $new_owner
				OR existing.locked_until < $now
		);
		UPSERT INTO %s
		SELECT * FROM $upsert_data;
		SELECT COUNT(*) as modified_count FROM $upsert_data;
	`, locksTableName, locksTableName)
	success := false

	err := repo.client.Retry(ctx, func(c context.Context, session *table.Session) error {
		_, res, err := session.Execute(c, repo.client.WriteTxControl, query, table.NewQueryParameters(
			table.ValueParam("$id", ydb.UTF8Value(id)),
			table.ValueParam("$new_owner", ydb.UTF8Value(owner)),
			table.ValueParam("$locked_until", ydb.TimestampValueFromTime(until)),
			table.ValueParam("$now", ydb.TimestampValueFromTime(time.Now()))),
			db.EnableQueryCache())
		if err != nil {
			return err
		}
		defer res.Close()

		if !res.NextResultSet(c) || !res.NextRow() {
			return errors.New("no data from lock table")
		}

		var count uint64
		if err := res.Scan(&count); err != nil {
			return err
		}
		success = count == 1
		return nil
	})
	return success, err
}

func (repo *ydbLockRepo) Unlock(ctx context.Context, id string, owner string) error {
	query := fmt.Sprintf(`--!syntax_v1
		DECLARE $id AS Utf8;
		DECLARE $owner AS Utf8;

		UPDATE %s
		SET locked_by = NULL, locked_until = NULL
		WHERE id = $id AND locked_by = $owner
	`, locksTableName)

	return repo.client.ExecuteWriteQuery(ctx, query,
		table.ValueParam("$id", ydb.UTF8Value(id)),
		table.ValueParam("$owner", ydb.UTF8Value(owner)))
}

type LockFilter struct {
	Prefix      string
	LockedUntil time.Time
}

func (repo *ydbLockRepo) FindOwners(ctx context.Context, filter *LockFilter) ([]string, error) {
	query := fmt.Sprintf(`--!syntax_v1
		DECLARE $prefix AS Utf8;
		DECLARE $locked_until AS Timestamp;

		SELECT locked_by FROM %s
		WHERE id LIKE $prefix
		AND locked_until >= $locked_until;
	`, locksTableName)
	res, err := repo.client.ExecuteReadQuery(
		ctx,
		query,
		table.ValueParam("$prefix", ydb.UTF8Value(filter.Prefix+"%")),
		table.ValueParam("$locked_until", ydb.TimestampValueFromTime(filter.LockedUntil)),
	)
	if err != nil {
		return nil, err
	}
	defer res.Close()

	if !res.NextResultSet(ctx) {
		return nil, errors.New("no data from table")
	}
	var owners []string
	for res.NextRow() {
		var owner string
		if err := res.ScanWithDefaults(&owner); err != nil {
			return nil, err
		}
		owners = append(owners, owner)
	}
	return owners, nil
}

func (repo *ydbLockRepo) Delete(ctx context.Context, id string) error {
	query := fmt.Sprintf(`--!syntax_v1
		DECLARE $id AS Utf8;

		DELETE FROM %s
		WHERE id = $id
	`, locksTableName)
	return repo.client.ExecuteWriteQuery(ctx, query, table.ValueParam("$id", ydb.UTF8Value(id)))
}
