import os
import platform
import signal
import subprocess
import sys
import tarfile
import time

import pymongo
import pymongo.errors
import yatest.common
from library.python.testing.recipe import declare_recipe, set_env
from library.recipes.common import find_free_ports


def log(msg):
    print(msg)
    sys.stdout.flush()


CONFIG_TEMPLATE = """
storage:
  dbPath: {0}/mongodb
  journal:
    enabled: true
systemLog:
  destination: file
  logAppend: true
  path: {0}/mongod.log
  quiet: true
net:
  port: {1}
  bindIp: 127.0.0.1
  ipv6: true
"""


def start(argv):
    if 'RECIPE_MONGO_PORT' in os.environ:
        set_env('SKIP_CLEANUP_MONGO', '1')
        log('Using preconfigured mongo')
        return

    (port,) = find_free_ports(count=1)
    log('Will use port {}'.format(port))
    run_mongodb(port)
    set_env("RECIPE_MONGO_PORT", str(port))


def stop(argv):
    if 'SKIP_CLEANUP_MONGO' in os.environ:
        return
    log('Stopping mongodb...')
    try:
        with open("mongod.pid") as f:
            pid = int(f.read())
            os.kill(pid, signal.SIGTERM)
    except Exception as e:
        log('Termination error (ignored): {}'.format(e))


def _is_mongo_alive(pid, port):
    try:
        os.kill(pid, 0)
    except OSError:
        return False

    with pymongo.MongoClient(port=port, serverSelectionTimeoutMS=500) as client:
        try:
            client.admin.command('ping')
            return True
        except pymongo.errors.ConnectionFailure:
            return False


if platform.system() == "Linux":
    DIST_DIR = "mongodb-linux-x86_64-3.4.23"
    DIST_ARC = "mongodb-linux-x86_64-3.4.23.tgz"
elif platform.system() == "Darwin":
    DIST_DIR = "mongodb-osx-x86_64-3.4.24"
    DIST_ARC = "mongodb-osx-ssl-x86_64-3.4.24.tgz"


def log_files(d):
    try:
        log(f"Dir '{d}' content: {os.listdir(d)}")
    except Exception:
        log(f"Can't get dir '{d}' content")


def run_mongodb(port):
    test_work_path = yatest.common.work_path()
    config_path = os.path.join(test_work_path, "mongod.conf")

    os.mkdir(os.path.join(test_work_path, "mongodb"))

    with open(config_path, "w") as stream:
        stream.write(CONFIG_TEMPLATE.format(test_work_path, port))

    with tarfile.open(
        yatest.common.binary_path(os.path.join("infra/walle/server/recipes/mongodb/pkg", DIST_ARC))
    ) as stream:
        stream.extractall()

    custom_mongo_run_script = os.getenv("WALLE_TEST_MONGO_RUN_SCRIPT")
    if custom_mongo_run_script:
        command = [custom_mongo_run_script]
    else:
        command = [
            os.path.join(DIST_DIR, "bin/mongod"),
            "--traceExceptions",
            "--nounixsocket",
            "--nohttpinterface",
            "--nojournal",
        ]

    command += ["--pidfilepath", "{}/mongod.pid".format(test_work_path), "--config", config_path]

    log('Starting mongodb...')
    child = subprocess.Popen(command, stdout=subprocess.PIPE, stderr=subprocess.STDOUT)

    deadline = time.time() + 60.0
    started = False
    while time.time() < deadline:
        if not _is_mongo_alive(child.pid, port):
            time.sleep(0.1)
        else:
            started = True
            break
    if not started:
        log("MONGO FAILED STDOUT" + "=" * 20)
        try:
            child.kill()
        except Exception as exc:
            log(f"can't kill mongo child process: {str(exc)}")
        child_stdout, _child_stderr = child.communicate()
        log(child_stdout)
        log("=" * 20)
        log_files(DIST_DIR)
        log_files(os.path.join(DIST_DIR, "bin"))
        log_files(test_work_path)
        mongo_log_path = os.path.join(test_work_path, "mongod.log")
        with open(mongo_log_path) as mongo_log:
            log(f"'{mongo_log_path}' content:\n{mongo_log.read()}")
        raise Exception("Mongo failed to start")


if __name__ == "__main__":
    declare_recipe(start, stop)
