#!/bin/bash

pexit()
{
    echo "$@"
    exit -1
}

perr()
{
    echo "ERROR: $@"
    exit -1
}

check_file_path()
{
    PATH=$1
    if [ -z ${PATH} ]; then
        pexit "You should specify path!"
    fi

    if [ ! -e ${PATH} ]; then
        pexit "${PATH} does not exists!"
    fi

    if [ -d ${PATH} ]; then
        pexit "${PATH} is directory!"
    fi

    if [ ! -r ${PATH} ]; then
        pexit "${PATH} is not readable!"
    fi

    if [ ! -x ${PATH} ]; then
        pexit "${PATH} is not executable!"
    fi

    echo "${PATH}"
}

check_dir_path()
{
    PATH=$1
    if [ -z ${PATH} ]; then
        pexit "You should specify path!"
    fi

    if [ ! -e ${PATH} ]; then
        pexit "${PATH} does not exists!"
    fi

    if [ ! -d ${PATH} ]; then
        pexit "${PATH} is not a directory!"
    fi

    if [ ! -r ${PATH} ]; then
        pexit "${PATH} is not readable!"
    fi

    if [ ! -x ${PATH} ]; then
        pexit "${PATH} is not executable!"
    fi

    echo "${PATH}"
}

is_master()
{
    "${MONGO}" --host=localhost --port="${PORT}" --quiet --username "${USER}" --password "${PASSWORD}" --authenticationDatabase "${AUTH_DATABASE}" --eval 'db.isMaster().ismaster'
}

clean_backups()
{
    CURR_BACKUPS_COUNT="$(ls -1t "${B_DIR}" | egrep -c "^${BACKUP_PREFIX}")"
    if [ ${CURR_BACKUPS_COUNT} -gt ${BACKUPS_KEEP} ] ; then
        TO_REMOVE=$(( ${CURR_BACKUPS_COUNT} - ${BACKUPS_KEEP} ))
        echo ${TO_REMOVE}

        for BACKUP in $(ls -1t "${B_DIR}" | egrep "^${BACKUP_PREFIX}" | tail -n ${TO_REMOVE}); do
            echo "Removing old ${BACKUP}"
            rm -fr "${B_DIR}/${BACKUP}"

            if [ $? -ne 0 ]; then
                perr "Could not remove old file ${BACKUP}!"
            fi
        done
    fi
}

clean_backups_on_s3_mds()
{
    PYTHON="/skynet/python/bin/python"
    S3_MDS="$(dirname -- $0)/s3-mds"
    S3_BACKUP_PREFIX="${HOST}/${BACKUP_PREFIX}"
    CURR_BACKUPS_COUNT="$($PYTHON "${S3_MDS}" -T -k "${S3_KEY_ID}" -s "${S3_SECRET_KEY}" list-names "${S3_BUCKET}" | egrep -c "^${S3_BACKUP_PREFIX}")"
    if [ ${CURR_BACKUPS_COUNT} -gt ${BACKUPS_KEEP} ] ; then
        TO_REMOVE=$(( ${CURR_BACKUPS_COUNT} - ${BACKUPS_KEEP} ))
        $PYTHON "${S3_MDS}" -T -k "${S3_KEY_ID}" -s "${S3_SECRET_KEY}" list-names "${S3_BUCKET}" | egrep "^${S3_BACKUP_PREFIX}" | sort -t"|" -k3 -r | tail -n ${TO_REMOVE} | cut -f 1 -d"|" | while read BACKUP; do
            echo "Removing old key ${BACKUP} from S3 bucket ${S3_BUCKET}"
            $PYTHON "${S3_MDS}" -T -k "${S3_KEY_ID}" -s "${S3_SECRET_KEY}" remove "${S3_BUCKET}" "${BACKUP}"

            if [ $? -ne 0 ]; then
                perr "Could not remove old key ${BACKUP} from S3 bucket ${S3_BUCKET}!"
            fi
        done
    fi
}

upload_to_s3()
{
    BACKUP_FILE="${1}"
    S3_BACKUP_KEY="${HOST}/$(basename -- ${BACKUP_FILE})"
    PYTHON="/skynet/python/bin/python"
    S3_MDS="$(dirname -- $0)/s3-mds"
    ${PYTHON} "${S3_MDS}" -T -k "${S3_KEY_ID}" -s "${S3_SECRET_KEY}" upload "${S3_BUCKET}" "${S3_BACKUP_KEY}" "${BACKUP_FILE}"

    if [ $? -ne 0 ]; then
        perr "Could not upload ${BACKUP_FILE} to S3 bucket ${S3_BUCKET} with key ${S3_BACKUP_KEY}!"
    fi
}

while getopts "u:p:H:P:d:a:R:b:r:k:K:S:B:h" optname ; do
    case "$optname" in
        u)
            USER="${OPTARG}"
        ;;
        p)
            PASSWORD="${OPTARG}"
        ;;
        H)
            HOST="${OPTARG}"
        ;;
        P)
            PORT="${OPTARG}"
        ;;
        d)
            DATABASE="${OPTARG}"
        ;;
        a)
            AUTH_DATABASE="${OPTARG}"
        ;;
        R)
            RS_NAME="${OPTARG}"
        ;;
        b)
            BACKUP_DIR="${OPTARG}"
        ;;
        r)
            BACKUP_PREFIX="${OPTARG}"
        ;;
        k)
            BACKUPS_KEEP="${OPTARG}"
        ;;
        K)
            S3_KEY_ID="${OPTARG}"
        ;;
        S)
            S3_SECRET_KEY="${OPTARG}"
        ;;
        B)
            S3_BUCKET="${OPTARG}"
        ;;
        h|*)
            echo "`basename $0` -- simple mongodb backup script for mongodb running under Nanny Services"
            echo "usage: `basename $0` -u \$USER -p \$PASSWORD -P \$PORT -d \$DATABASE -a \$AUTH_DATABASE -b \$BACKUP_DIR -r \$BACKUP_PREFIX -k \$BACKUPS_KEEP"
            echo "    options:"
            echo "        -u \$USER - specify mongodb user"
            echo "        -p \$PASSWORD - specify mongodb password"
            echo "        -H \$HOST - specify localhost's name"
            echo "        -P \$PORT - specify mongodb port"
            echo "        -d \$DATABASE - specify database"
            echo "        -a \$AUTH_DATABASE - specify authentication database"
            echo "        -R \$RS_NAME - specify replicaset name"
            echo "        -b \$BACKUP_DIR - specify authentication database"
            echo "        -r \$BACKUP_PREFIX - specify backup name prefix"
            echo "        -k \$BACKUPS_KEEP - specify how many backups should be keeped"
            echo "        -K \$S3_KEY_ID - specify s3 key id"
            echo "        -S \$S3_SECRET_KEY - specify s3 secret key"
            echo "        -B \$S3_BUCKET - specify s3 bucket name"
            echo "        -h - print this message"
            exit 1
        ;;
    esac
done

MONGO="$(check_file_path "/db/bsconfig/webcache/mongodb_${PORT}/mongo")"
if [ $? -ne 0 ]; then
    perr "${MONGO}"
fi

MONGODUMP="$(check_file_path "/db/bsconfig/webcache/mongodb_${PORT}/mongodump")"
if [ $? -ne 0 ]; then
    perr "${MONGODUMP}"
fi

B_DIR="$(check_dir_path "${BACKUP_DIR}")"
if [ $? -ne 0 ]; then
    perr "${B_DIR}"
fi

NOW="$(/bin/date +%s)"

ismaster="$(is_master)"

if [ "${ismaster}" = "false" ] ; then
    clean_backups
    clean_backups_on_s3_mds
    echo "Creating backup"
    BACKUP_FILE="${BACKUP_DIR}/${BACKUP_PREFIX}_${NOW}.arz"
    "${MONGODUMP}" --host=localhost --port="${PORT}" -u "${USER}" -p "${PASSWORD}" --authenticationDatabase "${AUTH_DATABASE}" --oplog --gzip --archive="${BACKUP_FILE}"
    upload_to_s3 "${BACKUP_FILE}"
    curl -si -H 'Content-Type: application/json' -d '{"events": [{"host": "'${HOST}'.search.yandex.net", "instance": "'${PORT}'", "service": "mongo-rs-'${RS_NAME}'-backup", "status": "OK", "tags": ["mongodb","wall-e.srv","wall-e-healthdb"], "description": "Backup for '${RS_NAME}' replica set from '${HOST}'.search.yandex.net:'${PORT}' instance completed"}]}' "http://localhost:31579/events"
else
    echo "Skipping backup: host is master!"
fi
