#!/usr/bin/env python

import gevent.monkey

gevent.monkey.patch_all(subprocess=True)

import argparse
import re
from sepelib.core import config
from walle.main import load_config, app
from walle.hosts import Host
from walle.audit_log import LogEntry


def parse_args():
    parser = argparse.ArgumentParser(description="Wall-E private data cleaner")
    parser.add_argument("-c", "--config", help="configuration file path")
    parser.add_argument("-d", "--debug", default=False, action="store_true", help="debug mode")
    action = parser.add_subparsers()

    clean_data = action.add_parser("clean-data")
    clean_data.add_argument("--hosts", action="store_true", default=False)
    clean_data.add_argument("--audit-log", action="store_true", default=False)
    config.augment_args_parser(parser)
    return parser.parse_args()


def host_has_private_data(host):
    for subsys, messages in host.messages.items():
        for message in messages:
            if message.message and 'PRIVATE' in message.message:
                return True
    return False


def host_private_data_modify(host):
    for subsys, messages in host.messages.items():
        for i in range(len(messages)):
            if messages[i].message and 'PRIVATE' in messages[i].message.contains:
                print('modifying host: {} {}: {}'.format(host.id, subsys, messages[i]))
                start_pos = host.messages[subsys][i]['message'].find('PRIVATE')
                end_pos = host.messages[subsys][i]['message'].rfind('PRIVATE')
                host.messages[subsys][i]['message'] = (
                    host.messages[subsys][i]['message'][:start_pos] + host.messages[subsys][i]['message'][end_pos:]
                )
    host.messages.save()


def clean_hosts(modify=False):
    hosts_with_private_data = []
    for host in Host.objects:
        if host_has_private_data(host):
            print('found host: {}: {}'.format(host.id, host.messages))
            hosts_with_private_data.append(host)

    if modify:
        for host in hosts_with_private_data:
            host_private_data_modify(host)


def clean_audit_log(modify=False):
    if modify:
        collection = LogEntry.get_collection('primary')
        update_result = collection.update_many(
            {'reason': re.compile('PRIVATE')},
            {
                '$set': {
                    'reason': 'LUI API (private data cleared) FAILED with error: 404 Client Error: NOT FOUND for url:'
                    ' https://cauth.yandex.net:4443/passwd/serveradmins?q=<fqdn>'
                }
            },
        )
        print('found {} entries, updated: {} entries'.format(update_result.modified_count, update_result.matched_count))
    else:
        collection = LogEntry.get_collection('primary')
        result = collection.find({'reason': re.compile('PRIVATE')})
        for record in result:
            print('found reocrd: {}'.format(record))


def main():
    args = parse_args()
    load_config(args)
    app.setup_logging()

    with app.as_context():
        print('cleaning up hosts')
        clean_hosts(args.hosts)

        print('cleaning up audit_log')
        clean_audit_log(args.audit_log)


if __name__ == '__main__':
    main()
