#! /usr/bin/env python

import re
import sys

reload(sys)
sys.setdefaultencoding('utf-8')

import os
import json
import time
import shutil
import logging
import argparse
import datetime
import requests
import easywebdav
import numpy as np

import matplotlib as mpl

mpl.use('Agg')
import matplotlib.pyplot as plt

from bisect import bisect_left
from sepelib.core import config
from mongoengine import connect
from collections import OrderedDict, defaultdict

from walle.projects import Project
from walle.audit_log import LogEntry
from walle.clients.bot import get_preorder_info


WORK_DIR = 'host_graphics'
RETRY_COUNT = 10
LOGFILENAME = "{}/host-graphics.log"
JING_LOGIN = 'alexsmirnov'
WALLE_DB_NAME = 'walle'


STARTREK_TOKEN = 'OAuth {}'
WALLE_TOKEN = "OAuth {}"

STARTREK_TICKET_URL = "https://st-api.yandex-team.ru/v2/issues?query={}&page={}"
STARTREK_CHANGELOG_URL = "https://st-api.yandex-team.ru/v2/issues/{}/changelog?type=IssueWorkflow"
WALLE_AUDIT_LOG_URL = "https://api.wall-e.yandex-team.ru/v1/audit-log?start_time={}&host_inv={}"
JING_URL = "https://jing.yandex-team.ru/files/{}/".format(JING_LOGIN)
WIKI_URL = "https://wiki-api.yandex-team.ru/_api/frontend/wall-e/comp-hosts/{}/"

TAGS = ["hosts-comp:init", "hosts-comp:move", "hosts-comp:fini"]
STARTREK_STATUSES = ['createdAt', 'open', 'inProgress', 'closed']
WALLE_STATUSES = {'add-host', 'switch-project', 'prepare-host'}
GRAPHIC_STATUSES = ['createdAt', 'open', 'inProgress']
STATUS_MAP = {'createdAt': 'created', 'open': 'open', 'inProgress': 'in progress'}


PIC_ST_ALL_TAGS = "st_all_tags"
TITLE_ST_ALL_TAGS = "Тикеты: общее время на каждой стадии"

PIC_ST_ALL_PERIODS = "st_all_periods"
TITLE_ST_ALL_PERIODS = "Тикеты: общее время на каждой стадии по периодам"

PIC_ST_BY_PERIOD = "st_by_period_{}"
TITLE_ST_BY_PERIOD = "Тикеты: общее время на каждой стадии за период {}"

PIC_ST_TAG_PREFIX = "st_{}_tag_only"
TITLE_ST_TAG_PREFIX = "Тикеты: время на каждой стадии для тега {}"

PIC_ST_TOTAL_TIME_FOR_EVERY_TAG = "st_total_time_every_tag"
TITLE_ST_TOTAL_TIME_FOR_EVERY_TAG = "Тикеты: total time для каждого тега на одной картинке"

PIC_ST_COUNT_PREFIX = "st_{}_count_only"
TITLE_ST_COUNT_PREFIX = "Тикеты: время на каждой стадии для тикетов с количеством хостов X<{}"

PIC_ST_TOTAl_TIME_EVERY_COUNT = "st_total_time_every_prefix"
TITLE_ST_TOTAl_TIME_EVERY_COUNT = "Тикеты: total time для тикетов с количеством хостов каждого размера"

PIC_WALLE_DATA = 'walle_data'
TITLE_WALLE_DATA = "Хосты: сколько занимает выполнение операций в wall-e (от начального add-host/switch-project/prepare-host до конечного add-host/switch-project/prepare-host)"

PIC_REPAIR_TIME = 'repair_time'
TITLE_REPAIR_TIME = "График починки хостов по RTC c {} по {}"

PIC_BROKEN_TIME = "broken_time"
TITLE_BROKEN_TIME = "График хостов ожидающих ремонта по RTC c {} по {}"


TEMPLATE = """
{{TOC}}

===Что это===
На этой странице находятся графики ввода/перевода/вывода хостов, а также графики нахождения хостов в починке.

"""

TEMPLATE_BLOCK = """

{}
{}
"""


class CDF:
    def __init__(self, data, filter=None):
        if filter:
            self._data = self._filter_array_by(sorted(data), filter=filter)
        else:
            self._data = sorted(data)
        self._data_len = float(len(self._data))

    def __call__(self, point):
        return len(self._data[: bisect_left(self._data, point)]) / self._data_len

    def min(self):
        return int(self._data[0])

    def max(self):
        return int(self._data[-1])

    @staticmethod
    def _filter_array_by(array, filter=98):
        idx = bisect_left(array, filter)
        return array[:idx]


def load_configuration():
    def create_work_dir():
        if not os.path.isdir(WORK_DIR):
            os.mkdir(WORK_DIR)
            logging.info("Created new work dir: {}".format(WORK_DIR))

    def update_constants(args):
        global STARTREK_TOKEN, WALLE_TOKEN, RETRY_COUNT, WORK_DIR
        STARTREK_TOKEN = STARTREK_TOKEN.format(args.startrek_token)
        WALLE_TOKEN = WALLE_TOKEN.format(args.walle_token)
        if args.reconnect_count:
            RETRY_COUNT = args.reconnect_count
        if args.work_dir:
            WORK_DIR = args.work_dir

    parser = argparse.ArgumentParser(description='Create data and graphing for hosts move')

    parser.add_argument("--config", default="../conf/walle.prod.yaml", help="Config path")

    parser.add_argument("--startrek-token", required=True, help="Your startrek  OAuth token")
    parser.add_argument("--walle-token", required=True, help="Your wall-e OAuth token")
    parser.add_argument("--reconnect-count", help="Number of reconnect attempts")
    parser.add_argument("--work-dir", help="Name of working directory")

    parser.add_argument(
        "--hosts-comp", action='store_true', help="Get data about hosts init/move/fini/ from st and wall-e"
    )
    parser.add_argument("--hosts-comp-draw", action='store_true', help="Create graphing for all gotten data for comp")

    parser.add_argument(
        "--hosts-repair-data", action='store_true', help="Get data about hosts' moving from your local wall-e db"
    )
    parser.add_argument(
        "--hosts-repair-draw", action='store_true', help="Create graphing for all gotten data for repair"
    )

    parser.add_argument("--push-wiki", action='store_true', help="Send all created graphics to wiki page")
    parser.add_argument("--keep-work-dir", action='store_true', help="Keep all artefacts after finishing")

    parser.add_argument("--hosts-comp-all", action='store_true', help="Do all 'hosts-comp' actions")
    parser.add_argument("--hosts-repair-all", action='store_true', help="Do all 'hosts-repair-*' actions")

    parser.add_argument("--from-date", required=True, help="Fetch hosts-comp data starting from this date")
    parser.add_argument("--to-date", required=False, help="Fetch hosts-comp data up to this date")
    parser.add_argument("--save-dumps", action='store_true', help="Save all dumps with data from st/wall-e/db")

    args = parser.parse_args()

    if args.hosts_comp_draw and not args.hosts_comp:
        raise argparse.ArgumentError(None, "'--hosts-comp-draw' required data from '--hosts-comp-*'")
    if args.hosts_repair_draw and not args.hosts_repair_data:
        raise argparse.ArgumentError(None, "'--hosts-repair-draw' required data from '--hosts-repair-data'")

    update_constants(args)
    config.load(args.config, config_context={})
    create_work_dir()
    logging.basicConfig(filename=LOGFILENAME.format(WORK_DIR), level=logging.DEBUG, filemode="w")

    return args


def remove_work_dir():
    if os.path.isdir(WORK_DIR):
        shutil.rmtree(WORK_DIR)
        logging.info("Removed work dir: {}".format(WORK_DIR))


def transform_datetime_to_unix_time(timestamp):
    timestamp = timestamp[:-5]
    res = time.mktime(datetime.datetime.strptime(timestamp, "%Y-%m-%dT%H:%M:%S.%f").timetuple())
    return res


def draw_cdf_from_dict(data, filename, wiki_title, plot_title=None, max_xtick=None):
    plt.rcParams["figure.figsize"] = [30, 9]
    plt.grid(True)

    if max_xtick:
        plt.xticks(np.arange(0, max_xtick, 3.0))
    else:
        plt.xticks(np.arange(0, max([max(values) for values in data.values() if values]), 5.0))

    for title, values in sorted(data.iteritems()):
        if not values:
            continue
        cdf = CDF(values, filter=max_xtick)
        xvalues = range(0, cdf.max())
        yvalues = [cdf(point) for point in xvalues]
        plt.plot(xvalues, yvalues, label=title)

    plt.xlabel('time (days)')
    if plot_title:
        plt.title(plot_title)
    plt.legend()

    filename += '-{}.png'.format(time.time())
    path = "{}/{}".format(WORK_DIR, filename)
    plt.savefig(path, bbox_inches='tight')
    plt.close()

    logging.info("Draw {}".format(filename))

    return filename, path, wiki_title


def push_to_wiki(pics_info):
    global TEMPLATE
    for (filename, _, title) in pics_info:
        url = JING_URL + filename
        title = "==={}===".format(title)
        new_template_block = TEMPLATE_BLOCK.format(title, url)
        TEMPLATE += new_template_block
    data = {"title": "Графики ввода и починки хостов", "body": TEMPLATE}

    wiki_url = WIKI_URL.format(datetime.date.today().isoformat())
    requests.post(wiki_url, headers={"Authorization": WALLE_TOKEN}, json=data)
    logging.info("Pushed updated info to wiki")


def upload_to_jing(pics_info):
    for (filename, path, _) in pics_info:
        webdav = easywebdav.connect(
            'jingdav.yandex-team.ru', username='jing', password='jing', port=443, protocol="https"
        )
        webdav.upload(path, '{}/{}'.format(JING_LOGIN, filename))
        logging.info("Uploaded pic {} to jing".format(filename))


### Parse startrek ###


def parse_startrek_changelog(entry):
    changelog_url = STARTREK_CHANGELOG_URL.format(entry['id'])
    request = requests.get(changelog_url, headers={"Authorization": STARTREK_TOKEN})
    changelog = request.json()

    data = dict()
    data['createdAt'] = entry['createdAt']
    for entry in changelog:
        fields = entry['fields']
        for field in fields:
            if field['field']['id'] == 'status':
                data[field['to']['key']] = entry['updatedAt']
                break
    return data


def parse_ticket_description(description):
    splitted_desc = description.split()
    last_url = None
    preorder_ids = None
    for word in splitted_desc:
        word = word.strip('()')
        if word.startswith("https://paste.yandex-team.ru/"):
            last_url = word
        if word.startswith("https://bot.yandex-team.ru/hwr/preorders/"):
            if "filter=" in word:
                preorder_ids = word.split('id:')[1].split(',')
            else:
                preorder_ids = [word.split('preorders/')[1]]

    if preorder_ids:
        result = []
        for preorder_id in preorder_ids:
            result += get_preorder_info(preorder_id, authenticate=False)['servers'].keys()
            logging.info("Parsed preorder {} info from bot".format(preorder_id))
    else:
        if last_url:
            last_url = last_url.strip('/')
            if not last_url.endswith("/text"):
                last_url += "/text"
            info = requests.get(last_url).content
        else:
            info = description
        result = parse_sereglond_host_format(info)
        logging.info("Parsed info from sereglond copy/paste")
    return result


def parse_sereglond_host_format(info):
    info = info.strip().split('\n')
    inv_column = parse_inv_only_column(info)
    if inv_column:
        return inv_column
    inv_column = parse_inv_first_int_column_between_tabs(info)
    if inv_column:
        return inv_column
    return None


def parse_inv_only_column(info):
    result = []
    for line in info:
        try:
            inv = int(line)
            result.append(inv)
        except ValueError:
            pass
    return result


def parse_inv_first_int_column_between_tabs(info):
    result = []
    supposed_inv_idx = None
    for line in info:
        elems = [x for x in line.split("\t") if x != ""]
        if len(elems) <= 2:
            continue
        if supposed_inv_idx:
            try:
                inv = int(elems[supposed_inv_idx])
                result.append(inv)
                continue
            except ValueError:
                result = []
        for idx, elem in enumerate(elems):
            try:
                inv = int(elem)
                supposed_inv_idx = idx
                result.append(inv)
                break
            except ValueError:
                pass
    return result


def get_all_data_tags_together(data):
    STATUSES = {'created': [], 'open': [], 'in progress': [], "total": []}
    for tag in data.keys():
        for entry in data[tag]["data"]:
            logging.debug(entry)

            statuses = sorted(entry["changelog"].iteritems(), key=lambda x: x[1])
            total = 0
            for idx, status in enumerate(statuses[:-1]):
                time = (
                    transform_datetime_to_unix_time(statuses[idx + 1][1]) - transform_datetime_to_unix_time(status[1])
                ) / 86400
                total += time

                status_name = status[0]
                if status_name not in STATUS_MAP:
                    continue
                status_name = STATUS_MAP[status_name]
                STATUSES[status_name].append(time)

            STATUSES["total"].append(total)

    return {k: v for k, v in STATUSES.iteritems() if k in {"in progress", "total"}}


def get_totals_by_quarter(data):
    def get_period_label(created_date):
        create_date = datetime.date(*map(int, created_date[:10].split("-")))
        if datetime.date(2019, 9, 1) <= create_date <= datetime.date(2020, 2, 29):
            return "Manual 2019.09-2020.02"
        elif datetime.date(2020, 3, 1) <= create_date <= datetime.date(2020, 4, 6):
            return "HalfManual 2020.03-04"
        elif create_date > datetime.date(2020, 4, 6):
            return "Automated 2020.04..."
        else:
            return "Prehistoric ..2019.08"

    result = defaultdict(list)
    for tag in data.keys():
        for entry in data[tag]["data"]:
            period_label = get_period_label(entry["changelog"]["createdAt"])
            logging.debug(period_label)
            logging.debug(entry)

            statuses = sorted(entry["changelog"].iteritems(), key=lambda x: x[1])
            total = 0
            for idx, status in enumerate(statuses[:-1]):
                time = (
                    transform_datetime_to_unix_time(statuses[idx + 1][1]) - transform_datetime_to_unix_time(status[1])
                ) / 86400
                total += time

            total_key = "{} total".format(period_label)
            result[total_key].append(total)

    return result


def get_all_data_by_quarter(data):
    STATUSES = {'in progress', "total"}

    def get_period_label(created_date):
        create_date = datetime.date(*map(int, created_date[:10].split("-")))
        if datetime.date(2019, 9, 1) <= create_date <= datetime.date(2020, 2, 29):
            return "Manual 2019.09-2020.02"
        elif datetime.date(2020, 3, 1) <= create_date <= datetime.date(2020, 4, 6):
            return "HalfManual 2020.03-04"
        elif create_date > datetime.date(2020, 4, 6):
            return "Automated 2020.04..."
        else:
            return "Prehistoric ..2019.08"

    result = defaultdict(lambda: defaultdict(list))

    for tag in data.keys():
        for entry in data[tag]["data"]:
            period_label = get_period_label(entry["changelog"]["createdAt"])
            logging.debug(period_label)
            logging.debug(entry)

            statuses = sorted(entry["changelog"].iteritems(), key=lambda x: x[1])
            total = 0
            for idx, status in enumerate(statuses[:-1]):
                time = (
                    transform_datetime_to_unix_time(statuses[idx + 1][1]) - transform_datetime_to_unix_time(status[1])
                ) / 86400
                total += time

                status_name = status[0]
                if status_name not in STATUS_MAP:
                    continue

                status_name = STATUS_MAP[status_name]
                if status_name in STATUSES:
                    result[period_label][status_name].append(time)

            result[period_label]["total"].append(total)

    return result


def get_all_data_tags_separately(data):
    result = {}
    for tag in data.keys():
        STATUSES = {'created': [], 'open': [], 'in progress': [], "total": []}
        for entry in data[tag]["data"]:
            statuses = sorted(entry["changelog"].iteritems(), key=lambda x: x[1])
            total = 0
            for idx, status in enumerate(statuses[:-1]):
                status_name = status[0]
                if status_name not in STATUS_MAP:
                    continue
                status_name = STATUS_MAP[status_name]
                time = (
                    transform_datetime_to_unix_time(statuses[idx + 1][1]) - transform_datetime_to_unix_time(status[1])
                ) / 86400
                STATUSES[status_name].append(time)
                total += time
            STATUSES["total"].append(total)
        result[tag] = STATUSES
    return result


def get_all_data_tags_together_for_different_amount_of_tags(data):
    result = OrderedDict(
        [
            (10, {'created': [], 'open': [], 'in progress': [], "total": []}),
            (50, {'created': [], 'open': [], 'in progress': [], "total": []}),
            (100, {'created': [], 'open': [], 'in progress': [], "total": []}),
            (1000, {'created': [], 'open': [], 'in progress': [], "total": []}),
            (float("inf"), {'created': [], 'open': [], 'in progress': [], "total": []}),
        ]
    )
    for tag in data.keys():
        for entry in data[tag]["data"]:
            statuses = sorted(entry["changelog"].iteritems(), key=lambda x: x[1])
            total = 0
            ln = len(entry["invs"]) if entry["invs"] != None else 0
            key_size = None
            if len(statuses) > 1:
                for idx, status in enumerate(statuses[:-1]):
                    status_name = status[0]
                    if status_name not in STATUS_MAP:
                        continue
                    status_name = STATUS_MAP[status_name]
                    time = (
                        transform_datetime_to_unix_time(statuses[idx + 1][1])
                        - transform_datetime_to_unix_time(status[1])
                    ) / 86400
                    total += time
                    if not key_size:
                        for key in result:
                            if ln <= key:
                                result[key][status_name].append(time)
                                key_size = key
                                break
                    else:
                        result[key_size][status_name].append(time)
                result[key_size]['total'].append(total)
    return result


### Parse audit log ###


def get_audit_logs_for_host(inv, start_time, end_time=None):
    url = WALLE_AUDIT_LOG_URL.format(start_time, inv)
    if end_time:
        url += "&end_time={}".format(end_time)
    counter = 0
    while counter < RETRY_COUNT:
        try:
            request = requests.get(url, headers={"Authorization": WALLE_TOKEN})
            return request.json()
        except requests.exceptions.ConnectionError:
            counter += 1


def parse_audit_log(audit_log):
    result = {}
    for entry in audit_log['result']:
        inv = entry['host_inv']
        if inv not in result:
            result[inv] = {"low": float("inf"), "high": float("-inf")}

        status = entry['type']
        if status not in WALLE_STATUSES:
            continue

        time = entry['status_time']

        result[inv]["low"] = min(result[inv]["low"], time)
        result[inv]["high"] = max(result[inv]["high"], time)
    return result


def filter_hosts(data):
    result = {}
    for tag in data:
        if tag == "hosts-comp:fini":
            continue
        result[tag] = []
        for ticket in data[tag]["data"]:
            for host, times in ticket.iteritems():
                if times['high'] != float('-inf') and times['low'] != float('inf') and times['high'] != times['low']:
                    diff = (times['high'] - times['low']) / 86400
                    result[tag].append(diff)
    return result


### Parse audit log in walle db


def get_audit_log_reports(projects):
    connect(WALLE_DB_NAME)
    entries = LogEntry.objects.filter(type="report").filter(project__in=projects)
    start_date, end_date, result = float("inf"), float("-inf"), []
    for entry in entries:
        result.append(entry)
        start_date = min(entry.time, start_date)
        end_date = max(entry.time, end_date)
    start_date = datetime.datetime.fromtimestamp(start_date).strftime('%Y-%m-%d')
    end_date = datetime.datetime.fromtimestamp(end_date).strftime('%Y-%m-%d')
    return result, start_date, end_date


def parse_data(logs):
    times = []
    hosts = []
    bucket = defaultdict(list)

    for log in logs:
        inv = log.host_inv
        if log.status == "completed":
            bucket[inv].append(log)
            val = bucket.pop(inv)
            if len(val) > 1:
                time = (val[-1].time - val[0].time) / 86400
                times.append(time)
                hosts.append(inv)
        else:
            bucket[inv].append(log)
    not_fixed_hosts = [(val[-1].time - val[0].time) / 86400 for inv, val in bucket.iteritems() if len(val) > 1]
    return sorted(times), hosts, sorted(not_fixed_hosts)


def get_all_projects_with_rtc_tag():
    connect(WALLE_DB_NAME)
    entries = Project.objects.all()
    result = []
    for project in entries:
        if project.tags and 'rtc' in project.tags:
            result.append(project.id)
    return result


def get_repair_and_broken_info(save_dumps=False):
    projects = get_all_projects_with_rtc_tag()
    audit_log, start_date, end_date = get_audit_log_reports(projects)
    repair, hosts, broken = parse_data(audit_log)

    if save_dumps:
        with open("times.json", "w") as data_file:
            json.dump(repair, data_file, indent=4, sort_keys=True)
        with open("invs.json", "w") as data_file:
            json.dump(hosts, data_file, indent=4, sort_keys=True)
        with open("broken.json", "w") as data_file:
            json.dump(broken, data_file, indent=4, sort_keys=True)
    return repair, broken, start_date, end_date


def get_info_from_startrek_ticket_page(from_date, tag, page, to_date=None):
    st_bucket = []
    walle_bucket = []
    hosts_amount = 0
    tickets_with_invs = []
    tickets_without_invs = []

    query = {
        "Queue": "RUNTIMECLOUD",
        "Created": "{}..{}".format(from_date, to_date) if to_date else ">={}".format(from_date),
        "Tags": '"{}"'.format(tag),
    }
    query = " ".join(map(":".join, query.iteritems()))
    url = STARTREK_TICKET_URL.format(query, page)
    request = requests.get(url, headers={"Authorization": STARTREK_TOKEN})
    data_chunk = request.json()

    for entry in data_chunk:
        ticket_key = entry['self'].split('/')[-1]
        logging.info("Parse ticket with key: {}".format(ticket_key))

        invs = parse_ticket_description(entry['description'])
        parsed_changelog = parse_startrek_changelog(entry)
        scheme = {"ticket_key": ticket_key, "changelog": parsed_changelog, "invs": invs}
        st_bucket.append(scheme)

        if invs:
            logging.info("Found {} hosts in ticket: {}".format(len(invs), ticket_key))
            tickets_with_invs.append(ticket_key)
            hosts_amount += len(invs)
            for idx in range(0, len(invs), 200):
                start_time = transform_datetime_to_unix_time(parsed_changelog['createdAt'])
                end_time = (
                    transform_datetime_to_unix_time(parsed_changelog['closed'])
                    if 'closed' in parsed_changelog
                    else None
                )
                inv = ",".join([str(x) for x in invs[idx : min(idx + 200, len(invs) - 1)]])
                audit_log = get_audit_logs_for_host(inv, start_time, end_time)
                parsed_audit_log = parse_audit_log(audit_log)
                walle_bucket.append(parsed_audit_log)
        else:
            logging.info("Don't found hosts in ticket: {}".format(ticket_key))
            tickets_without_invs.append(ticket_key)

        logging.info("Finished parsing of ticket: {}".format(ticket_key))

    return st_bucket, walle_bucket, hosts_amount, tickets_with_invs, tickets_without_invs


def get_changelogs(from_date, to_date=None):
    tickets_with_parsed_invs = []
    tickets_without_parsed_invs = []
    startrek_changelogs = {}
    audit_logs = {}

    for tag in TAGS:
        page = 1
        amount = 0
        hosts_amount = 0
        startrek_bucket = []
        walle_bucket = []

        while True:
            (
                _st_bucket,
                _walle_bucket,
                _hosts_amount,
                tickets_with_invs,
                tickets_without_invs,
            ) = get_info_from_startrek_ticket_page(from_date, tag, page, to_date=to_date)

            startrek_bucket += _st_bucket
            walle_bucket += _walle_bucket
            hosts_amount += _hosts_amount
            tickets_with_parsed_invs += tickets_with_invs
            tickets_without_parsed_invs += tickets_without_invs

            amount += len(_st_bucket)

            page += 1

            if len(_st_bucket) == 0:
                break
            # break

        startrek_changelogs[tag] = {"data": startrek_bucket, "amount": amount}

        audit_logs[tag] = {"data": walle_bucket, "amount": hosts_amount}
    return startrek_changelogs, audit_logs, tickets_without_parsed_invs, tickets_with_parsed_invs


def parse_startrek(from_date, to_date=None, save_dumps=False):
    changelogs, hosts, tickets_without, tickets_with = get_changelogs(from_date, to_date=to_date)

    if save_dumps:
        with open("{}/changelogs.json".format(WORK_DIR), "w") as data_file:
            json.dump(changelogs, data_file, indent=4, sort_keys=True)

        with open("{}/hosts.json".format(WORK_DIR), "w") as data_file:
            json.dump(hosts, data_file, indent=4, sort_keys=True)

        with open("{}/tickets_with.json".format(WORK_DIR), "w") as data_file:
            json.dump(tickets_with, data_file, indent=4, sort_keys=True)

        with open("{}/tickets_without.json".format(WORK_DIR), "w") as data_file:
            json.dump(tickets_without, data_file, indent=4, sort_keys=True)
    return changelogs, hosts


### Draw everything ###


def draw_hosts(data):
    pics_info = []

    data = filter_hosts(data)
    pics_info.append(draw_cdf_from_dict(data, PIC_WALLE_DATA, TITLE_WALLE_DATA, plot_title="All hosts"))

    return pics_info


def draw_startrek(raw_data):
    pics_info = []

    data = get_all_data_tags_together(raw_data)
    pics_info.append(draw_cdf_from_dict(data, PIC_ST_ALL_TAGS, TITLE_ST_ALL_TAGS, plot_title="All tags", max_xtick=100))

    data = get_totals_by_quarter(raw_data)
    pics_info.append(
        draw_cdf_from_dict(data, PIC_ST_ALL_PERIODS, TITLE_ST_ALL_PERIODS, plot_title="Totals by period", max_xtick=100)
    )

    data = get_all_data_by_quarter(raw_data)
    for period, period_data in data.iteritems():
        period_tag = re.sub(r"[a-zA-Z\s.]", "", period)
        pics_info.append(
            draw_cdf_from_dict(
                period_data,
                PIC_ST_BY_PERIOD.format(period_tag),
                TITLE_ST_BY_PERIOD.format(period),
                plot_title="Totals for {}".format(period),
                max_xtick=100,
            )
        )

    data = get_all_data_tags_separately(raw_data)
    for tag, values in data.iteritems():
        pics_info.append(
            draw_cdf_from_dict(
                values, PIC_ST_COUNT_PREFIX.format(tag), TITLE_ST_TAG_PREFIX.format(tag), plot_title=tag, max_xtick=100
            )
        )

    result = {}
    for tag, values in data.iteritems():
        result[tag] = values["total"]
    pics_info.append(
        draw_cdf_from_dict(
            result,
            PIC_ST_TOTAL_TIME_FOR_EVERY_TAG,
            TITLE_ST_TOTAL_TIME_FOR_EVERY_TAG,
            plot_title="Total time for every tag",
            max_xtick=100,
        )
    )

    data = get_all_data_tags_together_for_different_amount_of_tags(raw_data)
    data.pop(float('inf'))
    for tag, values in data.iteritems():
        pics_info.append(
            draw_cdf_from_dict(
                values,
                PIC_ST_COUNT_PREFIX.format(tag),
                TITLE_ST_COUNT_PREFIX.format(tag),
                plot_title=tag,
                max_xtick=100,
            )
        )

    result = {}
    for size, values in data.iteritems():
        result[size] = values["total"]
    pics_info.append(
        draw_cdf_from_dict(
            result,
            PIC_ST_TOTAl_TIME_EVERY_COUNT,
            TITLE_ST_TOTAl_TIME_EVERY_COUNT,
            plot_title="Total time for every ticket size",
            max_xtick=100,
        )
    )
    return pics_info


def draw_repair_and_broken(repair, broken, start_date, end_date):
    pics_info = []

    repair = {'repair time': repair}
    pics_info.append(
        draw_cdf_from_dict(repair, PIC_REPAIR_TIME, TITLE_REPAIR_TIME.format(start_date, end_date), plot_title="Repair")
    )

    broken = {'broken time': broken}
    pics_info.append(
        draw_cdf_from_dict(broken, PIC_BROKEN_TIME, TITLE_BROKEN_TIME.format(start_date, end_date), plot_title="Broken")
    )

    return pics_info


if __name__ == "__main__":
    args = load_configuration()

    pics_info = []

    if args.hosts_comp or args.hosts_comp_all:
        startrek_data, walle_data = parse_startrek(args.from_date, to_date=args.to_date, save_dumps=args.save_dumps)
        if args.hosts_comp_draw or args.hosts_comp_all:
            pics_info += draw_startrek(startrek_data)
            pics_info += draw_hosts(walle_data)

    if args.hosts_repair_data or args.hosts_repair_all:
        repair, broken, start_date, end_date = get_repair_and_broken_info(args.save_dumps)
        if args.hosts_repair_draw or args.hosts_repair_all:
            pics_info += draw_repair_and_broken(repair, broken, start_date, end_date)

    if args.push_wiki:
        upload_to_jing(pics_info)
        push_to_wiki(pics_info)

    if not args.keep_work_dir:
        remove_work_dir()
