from collections import OrderedDict

import yaml
from sepelib.core import config

from walle import constants
from walle.expert.types import CheckType


def represent_ordered_dict(dumper, data):
    return dumper.represent_dict(data.items())


yaml.add_representer(OrderedDict, represent_ordered_dict, Dumper=yaml.SafeDumper)


def generate_panel():
    # panel address https://juggler.yandex-team.ru/dashboards/wall-e-bundle/

    panel = OrderedDict()
    panel["address"] = "wall-e-bundle"
    panel["name"] = "Wall-E Bundle"
    panel["description"] = "Проверки из Wall-E.Бандла, требующие внимания"
    panel["owners"] = ["dldmitry", "efmv", "alexsmirnov", "dstudentsov", "rocco66", "squirrel"]
    panel["links"] = []
    panel["components"] = []

    position = 0  # fill spots row by row
    num_cols = 3

    for check_name, check in iterate_checks():
        raw_events_options = OrderedDict()
        raw_events_options["limit"] = 5
        raw_events_options["statuses"] = ["WARN"]
        raw_events_options["offset"] = 0
        raw_events_options["selector"] = "service = {}".format(check)

        component = OrderedDict()
        component["links"] = []
        component["raw_events_options"] = raw_events_options
        component["name"] = check_name
        component["component_type"] = "RAW_EVENTS"
        component["col"] = position % num_cols + 1
        component["row"] = position // num_cols + 1
        component["colspan"] = 0
        component["rowspan"] = 0
        component["view_type"] = "DETAILED"

        panel["components"].append(component)
        position += 1

    print(yaml.safe_dump(panel, default_flow_style=False, indent=4, allow_unicode=True))


def iterate_checks():
    seen = set()

    for check in CheckType.ALL_PASSIVE:
        seen.add(check)
        if check == CheckType.META:
            continue

        yield get_walle_check_type(check).replace("_", " ").title(), check

    for section, checks in config.get_value("juggler.extra_checks").iteritems():
        for check in checks:
            if check in seen:
                continue
            seen.add(check)

            name = check.replace("_", " ")
            if name.startswith("walle "):
                name = name[len("walle ") :]

            yield name.title(), check


def get_walle_check_type(check_type):
    """Maps Expert System's check types to Wall-E check types."""

    try:
        return CheckType.WALLE_MAPPING[check_type]
    except KeyError:
        # project's custom check. may be transform it somehow
        print("Got a check that does not have name, weird...")
        exit(130)


if __name__ == "__main__":
    config.load(path='conf/walle.prod.yaml', defaults=constants.DEFAULT_CONFIG_PATH, config_context={})
    generate_panel()
