#!/usr/bin/env python

import gevent.monkey

gevent.monkey.patch_all(subprocess=True)

from walle.util.setup import setup_logging
from walle.clients.staff import is_group, resolve_owners, check_logins, get_group_members

import argparse

from sepelib.core import config
from sepelib.mongo.util import register_database

from walle.main import load_config
from walle.projects import Project


def dump_all_owners(args):
    projects = Project.objects(id__in=list(args.project_id))
    owners = {owner for p in projects for owner in p.owners}
    resolved_owners = resolve_owners(tuple(sorted(owners)), allow_robots=False)

    print("\n".join(resolved_owners))


def dump_trace_owner(args):
    traced_owner = args.owner
    bot_project_id = _get_bot_project_id(args)
    print("// tracing projects with bot project id = {}".format(bot_project_id))

    projects = Project.objects(bot_project_id=bot_project_id).only("id", "name", "owners")
    count_owned = 0
    for project in projects:
        print("// checking project {} ({})".format(project.id, project.name))
        count_owned += int(_trace_project_owner(traced_owner, project))

    if not count_owned:
        print("// user {} does not own any projects with bot project id '{}'".format(traced_owner, bot_project_id))


def _get_bot_project_id(args):
    if args.project_id:
        try:
            source_project = Project.objects.only("bot_project_id").get(id=args.project_id)
        except Project.DoesNotExist:
            print("Project {} does not exist, sorry.".format(args.project_id))
            raise SystemExit(2)
        else:
            bot_project_id = source_project.bot_project_id
    else:
        bot_project_id = args.bot_project_id

    return bot_project_id


def _trace_project_owner(traced_owner, project):
    owned = False

    project_owners = set(project.owners)
    owner_groups = set(filter(is_group, project.owners))
    owner_logins = check_logins(tuple(sorted(project_owners - owner_groups)), skip_missing=True)

    if traced_owner in owner_logins:
        print("! {} owns project '{}' ({})".format(traced_owner, project.id, project.name))
        owned = True

    for group in owner_groups:
        if traced_owner in get_group_members(group):
            print("! {} owns project '{}' ({}) via group {}".format(traced_owner, project.id, project.name, group))
            owned = True

    return owned


def main():
    args = _parse_args()
    load_config(args)
    setup_logging(config)

    register_database("mongodb")
    args.handle(args)


def _parse_args():
    parser = argparse.ArgumentParser(description="Clone one wall-e project to a new one.")
    parser.add_argument("-c", "--config", help="configuration file path")
    parser.add_argument("-d", "--debug", action="store_true", help="Set logging level to 'debug'")

    actions = parser.add_subparsers()

    get_owners = actions.add_parser("owners", description="Get list of all owners of specified projects")
    get_owners.set_defaults(handle=dump_all_owners)
    get_owners.add_argument("project_id", nargs="+", type=unicode, help="project id")

    trace_owner = actions.add_parser("trace", description="Trace how exactly user owns projects")
    trace_owner.set_defaults(handle=dump_trace_owner)
    trace_owner.add_argument("owner", type=unicode, help="owner's login")

    project_identity = trace_owner.add_mutually_exclusive_group(required=True)
    project_identity.add_argument(
        "--project-id", type=unicode, help="Trace by project id: get bot project id and follow all projects"
    )
    project_identity.add_argument("--bot-project-id", type=int, help="Trace by bot project id: follow all projects.")

    config.augment_args_parser(parser)
    return parser.parse_args()


if __name__ == "__main__":
    main()
