#!/bin/bash -x

pexit()
{
    echo "$@"
    exit -1
}

perr()
{
    echo "ERROR: $@"
    exit -1
}

check_file_path()
{
    PATH=$1
    if [ -z ${PATH} ]; then
        pexit "You should specify path!"
    fi

    if [ ! -e ${PATH} ]; then
        pexit "${PATH} does not exists!"
    fi

    if [ -d ${PATH} ]; then
        pexit "${PATH} is directory!"
    fi

    if [ ! -r ${PATH} ]; then
        pexit "${PATH} is not readable!"
    fi

    if [ ! -x ${PATH} ]; then
        pexit "${PATH} is not executable!"
    fi

    echo "${PATH}"
}

check_dir_path()
{
    PATH=$1
    if [ -z ${PATH} ]; then
        pexit "You should specify path!"
    fi

    if [ ! -e ${PATH} ]; then
        pexit "${PATH} does not exists!"
    fi

    if [ ! -d ${PATH} ]; then
        pexit "${PATH} is not a directory!"
    fi

    if [ ! -r ${PATH} ]; then
        pexit "${PATH} is not readable!"
    fi

    if [ ! -x ${PATH} ]; then
        pexit "${PATH} is not executable!"
    fi

    echo "${PATH}"
}

is_master()
{
    "${MONGO}" --host=localhost --port="${PORT}" --quiet --username "${USER}" --password "${PASSWORD}" --authenticationDatabase "${AUTH_DATABASE}" --eval 'db.isMaster().ismaster'
}


s3_mds_exec()
{
    PYTHON="/skynet/python/bin/python"
    S3_MDS="$(dirname -- $0)/s3-mds"

    ${PYTHON} ${S3_MDS} -T -k "${S3_KEY_ID}" -s "${S3_SECRET_KEY}" "$@"
}

clean_backups_on_s3_mds()
{
    S3_BACKUP_PREFIX="${HOST}/${BACKUP_PREFIX}"
    CURR_BACKUPS_COUNT="$(s3_mds_exec list-names "${S3_BUCKET}" | egrep -c "^${S3_BACKUP_PREFIX}")"

    if [ ${CURR_BACKUPS_COUNT} -gt ${BACKUPS_KEEP} ] ; then
        TO_REMOVE=$(( ${CURR_BACKUPS_COUNT} - ${BACKUPS_KEEP} ))

        s3_mds_exec list-names "${S3_BUCKET}" | egrep "^${S3_BACKUP_PREFIX}" | sort -t"|" -k3 -r | tail -n ${TO_REMOVE} | cut -f 1 -d"|" | while read BACKUP; do
            echo "Removing old key ${BACKUP} from S3 bucket ${S3_BUCKET}"
            s3_mds_exec remove "${S3_BUCKET}" "${BACKUP}"

            if [ $? -ne 0 ]; then
                perr "Could not remove old key ${BACKUP} from S3 bucket ${S3_BUCKET}!"
            fi
        done
    fi
}

find_last_local()
{
    local IFS=$'\n'
    BACKUPS_LIST=$(ls -1 ${BACKUP_DIR})

    LAST_BACKUP=""
    LAST_BACKUP_TIMESTAMP="0"

    for BACKUP in ${BACKUPS_LIST}; do
        if [[ ${BACKUP} =~ ^.*_([0-9]+).arz$ && ${BASH_REMATCH[1]} -gt "$LAST_BACKUP_TIMESTAMP" ]]; then
            LAST_BACKUP_TIMESTAMP=${BASH_REMATCH[1]}
            LAST_BACKUP=${BACKUP}
        fi
    done

    echo "${BACKUP_DIR}/${LAST_BACKUP}"
}

find_last_from_s3()
{
    local IFS=$'\n'
    BACKUPS_LIST=$(s3_mds_exec list-names "${S3_BUCKET}" | awk '{print $1}')

    LAST_BACKUP=""
    LAST_BACKUP_TIMESTAMP="0"

    for BACKUP in ${BACKUPS_LIST}; do
        if [[ ${BACKUP} =~ ^.*_([0-9]+).arz$ && ${BASH_REMATCH[1]} -gt "$LAST_BACKUP_TIMESTAMP" ]]; then
            LAST_BACKUP_TIMESTAMP=${BASH_REMATCH[1]}
            LAST_BACKUP=${BACKUP}
        fi
    done

    echo ${LAST_BACKUP}
}

fetch_from_s3()
{
    S3_BACKUP_NAME="${1}"
    BACKUP_FILE="/tmp/$(basename -- ${S3_BACKUP_NAME})"

    s3_mds_exec fetch "${S3_BUCKET}" "${S3_BACKUP_NAME}" "${BACKUP_FILE}"

    echo "${BACKUP_FILE}"
}

while getopts "u:p:H:P:d:a:R:b:r:k:K:S:B:h" optname ; do
    case "$optname" in
        u)
            USER="${OPTARG}"
        ;;
        p)
            PASSWORD="${OPTARG}"
        ;;
        H)
            HOST="${OPTARG}"
        ;;
        P)
            PORT="${OPTARG}"
        ;;
        d)
            DATABASE="${OPTARG}"
        ;;
        a)
            AUTH_DATABASE="${OPTARG}"
        ;;
        R)
            RS_NAME="${OPTARG}"
        ;;
        b)
            BACKUP_DIR="${OPTARG}"
        ;;
        r)
            BACKUP_PREFIX="${OPTARG}"
        ;;
        k)
            BACKUPS_KEEP="${OPTARG}"
        ;;
        K)
            S3_KEY_ID="${OPTARG}"
        ;;
        S)
            S3_SECRET_KEY="${OPTARG}"
        ;;
        B)
            S3_BUCKET="${OPTARG}"
        ;;
        h|*)
            echo "`basename $0` -- simple mongodb backup script for mongodb running under Nanny Services"
            echo "usage: `basename $0` -u \$USER -p \$PASSWORD -P \$PORT -d \$DATABASE -a \$AUTH_DATABASE -b \$BACKUP_DIR -r \$BACKUP_PREFIX -k \$BACKUPS_KEEP"
            echo "    options:"
            echo "        -u \$USER - specify mongodb user"
            echo "        -p \$PASSWORD - specify mongodb password"
            echo "        -H \$HOST - specify localhost's name"
            echo "        -P \$PORT - specify mongodb port"
            echo "        -d \$DATABASE - specify database"
            echo "        -a \$AUTH_DATABASE - specify authentication database"
            echo "        -R \$RS_NAME - specify replicaset name"
            echo "        -b \$BACKUP_DIR - specify authentication database"
            echo "        -r \$BACKUP_PREFIX - specify backup name prefix"
            echo "        -k \$BACKUPS_KEEP - specify how many backups should be keeped"
            echo "        -K \$S3_KEY_ID - specify s3 key id"
            echo "        -S \$S3_SECRET_KEY - specify s3 secret key"
            echo "        -B \$S3_BUCKET - specify s3 bucket name"
            echo "        -h - print this message"
            exit 1
        ;;
    esac
done

MONGO="mongo"
if [ $? -ne 0 ]; then
    perr "${MONGO}"
fi

#MONGORESTORE="$(check_file_path "/db/bsconfig/webcache/mongodb_${PORT}/mongorestore")"
MONGORESTORE="mongorestore"
if [ $? -ne 0 ]; then
    perr "${MONGORESTORE}"
fi

B_DIR="$(check_dir_path "${BACKUP_DIR}")"
if [ $? -ne 0 ]; then
    perr "${B_DIR}"
fi

NOW="$(/bin/date +%s)"

ismaster="$(is_master)"

if [ "${ismaster}" = "true" ] ; then

    S3_BACKUP=$(find_last_from_s3)
    if [[ -n "${S3_BACKUP}" ]]; then
        echo "fetching $S3_BACKUP from s3"
        BACKUP_FILE=$(fetch_from_s3 ${S3_BACKUP})
    fi

    if [[ -z "${BACKUP_FILE}" ]]; then
        echo "Can not fetch backup from s3, try to use local backup"
        BACKUP_FILE=$(find_last_local)
    fi

    if [[ -z "${BACKUP_FILE}" ]]; then
        perr "Can't find any backup to restore"
    fi

    "${MONGORESTORE}" --host=localhost --port="${PORT}" -u "${USER}" -p "${PASSWORD}" --authenticationDatabase "${AUTH_DATABASE}" --oplogReplay --gzip --archive="${BACKUP_FILE}"
else
    echo "Can not restore backup: host is not a master!"
fi
