from juggler_sdk import JugglerApi, Child, Check
from juggler_sdk.types import FlapOptions

from walle.setup_cron import setup_cron
from walle.util.cloud_tools import get_system_suffix
from infra.walle.server.scripts.walle_backend_monitoring_updater import common

SECONDS_PER_DAY = 24 * 60 * 60


class _JobInfo:
    def __init__(self, job_id, period, expected_work_time) -> None:
        self.id = job_id
        self.period = period
        self.expected_work_time = expected_work_time

    def make_juggler_host(self):
        return f"{common.make_host()}.{self.id}"


class _JobInfoCron:
    def __init__(self) -> None:
        self.jobs_info = {}

    def add_job(self, job_type, func, period, retry_period=None, expected_work_time=None, stop_handler=None):
        job_id = job_type.value
        self.jobs_info[job_id] = _JobInfo(job_id, period, expected_work_time)

    def get_jobs(self):
        return list(self.jobs_info.values())


def update_cron_juggler_aggregates(api: JugglerApi) -> list[common.UpdateResult]:
    cron = _JobInfoCron()
    setup_cron(cron)
    common_host = common.make_host()
    common_service = f"{common.CRON_JUGGLER_SERVICE}{get_system_suffix()}"
    results = []
    apply_result = api.upsert_check(
        Check(
            host=common_host,
            service=common_service,
            namespace=common.WALLE_JUGGLER_NAMESPACE,
            refresh_time=30,
            ttl=600,
            aggregator="logic_or",
            aggregator_kwargs={"nodata_mode": "force_crit"},
            tags=[common.CHAT_NOTIFICATION_TAG],
            children=[
                Child(
                    host=job.make_juggler_host(),
                    service=f"wall-e.cron.run_timeout{get_system_suffix()}",
                    instance="all",
                    group_type="HOST",
                )
                for job in cron.get_jobs()
            ],
        )
    )
    results.append(common.UpdateResult(common_host, common_service, apply_result))
    for job in cron.get_jobs():
        job_timeout = job.expected_work_time if job.expected_work_time else job.period
        if 20 * job_timeout >= 3 * SECONDS_PER_DAY:
            flaps_conf = None
        else:
            flaps_conf = FlapOptions(stable=3 * job_timeout, critical=15 * job_timeout)
        job_host = job.make_juggler_host()
        service = f"wall-e.cron.run_timeout{get_system_suffix()}"
        child_service = f"wall-e.cron.{job.id}.last_success_run{get_system_suffix()}"
        apply_result = api.upsert_check(
            Check(
                host=job_host,
                service=service,
                namespace=common.WALLE_JUGGLER_NAMESPACE,
                refresh_time=min(30, job_timeout),
                ttl=min(20 * job_timeout, 3 * SECONDS_PER_DAY),
                aggregator="logic_or",
                aggregator_kwargs={"nodata_mode": "force_crit"},
                flaps_config=flaps_conf,
                children=[Child(host=common_host, service=child_service, group_type="HOST")],
            )
        )
        results.append(common.UpdateResult(common_host, service, apply_result))
    return results
