import copy
import json
import urllib.request

from infra.walle.server.scripts.walle_backend_monitoring_updater import common
from walle.util.misc import get_existing_tiers


class Generator:
    def __init__(self, prefix):
        self._prefix = prefix
        self._ctype = common.get_ctype()
        self.alerts = []
        self._replaced = False

    @staticmethod
    def make_limits(crit_thresh, warn_thresh):
        if crit_thresh > warn_thresh:
            return [warn_thresh, crit_thresh], [crit_thresh, None]
        else:
            return [crit_thresh, warn_thresh], [None, crit_thresh]

    def make_juggler_check(self, host, service, stable=90, critical=600, tags=None):
        tags = tags or []
        return {
            "host": host,
            "service": service,
            "refresh_time": 5,
            "namespace": common.WALLE_JUGGLER_NAMESPACE,
            "aggregator": "logic_or",
            "flaps": {"stable": stable, "critical": critical},
            "tags": [self._ctype] + tags,
        }

    def make_alert(
        self,
        *,
        name,
        signal,
        itype,
        crit_thresh,
        warn_thresh,
        tags=None,
        alert_kwargs=None,
        percent_thresholds=False,
        juggler_tags=None,
    ):
        full_name = "{}.{}".format(self._prefix, name)
        tags_ = {
            "itype": [itype],
            "ctype": [self._ctype],
        }
        if tags:
            tags_.update(tags)
        warn_limit, crit_limit = (
            (warn_thresh, crit_thresh) if percent_thresholds else self.make_limits(crit_thresh, warn_thresh)
        )
        warn_literal, crit_literal = ("warn_perc", "crit_perc") if percent_thresholds else ("warn", "crit")
        alert = {
            crit_literal: crit_limit,
            warn_literal: warn_limit,
            "mgroups": ["ASEARCH"],
            "name": full_name,
            "abc": common.WALLE_ABC_SERVICE,
            "signal": signal,
            "tags": tags_,
        }
        if alert_kwargs is not None:
            alert.update(alert_kwargs)
        alert["juggler_check"] = self.make_juggler_check(
            host=common.make_host(),
            service=name,
            tags=juggler_tags,
        )
        self.alerts.append(alert)
        return alert

    def make_alerts_for_all_tiers(self, **kwargs):
        alerts = []
        if common.is_test():
            tiers = [2]
        else:
            tiers = get_existing_tiers()
        for tier in tiers:
            tier_specific_kwargs = copy.deepcopy(kwargs)
            if "tags" not in tier_specific_kwargs:
                tier_specific_kwargs["tags"] = {}
            tier_specific_kwargs["tags"]["tier"] = str(tier)
            tier_specific_kwargs["name"] = f"{tier_specific_kwargs['name']}.tier_{tier}"
            alerts.append(self.make_alert(**tier_specific_kwargs))
        return alerts

    def replace(self):
        assert not self._replaced
        self._replaced = True
        request = urllib.request.Request(
            url="https://yasm.yandex-team.ru/srvambry/alerts/replace",
            data=json.dumps({"prefix": self._prefix, "alerts": self.alerts}).encode("utf8"),
            headers={"Content-Type": "application/json", "User-Agent": "walle-self-monitoring-generator"},
        )
        return urllib.request.urlopen(request, timeout=15)
