import json
from collections import namedtuple

import pytest
from six.moves.http_client import BAD_REQUEST, UNAUTHORIZED

from tests.api.common import toggle_reboot_via_ssh
from walle.application import app
from walle.clients import staff, tvm
from walle.idm import tree_views, project_staff_id_converter
from walle.projects import get_project_owners
from walle.util.misc import iter_chunks, drop_none

MockedGroupInfo = namedtuple("MockedGroupInfo", "name, id, users")
GROUP1 = MockedGroupInfo("@yandex_mnt_sa_runtime_cross_9738", 91643, ["user1", "user2"])
GROUP2 = MockedGroupInfo("@yandex_mnt_sa_runtime_mondev_event", 91666, ["user2", "user3"])

PROJECT1_ID = "project1"
PROJECT2_ID = "project2"

BOT_PROJECT1 = 11111111
BOT_PROJECT2 = 22222222

STAFF_ID1 = 111

WALLE_TVM_APP_ID = 100500
IDM_TVM_APP_ID = 777777
UNKNOWN_TVM_APP_ID = 666666
# static public keys generated by `ya tool tvmknife unittest public_keys`
TVM_PUBLIC_KEYS = (
    '1:CpgCCpMCCAEQABqIAjCCAQQCggEAcLEXeH67FQESFUn4_7wnX7wN0PUrBoUsm3QQ4W5vC-qz6sXaEjSwnTV8w1o-z6X9KPLlhzM'
    'QvuS38NCNfK4uvJ4Zvfp3YsXJ25-rYtbnrYJHNvHohD-kPCCw_yZpMp21JdWigzQGuV7CtrxUhF-NNrsnUaJrE5-OpEWNt4X6nCIt'
    'KIYeVcSK6XJUbEWbrNCRbvkSc4ak2ymFeMuHYJVjxh4eQbk7_ZPzodP0WvF6eUYrYeb42imVEOR8ofVLQWE5DVnb1z_TqZm4i1XkS7'
    'jMwZuBxBRw8DGdYei0lT_sAf7KST2jC0590NySB3vsBgWEVs1OdUUWA6r-Dvx9dsOQtSCVkQYQAAqZAgqUAggCEAAaiQIwggEFAoIB'
    'AQDhEBM5-6YsPWfogKtbluJoCX1WV2KdzOaQ0-OlRbBzeCzw-eQKu12c8WakHBbeCMd1I1TU64SDkDorWjXGIa_2xT6N3zzNAE50ro'
    'TbPCcmeQrps26woTYfYIuqDdoxYKZNr0lvNLLW47vBr7EKqo1S4KSj7aXK_XYeEvUgIgf3nVIcNrio7VTnFmGGVQCepaL1Hi1gN4yI'
    'XjVZ06PBPZ-DxSRu6xOGbFrfKMJeMPs7KOyE-26Q3xOXdTIa1X-zYIucTd_bxUCL4BVbwW2AvbbFsaG7ISmVdGu0XUTmhXs1KrEfUV'
    'LRJhE4Dx99hAZXm1_HlYMUeJcMQ_oHOhV94ENFIJaRBhACCpYBCpEBCAMQABqGATCBgwKBgF9t2YJGAJkRRFq6fWhi3m1TFW1UOE0f'
    '6ZrfYhHAkpqGlKlh0QVfeTNPpeJhi75xXzCe6oReRUm-0DbqDNhTShC7uGUv1INYnRBQWH6E-5Fc5XrbDFSuGQw2EYjNfHy_HefHJX'
    'xQKAqPvxBDKMKkHgV58WtM6rC8jRi9sdX_ig2NIJeRBhABCpYBCpEBCAQQABqGATCBgwKBgGB4d6eLGUBv-Q6EPLehC4S-yuE2HB-_'
    'rJ7WkeYwyp-xIPolPrd-PQme2utHB4ZgpXHIu_OFksDe_0bPgZniNRSVRbl7W49DgS5Ya3kMfrYB4DnF5Fta5tn1oV6EwxYD4JONpF'
    'TenOJALPGTPawxXEfon_peiHOSBuQMu3_Vn-l1IJiRBhADCpcBCpIBCAUQABqHATCBhAKBgQCTJMKIfmfeZpaI7Q9rnsc29gdWawK7T'
    'npVKRHws1iY7EUlYROeVcMdAwEqVM6f8BVCKLGgzQ7Gar_uuxfUGKwqEQzoppDraw4F75J464-7D5f6_oJQuGIBHZxqbMONtLjBCXRU'
    'hQW5szBLmTQ_R3qaJb5vf-h0APZfkYhq1cTttSCZkQYQBAqWAQqRAQgLEAAahgEwgYMCgYBvvGVH_M2H8qxxv94yaDYUTWbRnJ1uiIY'
    'c59KIQlfFimMPhSS7x2tqUa2-hI55JiII0Xym6GNkwLhyc1xtWChpVuIdSnbvttbrt4weDMLHqTwNOF6qAsVKGKT1Yh8yf-qb-DSmic'
    'gvFc74mBQm_6gAY1iQsf33YX8578ClhKBWHSCVkQYQAAqXAQqSAQgMEAAahwEwgYQCgYEAkuzFcd5TJu7lYWYe2hQLFfUWIIj91BvQQ'
    'La_Thln4YtGCO8gG1KJqJm-YlmJOWQG0B7H_5RVhxUxV9KpmFnsDVkzUFKOsCBaYGXc12xPVioawUlAwp5qp3QQtZyx_se97YIoLzuL'
    'r46UkLcLnkIrp-Jo46QzYi_QHq45WTm8MQ0glpEGEAIKlwEKkgEIDRAAGocBMIGEAoGBAIUzbxOknXf_rNt17_ir8JlWvrtnCWsQd1M'
    'Anl5mgArvavDtKeBYHzi5_Ak7DHlLzuA6YE8W175FxLFKpN2hkz-l-M7ltUSd8N1BvJRhK4t6WffWfC_1wPyoAbeSN2Yb1jygtZJQ8w'
    'GoXHcJQUXiMit3eFNyylwsJFj1gzAR4JCdIJeRBhABCpYBCpEBCA4QABqGATCBgwKBgFMcbEpl9ukVR6AO_R6sMyiU11I8b8MBSUCEC'
    '15iKsrVO8v_m47_TRRjWPYtQ9eZ7o1ocNJHaGUU7qqInFqtFaVnIceP6NmCsXhjs3MLrWPS8IRAy4Zf4FKmGOx3N9O2vemjUygZ9vUi'
    'SkULdVrecinRaT8JQ5RG4bUMY04XGIwFIJiRBhADCpYBCpEBCA8QABqGATCBgwKBgGpCkW-NR3li8GlRvqpq2YZGSIgm_PTyDI2Zwfw'
    '69grsBmPpVFW48Vw7xoMN35zcrojEpialB_uQzlpLYOvsMl634CRIuj-n1QE3-gaZTTTE8mg-AR4mcxnTKThPnRQpbuOlYAnriwiasW'
    'iQEMbGjq_HmWioYYxFo9USlklQn4-9IJmRBhAEEpUBCpIBCAYQABqHATCBhAKBgQCoZkFGm9oLTqjeXZAq6j5S6i7K20V0lNdBBLqfm'
    'FBIRuTkYxhs4vUYnWjZrKRAd5bp6_py0csmFmpl_5Yh0b-2pdo_E5PNP7LGRzKyKSiFddyykKKzVOazH8YYldDAfE8Z5HoS9e48an5J'
    'sPg0jr-TPu34DnJq3yv2a6dqiKL9zSCakQYSlQEKkgEIEBAAGocBMIGEAoGBALhrihbf3EpjDQS2sCQHazoFgN0nBbE9eesnnFTfzQE'
    'LXb2gnJU9enmV_aDqaHKjgtLIPpCgn40lHrn5k6mvH5OdedyI6cCzE-N-GFp3nAq0NDJyMe0fhtIRD__CbT0ulcvkeow65ubXWfw6dB'
    'C2gR_34rdMe_L_TGRLMWjDULbNIJqRBg'
)
# service ticket from idm to wall-e
# generated by `ya tool tvmknife unittest service -s IDM_TVM_APP_ID -d WALLE_TVM_APP_ID`
IDM_TO_WALLE_ST = (
    '3:serv:CBAQ__________9_IggIsbwvEJSRBg:I9HkFxUNwJuEPbSW8atbl7nb421JHcw4ogQHoqvnYp_UmLxwiPAG6xQxXoKOx6i'
    'KhZi3WbPihHZ7G_uFsF8nettnbAL1S2AZu-rwTBOwVG13z-La3KfOVUBiq4bhMc-slCREzxkP3603x3s5eqSTq4PsWzSUFagYmegLTWpuOAA'
)
# service ticket from unknown service to wall-e
# generated by `ya tool tvmknife unittest service -s UNKNOWN_TVM_APP_ID -d WALLE_TVM_APP_ID`
UNALLOWED_SERVICE_TO_IDM_ST = (
    '3:serv:CBAQ__________9_IggIqtgoEJSRBg:B9QVXvFboyukEEHur3i9fk02huGiXN_lRHkYh1BRjBiEwYLL3I'
    'rpbf6Fyfe3TGjykN2FkkgPXt5JLKBOWiUsNLZeaqsMVfThN0VWX9wVS-En0ig-4xJrQ_6RKqRBeFr2WpitTcD21Kulsb98_xVd51IEv3ZpW6QgTGr0HqhwE6U'
)


@pytest.fixture()
def mock_idm_config(mp):
    mp.config("tvm.app_id", WALLE_TVM_APP_ID)
    mp.config("tvm.api_url", "https://tvm-api.yandex.net/2")
    mp.config("tvm.secret", None)
    mp.config("idm.tvm_aliases", {"idm": IDM_TVM_APP_ID})


@pytest.yield_fixture(autouse=True)
def app_with_tvm(mp, mock_idm_config, walle_test):
    """Enables tvm_ticket_manager in app object, mocking TVM ticket update"""
    mp.method(tvm.TvmServiceTicketManager._get_pub_keys, return_value=TVM_PUBLIC_KEYS, obj=tvm.TvmServiceTicketManager)
    mp.method(
        tvm.TvmServiceTicketManager._get_tickets_resp,
        return_value={
            str(IDM_TVM_APP_ID): {"ticket": IDM_TO_WALLE_ST},
            str(UNKNOWN_TVM_APP_ID): {"ticket": UNALLOWED_SERVICE_TO_IDM_ST},
        },
        obj=tvm.TvmServiceTicketManager,
    )

    with app.as_stopping_context():
        app.init_tvm_ticket_manager()
        app.tvm_ticket_manager._ready.wait(timeout=10)
        yield


@pytest.fixture()
def projects(walle_test):
    # remove default project
    default_proj = walle_test.default_project
    walle_test.projects.remove(walle_test.default_project)
    default_proj.delete()

    projects = {}

    projects[PROJECT1_ID] = walle_test.mock_project(
        {
            "id": PROJECT1_ID,
            "name": "Project 1",
            "owners": ["user1", "user100500", GROUP1.name],
            "bot_project_id": BOT_PROJECT1,
        }
    )
    toggle_reboot_via_ssh(projects[PROJECT1_ID], enable=True)

    projects[PROJECT2_ID] = walle_test.mock_project(
        {
            "id": PROJECT2_ID,
            "name": "Project 2",
            "owners": [GROUP1.name, GROUP2.name],
            "bot_project_id": BOT_PROJECT2,
        }
    )
    toggle_reboot_via_ssh(projects[PROJECT2_ID], enable=False)

    return projects


@pytest.fixture()
def staff_id_converter_mock(mp):
    def get_staff_id_mock(self_, project):
        if project.id == PROJECT1_ID:
            return STAFF_ID1
        else:
            raise project_staff_id_converter.MissingItem("Soryan")

    mp.method(
        project_staff_id_converter.ProjectStaffIdConverter._convert_bpis_to_staff_ids,
        obj=project_staff_id_converter.ProjectStaffIdConverter,
    )
    mp.method(
        project_staff_id_converter.ProjectStaffIdConverter.get_staff_id,
        side_effect=get_staff_id_mock,
        obj=project_staff_id_converter.ProjectStaffIdConverter,
    )


# We must check basic response format of all IDM handles: same status code (200), code field, optional error message etc
# But there are several IDM handles, and for /info we have to check many subtrees
# So here is class hierarchy of response checkers intended to reduce amount of boilerplate in tests


class ResponseChecker:
    handle_path = None
    use_get = True

    resp = None
    rj = None  # shortcut for resp.json

    def set_resp(self, resp):
        self.resp = resp
        self.rj = resp.json
        return self.resp

    def check_successful_resp(self):
        self._check_http_status_code()
        self._check_code_in_body()

    def check_error_resp(self, status, message):
        self._check_http_status_code()
        self._check_code_in_body(error=True)
        self._check_error_message(status, message)

    def test_req_wo_tvm_ticket(self, walle_test):
        resp = self._make_request(walle_test, service_ticket=None)
        assert resp.status_code == BAD_REQUEST
        assert resp.json["message"] == "TVM ticket is missing in request"

    def test_req_from_wrong_tvm_source(self, walle_test):
        resp = self._make_request(walle_test, service_ticket=UNALLOWED_SERVICE_TO_IDM_ST)
        assert resp.status_code == UNAUTHORIZED
        expected_message = "Authentication failed: TVM app id {} is not allowed to access Wall-e".format(
            UNKNOWN_TVM_APP_ID
        )
        assert resp.json["message"] == expected_message

    def _check_http_status_code(self):
        assert self.resp.status_code == 200

    def _check_code_in_body(self, error=False):
        assert "code" in self.rj
        assert bool(self.rj["code"]) == (True if error else False)

    def _check_error_message(self, status, message):
        assert status in self.rj
        assert self.rj[status] == message

    def _make_request(self, walle_test, **kwargs):
        if self.use_get:
            return self._make_get_request(walle_test, self.handle_path, **kwargs)
        else:
            return self._make_post_request(walle_test, self.handle_path, **kwargs)

    @staticmethod
    def _make_get_request(walle_test, path, service_ticket=IDM_TO_WALLE_ST):
        headers = drop_none({"X-Ya-Service-Ticket": service_ticket})
        return walle_test.api_client.get(path, headers=headers)

    @staticmethod
    def _make_post_request(walle_test, path, data=None, service_ticket=IDM_TO_WALLE_ST):
        data = data or {}
        headers = drop_none(
            {"Content-Type": "application/x-www-form-urlencoded", "X-Ya-Service-Ticket": service_ticket}
        )
        return walle_test.api_client.post(path, data=data, headers=headers)


class IdmInfoChecker(ResponseChecker):
    def check_successful_resp(self):
        super().check_successful_resp()
        self.check_root_node()

    def get_node(self, path):
        cur_node = self.rj
        for idx, component in enumerate(path):
            # even idx means slug of role node, odd ones mean value node
            if idx % 2 == 0:  # role node, slug must match exactly
                assert "roles" in cur_node
                cur_node = cur_node["roles"]
                assert cur_node["slug"] == component
            else:  # value node, select by name
                assert component in cur_node["values"]
                cur_node = cur_node["values"][component]
        return cur_node

    @staticmethod
    def check_role_node(node, slug, name):
        assert node.get("slug") == slug
        assert node.get("name") == name
        assert node.get("values")

    @staticmethod
    def check_value_node(role_node, value_name, has_role=False, set_attr_val=None, is_invisible=True):
        value_node = role_node["values"].get(value_name)
        assert value_node is not None

        if has_role:
            assert "roles" in value_node

        if set_attr_val:
            assert value_node.get("set") == set_attr_val

        if is_invisible:
            assert value_node.get("visibility") is False

    def check_root_node(self):
        root_node = self.get_node(["scopes"])
        self.check_role_node(root_node, slug="scopes", name="Scopes")


class TestIdmInfoProjects(IdmInfoChecker):
    handle_path = "/v1/idm/info/"
    use_get = True

    @pytest.mark.usefixtures("projects", "staff_id_converter_mock")
    def test_successful_resp(self, walle_test):
        resp = self._make_request(walle_test)
        self.set_resp(resp)
        self.check_successful_resp()

    def test_error_resp(self, walle_test, mp):
        error_msg = "Something very weird and project-related happened"
        mp.method(tree_views.RoleTreeView.to_dict, side_effect=Exception(error_msg), obj=tree_views.RoleTreeView)
        self.set_resp(self._make_request(walle_test))
        self.check_error_resp("error", error_msg)

    def check_successful_resp(self):
        super().check_successful_resp()
        projects_role_node_path = ["scopes", "project", "project"]
        for project_id in [PROJECT1_ID, PROJECT2_ID]:
            concrete_project_path = projects_role_node_path + [project_id]
            self.get_node(concrete_project_path)
            self.check_concrete_project_roles(concrete_project_path)

        self.check_staff_id_in_alias()

    def check_concrete_project_roles(self, concrete_project_path):
        node = self.get_node(concrete_project_path + ["role", "owner"])
        assert "project_owner" in node["set"]

        node = self.get_node(concrete_project_path + ["role", "ssh_rebooter"])
        assert "ssh_rebooter" in node["set"]

    def check_staff_id_in_alias(self):
        node_path = ["scopes", "project", "project", PROJECT1_ID, "role", "owner"]
        node = self.get_node(node_path)
        assert node["aliases"] == [{"type": "project-staff-id", "name": str(STAFF_ID1)}]


class TestIdmGetAllRoles(ResponseChecker):
    handle_path = "/v1/idm/get-all-roles/"
    use_get = True

    @pytest.mark.usefixtures("projects", "mock_groups_to_ids", "staff_id_converter_mock")
    def test_group_aware_successful(self, walle_test):
        walle_test.mock_project(
            {
                "id": "wrong-prj",
                "name": "Prj with nonexistent owner group",
                "owners": ["@i_dont_exist_on_staff"],
                "bot_project_id": BOT_PROJECT1,
            }
        )

        resp = self._make_request(walle_test)
        self.set_resp(resp)
        self.check_successful_resp()

        self.check_user_has_role("user1", {"scopes": "project", "project": PROJECT1_ID, "role": "owner"})
        self.check_user_has_role("user100500", {"scopes": "project", "project": PROJECT1_ID, "role": "owner"})

        self.check_group_has_role(GROUP1.id, {"scopes": "project", "project": PROJECT1_ID, "role": "owner"})
        self.check_group_has_role(GROUP1.id, {"scopes": "project", "project": PROJECT2_ID, "role": "owner"})
        self.check_group_has_role(GROUP2.id, {"scopes": "project", "project": PROJECT2_ID, "role": "owner"})

    def test_error_resp(self, walle_test, mp):
        error_msg = "Something very weird and project-related happened"
        mp.method(tree_views.RoleMembersView.to_dict, side_effect=Exception(error_msg), obj=tree_views.RoleMembersView)
        self.set_resp(self._make_request(walle_test))
        self.check_error_resp("error", error_msg)

    @pytest.mark.usefixtures("projects", "mock_groups_to_ids", "staff_id_converter_mock")
    def test_ssh_rebooter_is_robot_walle(self, walle_test):
        resp = self._make_request(walle_test)
        self.set_resp(resp)
        self.check_successful_resp()

        self.check_user_has_role("robot-walle", {"scopes": "project", "project": PROJECT1_ID, "role": "ssh_rebooter"})

    @pytest.mark.usefixtures(
        "staff_id_converter_mock",
        "mock_groups_to_ids",
    )
    def test_project_wo_owners_doesnt_fail(self, walle_test):
        walle_test.mock_project({"id": "prj", "name": "Prj without owners", "bot_project_id": BOT_PROJECT1})

        resp = self._make_request(walle_test)
        self.set_resp(resp)
        self.check_successful_resp()

    def check_user_has_role(self, user, role):
        user_roles = [ui["roles"] for ui in self.rj["users"] if ui["login"] == user]
        assert len(user_roles) == 1
        user_roles = user_roles[0]
        assert role in user_roles

    def check_group_has_role(self, group_id, role):
        group_roles = [gi["roles"] for gi in self.rj["groups"] if gi["group"] == group_id]
        assert len(group_roles) == 1
        group_roles = group_roles[0]
        assert role in group_roles

    @pytest.fixture()
    def mock_resolve_owners(self, mp):
        def resolve_owners_mock(groups):
            groups_to_users = {group_info.name: group_info.users for group_info in [GROUP1, GROUP2]}
            users = []
            for group in groups:
                users.extend(groups_to_users[group])
            return list(set(users))

        mp.function(staff.resolve_owners, side_effect=resolve_owners_mock)

    @pytest.fixture()
    def mock_groups_to_ids(self, mp):
        def groups_to_ids_mock(group_names):
            predefined_group_to_id = {group_info.name: group_info.id for group_info in [GROUP1, GROUP2]}
            return {
                group_name: predefined_group_to_id[group_name]
                for group_name in group_names
                if group_name in predefined_group_to_id
            }

        mp.function(staff.groups_to_ids, side_effect=groups_to_ids_mock)


class PostHandleChecker(ResponseChecker):
    use_get = False

    def make_role_request(self, walle_test, path, service_ticket=IDM_TO_WALLE_ST, **data_kwargs):
        role = self._path_to_role(path)  # role param duplicates path param in different format
        resp = self._make_request(
            walle_test, data=drop_none(dict(path=path, role=role, **data_kwargs)), service_ticket=service_ticket
        )
        return resp

    @staticmethod
    def _path_to_role(path):
        """Convert role path from file-like path to dict form"""
        path = path.strip("/")
        parts = path.split("/")
        return json.dumps(dict(iter_chunks(parts, 2)))


@pytest.fixture(params=[None, {}, {"passport-login": "yndx.some.login"}])
def fields(request):
    return json.dumps(request.param)


@pytest.fixture()
def mock_staff_group(mp):
    mp.function(staff.id_to_group, return_value=GROUP1.name)


@pytest.mark.usefixtures("staff_id_converter_mock")
class TestIdmAddRoleProjects(PostHandleChecker):
    handle_path = "/v1/idm/add-role/"

    @pytest.mark.parametrize(
        "member_param, added_member",
        [
            ({"login": "rookie"}, "rookie"),
            ({"group": GROUP1.id}, GROUP1.name),
        ],
    )
    @pytest.mark.parametrize("existing_owners", (["user1", "user2", GROUP2.name], None))
    @pytest.mark.usefixtures("mock_staff_group")
    def test_add_role(self, walle_test, member_param, added_member, existing_owners, fields):
        project = walle_test.mock_project({"id": PROJECT1_ID, "owners": existing_owners})
        self.set_resp(
            self.make_role_request(
                walle_test, "/scopes/project/project/{}/role/owner/".format(PROJECT1_ID), fields=fields, **member_param
            )
        )
        self.check_successful_resp()
        assert added_member in get_project_owners(project)
        walle_test.projects.assert_equal()

    @pytest.mark.parametrize("member_param", [{"login": "rookie"}, {"group": GROUP1.id}])
    @pytest.mark.usefixtures("mock_staff_group")
    def test_add_role_member_already_exists(self, walle_test, member_param, fields):
        walle_test.mock_project({"id": PROJECT1_ID, "owners": ["user1", "user2", "rookie", GROUP1.name]})
        self.set_resp(
            self.make_role_request(
                walle_test, "/scopes/project/project/{}/role/owner/".format(PROJECT1_ID), fields=fields, **member_param
            )
        )
        self.check_successful_resp()
        walle_test.projects.assert_equal()

    @pytest.mark.parametrize(
        "path, missing_node_name",
        [
            ("/wrong_root/but/everything/else/looks/valid/", "wrong_root"),
            ("/scopes/project/project/non_existing_project/role/owner/", "non_existing_project"),
        ],
    )
    @pytest.mark.parametrize("member_param", [{"login": "rookie"}, {"group": GROUP1.id}])
    @pytest.mark.usefixtures("mock_staff_group")
    def test_wrong_path(self, walle_test, path, missing_node_name, member_param, fields):
        self.set_resp(self.make_role_request(walle_test, path, fields=fields, **member_param))
        self.check_error_resp("error", "Child node {} not found".format(missing_node_name))
        walle_test.projects.assert_equal()


@pytest.fixture(params=["1", "0", None])
def fired(request):
    return request.param


@pytest.fixture(params=["1", "0", None])
def deleted(request):
    return request.param


@pytest.mark.usefixtures("staff_id_converter_mock")
class TestIdmRemoveRoleProjects(PostHandleChecker):
    handle_path = "/v1/idm/remove-role/"

    @pytest.mark.parametrize(
        "member_param, removed_member",
        [
            ({"login": "rookie"}, "rookie"),
            ({"group": GROUP1.id}, GROUP1.name),
        ],
    )
    @pytest.mark.usefixtures("mock_staff_group")
    def test_remove_role(self, walle_test, member_param, removed_member, fired, deleted, fields):
        owners = ["user1", "rookie", GROUP1.name]
        project = walle_test.mock_project({"id": PROJECT1_ID, "owners": owners})
        self.set_resp(
            self.make_role_request(
                walle_test,
                "/scopes/project/project/{}/role/owner/".format(PROJECT1_ID),
                fired=fired,
                deleted=deleted,
                fields=fields,
                **member_param
            )
        )
        self.check_successful_resp()
        assert removed_member not in get_project_owners(project)
        walle_test.projects.assert_equal()

    def test_after_last_member_removal(self, walle_test, fired, fields):
        member_to_remove = "last_of_the_mohicans"
        project = walle_test.mock_project({"id": PROJECT1_ID, "owners": [member_to_remove]})
        self.set_resp(
            self.make_role_request(
                walle_test,
                "/scopes/project/project/{}/role/owner/".format(PROJECT1_ID),
                fired=fired,
                fields=fields,
                login=member_to_remove,
            )
        )
        self.check_successful_resp()
        project.owners = []
        walle_test.projects.assert_equal()

    @pytest.mark.parametrize("member_param", [{"login": "some_already_removed_login"}, {"group": GROUP1.id}])
    @pytest.mark.usefixtures("mock_staff_group")
    def test_remove_role_not_present_member(self, walle_test, member_param, fired, fields):
        walle_test.mock_project({"id": PROJECT1_ID, "owners": ["user1", "user2", "rookie"]})
        self.set_resp(
            self.make_role_request(
                walle_test,
                "/scopes/project/project/{}/role/owner/".format(PROJECT1_ID),
                fired=fired,
                fields=fields,
                **member_param
            )
        )
        self.check_successful_resp()
        walle_test.projects.assert_equal()

    @pytest.mark.parametrize(
        "path",
        ["/scopes/project/project/non_existing_project/role/owner/", "/wrong_root/but/everything/else/looks/valid/"],
    )
    @pytest.mark.parametrize("member_param", [{"login": "rookie"}, {"group": GROUP1.id}])
    @pytest.mark.usefixtures("mock_staff_group")
    def test_missing_project_node_is_ok(self, walle_test, path, member_param, fired, fields):
        """IDM can come to revoke project roles after project object is deleted"""
        self.set_resp(self.make_role_request(walle_test, path, fired=fired, fields=fields, **member_param))
        self.check_successful_resp()
        walle_test.projects.assert_equal()

    def test_remove_rebooter_unsets_reboot_via_ssh(self, walle_test, projects):
        self.set_resp(
            self.make_role_request(
                walle_test, "/scopes/project/project/{}/role/ssh_rebooter/".format(PROJECT1_ID), login="robot-walle"
            )
        )
        projects[PROJECT1_ID].reboot_via_ssh = None
        walle_test.projects.assert_equal()
