import walle.util
from walle._tasks import sb_helpers
from walle.constants import SshOperation
from walle.fsm_stages.constants import EineProfileOperation
from walle.stages import Stages, StageTerminals
from walle.util.tasks import StageBuilder


def get_power_on_stages(pxe=None, check_post_code=False):
    sb = StageBuilder()
    if check_post_code:
        with sb.nested(Stages.POWER_ON_COMPOSITE) as power_on_stages:
            sb_helpers.power_on(power_on_stages, check_post_code=True, pxe=pxe)
            sb_helpers.power_off(power_on_stages)
            sb_helpers.power_on(power_on_stages, pxe=pxe)
    else:
        sb_helpers.power_on(sb, pxe=pxe)
    return sb


def get_ssh_reboot_stages(check_post_code=False, terminators=None):
    sb = StageBuilder()
    if check_post_code:
        with sb.nested(Stages.SSH_REBOOT_COMPOSITE) as ssh_reboot_stages:
            sb_helpers.ssh_reboot(ssh_reboot_stages, terminators=terminators)
            sb_helpers.power_off(ssh_reboot_stages)
            sb_helpers.power_on(ssh_reboot_stages)
    else:
        sb_helpers.ssh_reboot(sb, terminators=terminators, check_post_code=False)
    return sb


def get_kexec_reboot_stages(task_args):
    terminators = {StageTerminals.FAIL: StageTerminals.SKIP, StageTerminals.SUCCESS: StageTerminals.COMPLETE_PARENT}
    sb = StageBuilder()
    with sb.nested(Stages.KEXEC_REBOOT_COMPOSITE) as kexec_reboot_stages:
        sb_helpers.kexec_reboot(kexec_reboot_stages, terminators=terminators)
        kexec_reboot_stages.add_stages(get_reboot_stages(task_args).get_stages())
    return sb


def get_reboot_stages(task_args):
    sb = StageBuilder()
    with sb.nested(Stages.REBOOT) as reboot_stages:

        if task_args.ssh == SshOperation.FALLBACK:
            if task_args.check_post_code:
                terminators = {StageTerminals.FAIL: StageTerminals.COMPLETE_PARENT}
            else:
                terminators = {StageTerminals.FAIL: StageTerminals.SKIP}
        else:
            terminators = None

        if task_args.ssh in (SshOperation.ONLY, SshOperation.FALLBACK):
            reboot_stages.add_stages(
                get_ssh_reboot_stages(check_post_code=task_args.check_post_code, terminators=terminators).get_stages()
            )

        if task_args.ssh != SshOperation.ONLY:
            if task_args.without_ipmi:
                sb_helpers.itdc_reboot(reboot_stages, reason=task_args.reason)
            else:
                sb_helpers.power_off(reboot_stages, soft=True)
                power_on_sb = get_power_on_stages(check_post_code=task_args.check_post_code)
                reboot_stages.add_stages(power_on_sb.get_stages())
    return sb


def get_network_update_stages(profile_configuration, full=False):
    """
    :param host: Host to create task for. Need to deduce profile configuration (optional, for full mode)
    :param full: Enable "full mode": run through eine profile and wait for netmap if info is not readily available
    """

    builder = StageBuilder()
    with builder.nested(Stages.NETWORK) as network_stage_builder:
        # wait for switch/port info. on success - complete, on fail - proceed with next step
        network_stage_builder.stage(
            Stages.UPDATE_NETWORK_LOCATION,
            wait_for_racktables=False,
            terminators={
                StageTerminals.FAIL: StageTerminals.SKIP,
                StageTerminals.SUCCESS: StageTerminals.COMPLETE_PARENT,
            },
        )

        if full:
            # run swp-up profile. profile stage should try to update host location info if available.
            network_stage_builder.add_stages(
                walle.util.tasks.get_profile_stages(EineProfileOperation.PROFILE, *profile_configuration),
            )

            # even if eine didn't get the right switch/port we can still count on racktables,
            # because host worked for a few minutes
            network_stage_builder.stage(Stages.UPDATE_NETWORK_LOCATION, wait_for_racktables=True)

    return builder.get_stages()
