from attr import attrs, attrib, asdict

from walle import audit_log
from walle.admin_requests.severity import EineTag
from walle.clients.cauth import CauthSource, CauthFlowType
from walle.clients.cms import CmsTaskAction
from walle.clients.juggler import JugglerDowntimeName
from walle.constants import NetworkTarget, SshOperation
from walle.expert.decision import Decision
from walle.hosts import HostStatus, HostState, HostOperationState, TaskType
from walle.operations_log.constants import Operation
from walle.util.misc import drop_none, is_not_none_arg_validator

PAYLOAD = "payload"


def is_eine_severity_tag_validator(instance, attribute, value):
    if value and value not in EineTag.ALL:
        raise AttributeError(
            "'{}' is not severity tag for Eine, but here what you can use: {}".format(value, EineTag.ALL)
        )


def is_flow_type_validator(instance, attribute, value):
    if value and value not in CauthFlowType.ALL:
        raise AttributeError("'{}' is not valid flow type. Possible values: {}".format(value, CauthFlowType.ALL))


def is_source_validator(instance, attribute, value):
    if value:
        invalid_values = set(value) - set(CauthSource.ALL)
        if invalid_values:
            raise AttributeError(
                "'{}' is not valid source(s). Possible values: {}".format(
                    "', '".join(invalid_values),
                    CauthSource.ALL,
                )
            )


@attrs
class BaseTaskArgs:
    issuer = attrib(validator=[is_not_none_arg_validator])
    task_type = attrib(validator=[is_not_none_arg_validator])
    operation_type = attrib(validator=[is_not_none_arg_validator])
    operation_host_status = attrib(validator=[is_not_none_arg_validator])
    cms_action = attrib(validator=[is_not_none_arg_validator])
    # "type" - request type for audit log
    type = attrib(validator=[is_not_none_arg_validator])

    project = attrib()
    host_inv = attrib()
    host_name = attrib()
    host_uuid = attrib()
    scenario_id = attrib(default=None)

    failure = attrib(default=None)
    failure_type = attrib(default=None)
    decision = attrib(default=None)
    check_names = attrib(default=None)
    operation_restrictions = attrib(default=list())
    checks_to_monitor = attrib(default=None)
    reason = attrib(default=None)
    ignore_cms = attrib(default=False)
    with_auto_healing = attrib(default=None)
    monitor_on_completion = attrib(default=True)
    ignore_maintenance = attrib(default=False)
    disable_admin_requests = attrib(default=False)
    from_current_task = attrib(default=False)
    release = attrib(default=False)
    force_new_task = attrib(default=False)
    ticket_key = attrib(default=None)
    keep_downtime = attrib(default=False)
    allowed_states = attrib(default=HostState.ALL_ASSIGNED)
    allowed_statuses = attrib(default=HostStatus.ALL_STEADY)
    health_status_accuracy = attrib(default=None)
    target_status = attrib(default=None)
    unset_ticket = attrib(default=False)
    operation_log_params = attrib(default=None)
    checks_for_use = attrib(default=None)
    use_cloud_post_processor = attrib(default=False)
    profile_after_task = attrib(default=False)
    redeploy_after_task = attrib(default=False)
    keep_task_id = attrib(default=False)

    def get_task_params(self):
        result = {PAYLOAD: dict()}
        log_fields = audit_log.LogEntry.task_fields

        for field, value in asdict(self).items():
            if isinstance(value, Decision):
                value = value.to_dict()
            if field in log_fields:
                result[field] = value
            else:
                result[PAYLOAD][field] = value

        result[PAYLOAD] = drop_none(result[PAYLOAD])
        return drop_none(result)


@attrs
class RebootTaskArgs(BaseTaskArgs):
    operation_type = attrib(default=Operation.REBOOT.type)
    operation_host_status = attrib(default=Operation.REBOOT.host_status)
    type = attrib(default=audit_log.TYPE_REBOOT_HOST)
    cms_action = attrib(default=CmsTaskAction.REBOOT)
    network = attrib(default=NetworkTarget.PROJECT)
    ssh = attrib(default=SshOperation.FORBID)
    check_post_code = attrib(default=False)
    check_post_code_reason = attrib(default=None)
    without_ipmi = attrib(default=False)


@attrs
class KexecRebootTaskArgs(BaseTaskArgs):
    operation_type = attrib(default=Operation.REBOOT.type)
    operation_host_status = attrib(default=Operation.REBOOT.host_status)
    type = attrib(default=audit_log.TYPE_KEXEC_REBOOT_HOST)
    cms_action = attrib(default=CmsTaskAction.REBOOT)
    network = attrib(default=NetworkTarget.PROJECT)
    ssh = attrib(default=SshOperation.FORBID)
    check_post_code = attrib(default=False)
    check_post_code_reason = attrib(default=None)
    without_ipmi = attrib(default=False)


@attrs
class PowerOnTaskArgs(BaseTaskArgs):
    operation_type = attrib(default=Operation.POWER_ON.type)
    operation_host_status = attrib(default=Operation.POWER_ON.host_status)
    type = attrib(default=audit_log.TYPE_POWER_ON_HOST)
    cms_action = attrib(default=None)
    check_post_code = attrib(default=False)
    check_post_code_reason = attrib(default=None)
    ignore_cms = attrib(default=True)


@attrs
class PowerOffTaskArgs(BaseTaskArgs):
    operation_type = attrib(default=Operation.POWER_OFF.type)
    operation_host_status = attrib(default=Operation.POWER_OFF.host_status)
    type = attrib(default=audit_log.TYPE_POWER_OFF_HOST)
    cms_action = attrib(default=CmsTaskAction.POWER_OFF)
    soft = attrib(default=True)
    monitor_on_completion = attrib(default=False)


@attrs
class DeleteHostTaskArgs(BaseTaskArgs):
    operation_type = attrib(default=Operation.DELETE.type)
    operation_host_status = attrib(default=Operation.DELETE.host_status)
    type = attrib(default=audit_log.TYPE_DELETE_HOST)
    cms_action = attrib(default=CmsTaskAction.DEACTIVATE)
    allowed_statuses = attrib(default=HostStatus.ALL_STEADY + [HostStatus.INVALID])
    monitor_on_completion = attrib(default=False)
    lui = attrib(default=False)
    cms_task_id = attrib(default=None)
    force_new_cms_task = attrib(default=True)


@attrs
class VlanSwitchTaskArgs(BaseTaskArgs):
    operation_type = attrib(default=Operation.SWITCH_VLANS.type)
    operation_host_status = attrib(default=Operation.SWITCH_VLANS.host_status)
    type = attrib(default=audit_log.TYPE_SWITCH_VLAN)
    # NB: we don't ask CMS for permission for this operation.
    cms_action = attrib(default='')
    vlans = attrib(default=None)
    native_vlan = attrib(default=None)
    network_target = attrib(default=None)
    ignore_cms = attrib(default=True)
    with_auto_healing = attrib(default=False)
    monitor_on_completion = attrib(default=False)
    network_update_args = attrib(default=None)
    update_network_location = attrib(default=False)


@attrs
class ProjectSwitchingArgs(BaseTaskArgs):
    operation_type = attrib(default=Operation.SWITCH_PROJECT.type)
    operation_host_status = attrib(default=Operation.SWITCH_PROJECT.host_status)
    type = attrib(default=audit_log.TYPE_SWITCH_PROJECT)
    allowed_states = attrib(default=[HostState.FREE, HostState.ASSIGNED, HostState.MAINTENANCE])
    target_project_id = attrib(default=None)
    target_project_bot_project_id = attrib(default=None)
    current_project_id = attrib(default=None)
    erase_disks = attrib(default=True)
    force = attrib(default=False)
    host_state = attrib(default=HostState.FREE)
    cms_action = attrib(default=CmsTaskAction.DEACTIVATE)
    force_new_cms_task = attrib(default=False)
    soft = attrib(default=False)
    network_target = attrib(default=NetworkTarget.PARKING)
    clear = attrib(default=True)
    free = attrib(default=True)
    cms_task_id = attrib(default=None)


@attrs
class HostReleaseArgs(BaseTaskArgs):
    operation_type = attrib(default=Operation.RELEASE_HOST.type)
    operation_host_status = attrib(default=Operation.RELEASE_HOST.host_status)
    type = attrib(default=audit_log.TYPE_RELEASE_HOST)
    allowed_states = attrib(default=[HostState.FREE, HostState.PROBATION, HostState.ASSIGNED, HostState.MAINTENANCE])
    current_project_id = attrib(default=None)
    erase_disks = attrib(default=True)
    host_state = attrib(default=HostState.FREE)
    cms_action = attrib(default=CmsTaskAction.DEACTIVATE)
    force_new_cms_task = attrib(default=False)
    soft = attrib(default=False)
    network_target = attrib(default=NetworkTarget.PARKING)
    clear = attrib(default=True)
    free = attrib(default=True)
    cms_task_id = attrib(default=None)


@attrs
class SwitchToMaintenanceTaskArgs(BaseTaskArgs):
    operation_type = attrib(default=Operation.SWITCH_TO_MAINTENANCE.type)
    operation_host_status = attrib(default=Operation.SWITCH_TO_MAINTENANCE.host_status)
    type = attrib(default=audit_log.TYPE_SET_HOST_MAINTENANCE)
    cms_action = attrib(default=CmsTaskAction.PROFILE)
    task_group = attrib(default=None)
    timeout_time = attrib(default=None)
    timeout_status = attrib(default=HostStatus.READY)
    power_off = attrib(default=False)
    operation_state = attrib(default=HostOperationState.OPERATION)
    workdays_only = attrib(default=None)
    soft = attrib(default=True)
    allowed_states = attrib(default=(HostState.ASSIGNED, HostState.PROBATION))


@attrs
class PrepareTaskArgs(BaseTaskArgs):
    operation_type = attrib(default=Operation.PREPARE.type)
    operation_host_status = attrib(default=Operation.PREPARE.host_status)
    type = attrib(default=audit_log.TYPE_PREPARE_HOST)
    cms_action = attrib(default=CmsTaskAction.PREPARE)
    juggler_downtime_name = attrib(default=JugglerDowntimeName.DELETED_HOST)
    get_full_network_location = attrib(default=True)
    create_dns = attrib(default=True)
    keep_fqdn = attrib(default=False)
    bot_project_id = attrib(default=None)
    skip_profile = attrib(default=False)
    profile_configuration = attrib(default=None)
    network_update_args = attrib(default=None)
    deploy_configuration = attrib(default=None)
    provisioner = attrib(default=None)
    config = attrib(default=None)
    extra_vlans = attrib(default=None)
    deploy_tags = attrib(default=None)
    deploy_config_policy = attrib(default=None)
    deploy_network = attrib(default=None)
    network_target = attrib(default=NetworkTarget.PROJECT)
    host_provisioner = attrib(default=None)
    host_config = attrib(default=None)
    host_deploy_config_policy = attrib(default=None)
    host_deploy_tags = attrib(default=None)
    host_deploy_network = attrib(default=None)
    update_firmware_configuration = attrib(default=None)
    firmware_update_needed = attrib(default=False)
    allowed_states = attrib(default=[HostState.FREE])
    allowed_statuses = attrib(default=HostStatus.ALL_STEADY)
    repair_request_severity = attrib(default=None, validator=[is_eine_severity_tag_validator])


@attrs
class EnsureHostPreorderAcquirementTaskArgs(BaseTaskArgs):
    operation_type = attrib(default=Operation.BOT_WAIT_FOR_HOST_ACQUIREMENT.type)
    operation_host_status = attrib(default=Operation.BOT_WAIT_FOR_HOST_ACQUIREMENT.host_status)
    type = attrib(default=audit_log.TYPE_BOT_WAIT_FOR_HOST_ACQUIREMENT)
    allowed_states = attrib(default=[HostState.FREE])

    cms_action = attrib(default='')
    monitor_on_completion = attrib(default=False)


@attrs
class SetAssignedTaskArgs(BaseTaskArgs):
    operation_type = attrib(default=Operation.SWITCH_TO_ASSIGNED.type)
    operation_host_status = attrib(default=Operation.SWITCH_TO_ASSIGNED.host_status)
    type = attrib(default=audit_log.TYPE_SET_HOST_STATE)

    cms_action = attrib(default='')
    power_on = attrib(default=False)
    ignore_cms = attrib(default=False)
    unset_ticket = attrib(default=True)
    target_status = attrib(default=HostStatus.READY)


@attrs
class FooBarTaskArgs(BaseTaskArgs):
    cms_action = attrib(default=None)
    operation_type = attrib(default=Operation.FOOBAR.type)
    operation_host_status = attrib(default=Operation.FOOBAR.host_status)
    type = attrib(default=audit_log.TYPE_FOOBAR_HOST)

    task_type = attrib(default=TaskType.MANUAL)
    cycles = attrib(default=1)
    foo_args = attrib(factory=dict)
    bar_args = attrib(factory=dict)


@attrs
class FqdnDeinvalidationArgs(BaseTaskArgs):
    operation_type = attrib(default=Operation.FQDN_DEINVALIDATION.type)
    operation_host_status = attrib(default=Operation.FQDN_DEINVALIDATION.host_status)
    type = attrib(default=audit_log.TYPE_FQDN_DEINVALIDATION)
    allowed_states = attrib(default=HostState.ALL)
    allowed_statuses = attrib(default=[HostStatus.INVALID])
    cms_action = attrib(default=CmsTaskAction.DEACTIVATE)

    soft = attrib(default=False)
    bot_project_id = attrib(default=None)
    release = attrib(default=False)
    monitor_on_completion = attrib(default=False)
    network_target = attrib(default=NetworkTarget.PARKING)
    clear = attrib(default=True)
    cms_task_id = attrib(default=None)
    clear_old_fqdn_records = attrib(default=False)
