"""Host task creation."""

import logging

import walle.stages
from sepelib.core.exceptions import LogicalError
from walle._tasks import sb_helpers
from walle._tasks.stages import (
    get_reboot_stages,
    get_power_on_stages,
    get_network_update_stages,
    get_kexec_reboot_stages,
)
from walle.clients.juggler import JugglerDowntimeName
from walle.constants import FLEXY_EINE_PROFILE
from walle.fsm_stages.constants import EineProfileOperation
from walle.hosts import Task, HostState, TaskType
from walle.stages import StageTerminals
from walle.util.tasks import StageBuilder, get_profile_stages, get_deploy_stages, _get_new_task_next_check
from walle.clients.eine import EineProfileTags

log = logging.getLogger(__name__)


def create_new_task(
    host, task_args, audit_entry, sb, next_check=None, keep_cms_task=False, monitoring_timeout=None, task_id=None
):
    sb_helpers.log_completed_operation(
        sb, operation_type=task_args.operation_type, params=task_args.operation_log_params
    )
    sb_helpers.reset_health_status(sb, health_status_accuracy=task_args.health_status_accuracy)

    if not task_args.disable_admin_requests:
        sb_helpers.cancel_admin_requests(sb)

    if task_args.monitor_on_completion:
        if task_args.checks_to_monitor is None:
            raise LogicalError()

        # monitoring timeout is not a final time limit, it is an extra time added to the default value.
        sb_helpers.monitor(
            sb,
            checks_to_monitor=task_args.checks_to_monitor,
            monitoring_timeout=monitoring_timeout,
            checks_for_use=task_args.checks_for_use,
        )

    if task_args.use_cloud_post_processor:
        sb_helpers.cloud_post_processor(
            sb, need_profile=task_args.profile_after_task, need_redeploy=task_args.redeploy_after_task
        )

    stages = walle.stages.set_uids(sb.get_stages())

    if next_check is None:
        next_check = _get_new_task_next_check(task_args.task_type)

    origin_decision = None
    if host.health:
        origin_decision = host.health.clone_decision()
    return Task(
        task_id=Task.next_task_id() if not task_id else task_id,
        type=task_args.task_type,
        owner=task_args.issuer,
        audit_log_id=audit_entry.id,
        status="pending",
        stages=stages,
        target_status=task_args.target_status,
        ignore_cms=task_args.ignore_cms,
        keep_downtime=task_args.keep_downtime,
        keep_cms_task=keep_cms_task,
        disable_admin_requests=task_args.disable_admin_requests,
        enable_auto_healing=task_args.with_auto_healing,
        next_check=next_check,
        revision=0,
        decision=origin_decision,
    )


def get_reboot_task_stages(task_args):
    sb = StageBuilder()
    sb_helpers.acquire_permission(
        sb,
        action=task_args.cms_action,
        reason=task_args.reason,
        failure=task_args.failure,
        check_names=task_args.check_names,
        failure_type=task_args.failure_type,
    )
    sb_helpers.set_downtime(sb)
    if task_args.task_type == TaskType.AUTOMATED_HEALING:
        sb_helpers.switch_vlans(sb, network=task_args.network)
    reboot_stage_builder = get_reboot_stages(task_args)
    sb.add_stages(reboot_stage_builder.get_stages())
    return sb


def get_kexec_reboot_task_stages(task_args):
    sb = StageBuilder()
    sb_helpers.acquire_permission(
        sb,
        action=task_args.cms_action,
        reason=task_args.reason,
        failure=task_args.failure,
        check_names=task_args.check_names,
        failure_type=task_args.failure_type,
    )
    sb_helpers.set_downtime(sb)
    if task_args.task_type == TaskType.AUTOMATED_HEALING:
        sb_helpers.switch_vlans(sb, network=task_args.network)
    kexec_reboot_stage_builder = get_kexec_reboot_stages(task_args)
    sb.add_stages(kexec_reboot_stage_builder.get_stages())
    return sb


def get_bot_acquirement_task_stages(task_args):
    sb = StageBuilder()
    sb_helpers.ensure_host_distribution(sb)
    return sb


def get_power_on_task_stages(task_args):
    sb = StageBuilder()
    sb_helpers.acquire_permission(sb)
    sb_helpers.set_downtime(sb)
    sb.add_stages(get_power_on_stages(check_post_code=task_args.check_post_code).get_stages())
    return sb


def get_power_off_task_stages(task_args):
    sb = StageBuilder()
    sb_helpers.acquire_permission(sb, action=task_args.cms_action, reason=task_args.reason)
    sb_helpers.set_downtime(sb)
    sb_helpers.power_off(sb, soft=task_args.soft)
    return sb


def get_delete_host_task_stages(task_args):
    sb = StageBuilder()
    sb_helpers.acquire_permission(
        sb, action=task_args.cms_action, reason=task_args.reason, force_new_cms_task=task_args.force_new_cms_task
    )

    if task_args.lui:
        sb_helpers.lui_remove(sb)

    # Delete CMS task that was caused by MAINTENANCE state
    if task_args.cms_task_id:
        sb_helpers.delete_task_from_cms(sb, task_args.cms_task_id)

    sb_helpers.complete_deletion(sb)
    return sb


def get_vlan_switch_task_stages(task_args):
    sb = StageBuilder()
    if task_args.update_network_location:
        sb_helpers.power_on(sb)
        sb.add_stages(get_network_update_stages(task_args.network_update_args, full=True))
    else:
        sb_helpers.wait_for_active_mac(sb)
    sb_helpers.switch_vlans(
        sb, network=task_args.network_target, vlans=task_args.vlans, native_vlan=task_args.native_vlan
    )
    return sb


def get_switch_project_task_stages(task_args):
    sb = StageBuilder()
    if task_args.release and task_args.host_state != HostState.FREE:
        sb.add_stages(get_release_host_stages(task_args).get_stages())
    else:
        sb_helpers.acquire_permission(sb, action=task_args.cms_action, ignore_cms=task_args.ignore_cms)
        sb_helpers.switch_default_cms_task_project(
            sb, task_args.cms_task_id, task_args.current_project_id, task_args.target_project_id
        )

    sb_helpers.set_downtime(sb, juggler_downtime_name=JugglerDowntimeName.TRANSITION_STATE_HOST)
    sb_helpers.switch_project(sb, task_args.target_project_id, task_args.operation_restrictions)
    sb_helpers.add_fqdn_to_cauth(sb, terminators={StageTerminals.FAIL: StageTerminals.SKIP})
    sb_helpers.assign_bot_project(sb, task_args.target_project_bot_project_id)
    return sb


def get_release_host_stages(task_args):
    sb = StageBuilder()
    sb_helpers.acquire_permission(
        sb,
        action=task_args.cms_action,
        reason=task_args.reason,
        ignore_cms=task_args.ignore_cms,
        force_new_cms_task=task_args.force_new_cms_task,
    )
    sb_helpers.set_downtime(sb, juggler_downtime_name=JugglerDowntimeName.DELETED_HOST)

    # We are possibly releasing hosts for demounting, let user skip this if host can't even boot into eine.
    if task_args.erase_disks:
        sb.add_stages(
            get_profile_stages(
                EineProfileOperation.RELEASE, FLEXY_EINE_PROFILE, profile_tags=[EineProfileTags.DRIVES_WIPE]
            )
        )

    # don't waste time trying to soft power off, also decrease chance for a failure
    # skip IPMI errors during host release (see WALLE-1421)
    sb_helpers.power_off(sb, soft=task_args.soft, skip_errors=task_args.disable_admin_requests)

    sb_helpers.switch_vlans(sb, task_args.network_target)
    sb_helpers.setup_dns(sb, clear=task_args.clear)
    sb_helpers.lui_remove(sb)
    sb_helpers.set_hostname(sb, free=task_args.free)

    # Delete CMS task that was caused by MAINTENANCE state
    sb_helpers.delete_task_from_cms(sb, task_args.cms_task_id)

    sb_helpers.release_host(sb)

    return sb


def get_assign_stages(task_args):
    sb = StageBuilder()

    if task_args.power_on:
        sb_helpers.power_on(sb)

    sb_helpers.set_downtime(sb, juggler_downtime_name=JugglerDowntimeName.TRANSITION_STATE_HOST)
    sb_helpers.set_assigned(sb, reason=task_args.reason)
    return sb


def get_switch_to_maintenance_stages(task_args):
    sb = StageBuilder()
    sb_helpers.acquire_permission(
        sb,
        action=task_args.cms_action,
        task_group=task_args.task_group,
        workdays=task_args.workdays_only,
        reason=task_args.reason,
    )
    sb_helpers.set_downtime(sb)
    sb_helpers.set_maintenance(
        sb,
        ticket_key=task_args.ticket_key,
        timeout_time=task_args.timeout_time,
        timeout_status=task_args.timeout_status,
        operation_state=task_args.operation_state,
        reason=task_args.reason,
    )
    if task_args.power_off:
        sb_helpers.power_off(sb, soft=task_args.soft)
    return sb


def get_prepare_assign_hostname_stages(task_args):
    sb = StageBuilder()
    sb_helpers.ensure_host_distribution(sb)
    sb_helpers.set_downtime(sb, juggler_downtime_name=task_args.juggler_downtime_name)
    if not task_args.keep_fqdn:
        sb_helpers.set_hostname(sb)
    return sb


def get_prepare_profile_deploy_stages(task_args, update_firmware=False):
    sb = StageBuilder()
    sb_helpers.set_downtime(sb, juggler_downtime_name=JugglerDowntimeName.DEFAULT)
    sb_helpers.acquire_permission(
        sb,
        action=task_args.cms_action,
        reason=task_args.reason,
        failure=task_args.failure,
        check_names=task_args.check_names,
        ignore_cms=task_args.ignore_cms,
        failure_type=task_args.failure_type,
    )
    sb_helpers.set_probation(sb, reason=task_args.reason)
    sb_helpers.assign_bot_project(sb, bot_project_id=task_args.bot_project_id)
    if task_args.skip_profile:
        sb_helpers.power_on(sb)
        sb.add_stages(get_network_update_stages(task_args.network_update_args, full=True))
    else:
        profile_stages = get_profile_stages(
            EineProfileOperation.PREPARE,
            *task_args.profile_configuration,
            repair_request_severity=task_args.repair_request_severity
        )
        sb.add_stages(profile_stages)

    if update_firmware:
        sb.add_stages(get_prepare_update_firmware_stages(task_args).get_stages())

    sb_helpers.switch_vlans(sb, task_args.network_target, extra_vlans=task_args.extra_vlans or None)
    sb_helpers.setup_dns(sb, create=True)
    sb_helpers.add_fqdn_to_cauth(sb, terminators={StageTerminals.FAIL: StageTerminals.SKIP})

    deploy_stages = get_deploy_stages(
        task_args.deploy_configuration,
        config_forced=task_args.config is not None,
        extra_vlans=task_args.extra_vlans,
        with_autohealing=False,
    )
    sb.add_stages(deploy_stages)

    sb_helpers.complete_preparing(
        sb,
        task_args.host_provisioner,
        task_args.host_config,
        task_args.deploy_config_policy,
        task_args.host_deploy_tags,
        task_args.host_deploy_network,
        task_args.extra_vlans or None,
        task_args.operation_restrictions,
    )
    return sb


def get_prepare_update_firmware_stages(task_args):
    sb = StageBuilder()
    profile_stages = get_profile_stages(
        EineProfileOperation.PROFILE,
        *task_args.update_firmware_configuration,
        repair_request_severity=task_args.repair_request_severity
    )
    sb.add_stages(profile_stages)
    return sb


def get_prepare_stages(task_args):
    sb = StageBuilder()
    sb.add_stages(get_prepare_assign_hostname_stages(task_args).get_stages())
    sb.add_stages(
        get_prepare_profile_deploy_stages(task_args, update_firmware=task_args.firmware_update_needed).get_stages()
    )
    return sb


def get_foobar_stages(task_args):
    sb = StageBuilder()
    sb_helpers.acquire_permission(sb, ignore_cms=task_args.ignore_cms)
    sb_helpers.foo_and_bar_stages(sb, task_args.cycles, task_args.foo_args, task_args.bar_args)
    return sb


def get_fqdn_deinvalidation_stages(task_args):
    sb = StageBuilder()

    sb_helpers.acquire_permission(
        sb, action=task_args.cms_action, ignore_cms=task_args.ignore_cms, reason=task_args.reason
    )
    sb_helpers.set_downtime(sb, juggler_downtime_name=JugglerDowntimeName.DELETED_HOST)

    if task_args.clear_old_fqdn_records:
        sb_helpers.setup_dns(sb, clear=task_args.clear)
        sb_helpers.lui_remove(sb)

    sb_helpers.fqdn_deinvalidation(sb)
    sb_helpers.set_downtime(sb)

    if task_args.release:
        sb_helpers.switch_vlans(sb, task_args.network_target)
        sb_helpers.setup_dns(sb, clear=task_args.clear)
        sb_helpers.lui_remove(sb)
        sb.add_stages(
            get_profile_stages(
                EineProfileOperation.RELEASE, FLEXY_EINE_PROFILE, profile_tags=[EineProfileTags.DRIVES_WIPE]
            )
        )
        sb_helpers.power_off(sb, soft=task_args.soft, skip_errors=task_args.disable_admin_requests)
        sb_helpers.release_host(sb)
    else:
        sb_helpers.add_fqdn_to_cauth(sb, terminators={StageTerminals.FAIL: StageTerminals.SKIP})

    if task_args.cms_task_id:
        sb_helpers.delete_task_from_cms(sb, task_args.cms_task_id)

    sb_helpers.assign_bot_project(sb, task_args.bot_project_id)

    return sb
