import re
from functools import wraps

from sepelib.core.exceptions import LogicalError
from walle.util.misc import filter_dict_keys, drop_none

STATUS_IN_PROCESS = "in-process"
STATUS_PROCESSED = "processed"
STATUS_NOT_EXIST = "does-not-exist"
STATUS_DELETED = "closed"


class EineCode:
    # eine codes for admin requests originated from Wall-E, not from checks and monitoring pipeline.
    # for IPMI requests they match 1-to-1 with request type

    IPMI_PROTO_ERROR = "IPMI_PROTO_ERROR"
    IPMI_IKVM_NO_SIGNAL = "IPMI_IKVM_NO_SIGNAL"
    IPMI_DNS_RESOLUTION_FAILED = "IPMI_DNS_RESOLUTION_FAILED"
    IPMI_PORT_UNREACHABLE = "IPMI_PORT_UNREACHABLE"
    IPMI_HTTP_FAILED = "IPMI_HTTP_FAILED"


class _RequestPattern:
    def __init__(
        self, operation, problem, type, description, params_fetcher=None, details_fetcher=None, request_id_parts=None
    ):
        """
        :param operation: Operation group name from https://bot.yandex-team.ru/adm/js/request_patterns.php
        :param problem: Problem id from https://bot.yandex-team.ru/adm/js/request_patterns.php
        :param type: request type name used in wall-e
        :param description: request type description used in wall-e
        :param params_fetcher: optional function to convert decision params into bot request params
        :param details_fetcher: optional function to create detailed description for bot request
        :param request_id_parts: optional function to create extra params for request id used in wall-e

        :type operation: str
        :type problem: int
        :type type: str
        :type description: str
        """
        self.operation = operation
        self.problem = problem
        self.type = type
        self.description = description

        if params_fetcher is None:
            params_fetcher = default_params_fetcher

        self.params = with_eine_code(params_fetcher)

        if details_fetcher is not None:
            self.details = details_fetcher

        if request_id_parts is not None:
            self.request_id_extra_parts = request_id_parts

    @staticmethod
    def params(decision_params):
        return default_params_fetcher(decision_params)

    @staticmethod
    def details(host, reason, decision_params):
        return reason

    @staticmethod
    def request_id_extra_parts(decision_params):
        return ()

    def __repr__(self):
        return "{class_name}(type={request_type}...)".format(class_name=type(self), request_type=self.type)


class _NocRequestPattern:
    def __init__(self, error_type, instruction):
        self.type = error_type
        self.instruction = instruction


def with_eine_code(func):
    """Add eine code to admin request params"""

    @wraps(func)
    def wrapper(decision_params):
        params = func(decision_params)

        eine_code = decision_params.get("eine_code")
        if eine_code:
            if not isinstance(eine_code, str):
                eine_code = ",".join(eine_code)

            params["eine_code"] = eine_code

        return params

    return wrapper


def default_params_fetcher(decision_params):
    return drop_none(decision_params)


def params_fetcher_with_default_eine_code(default):
    def params_fetcher(decision_params):
        params = drop_none(decision_params)
        params.setdefault("eine_code", default)

        return params

    return params_fetcher


def mk_params_fetcher(keys_to_fetch):
    def params_fetcher(decision_params):
        return drop_none(filter_dict_keys(decision_params, keys_to_fetch))

    return params_fetcher


def mk_params_for_gpu(decision_params):
    if "slot" in decision_params or "serial" in decision_params:
        return drop_none(filter_dict_keys(decision_params, ("slot", "serial")))

    if "errors" in decision_params:
        # example with serial number: "availability: less local GPUs 6 than in bot 8: 0321418015041, 0321418015869"
        # example with slot: "availability: GPU Tesla V100-PCIE-32GB PCIe 0000:87:00.0 not found"
        # more examples here https://wiki.yandex-team.ru/haas/services/hw-watcher/gpu/
        # NB: only fetch first serial. BOT does not support multiple GPU-s per request.
        serial_re = re.compile(r'than in bot \d+:\s+(\d+)')
        slot_re = re.compile(r'PCIe\s+(\d+:\d+:\d+.\d+)')

        for error in decision_params["errors"]:
            for param_name, regex in [("slot", slot_re), ("serial", serial_re)]:
                match = regex.search(error)
                if match:
                    return {param_name: match.group(1)}

    return {}  # default result for this kind of functions


def mk_details_from_errors(prefix):
    def details_fetcher(host, reason, decision_params):
        if "errors" in decision_params:
            errors = "\n".join("* " + error for error in decision_params["errors"])
            return "{reason}\n\n{prefix}:\n{errors}".format(reason=reason, prefix=prefix, errors=errors)
        else:
            return reason

    return details_fetcher


def details_for_crc_errors(host, reason, decision_params):
    return (
        "{}\nYASM graph: https://yasm.yandex-team.ru/chart/itype=common;hosts={};"
        "signals=netstat-ierrs_summ/\n".format(reason, host.name)
    )


def mk_request_id_parts_fetcher(keys):
    def request_id_extra_parts(decision_params):
        # we need to filter None anyways because value itself may be None
        return tuple(filter(None, (decision_params.get(k) for k in keys)))

    return request_id_extra_parts


class RequestTypes:
    IPMI_HOST_MISSING = _RequestPattern(
        "check_ipmi",
        107,
        "ipmi-host-missing",
        "broken IPMI",
        params_fetcher=params_fetcher_with_default_eine_code(EineCode.IPMI_DNS_RESOLUTION_FAILED),
    )

    IPMI_UNREACHABLE = _RequestPattern(
        "check_ipmi",
        105,
        "ipmi-unreachable",
        "broken IPMI",
        params_fetcher=params_fetcher_with_default_eine_code(EineCode.IPMI_PROTO_ERROR),
    )

    MALFUNCTIONING_LINK = _RequestPattern("check_eth", 101, "malfunctioning-link", "malfunctioning link")
    CORRUPTED_MEMORY = _RequestPattern(
        "memory",
        125,
        "corrupted-memory",
        "corrupted memory",
        params_fetcher=mk_params_fetcher({"slot"}),
        details_fetcher=mk_details_from_errors("ECC errors"),
        request_id_parts=mk_request_id_parts_fetcher(["slot"]),
    )
    UNCORRECTABLE_ERRORS = _RequestPattern(
        "memory",
        126,
        "uncorrectable-errors",
        "uncorrectable errors",
        params_fetcher=mk_params_fetcher({"slot"}),
        details_fetcher=mk_details_from_errors("ECC errors"),
        request_id_parts=mk_request_id_parts_fetcher(["slot"]),
    )
    INVALID_MEMORY_SIZE = _RequestPattern(
        "memory",
        127,
        "invalid-memory-size",
        "invalid memory size",
        params_fetcher=lambda params: {"realmem": params["real"]},  # this is a legacy naming
    )
    INVALID_NUMA_MEMORY_NODES_SIZES = _RequestPattern(
        "memory", 142, "invalid-numa-memory-nodes-sizes", "invalid numa memory nodes sizes"
    )
    LOW_MEMORY_SPEED = _RequestPattern("memory", 141, "low-memory-speed", "low memory speed")
    CORRUPTED_DISK_BY_SLOT = _RequestPattern(
        "hdd",
        123,
        "corrupted-disk",
        "corrupted disk",
        params_fetcher=mk_params_fetcher({"slot", "serial", "diskperformance"}),
        details_fetcher=mk_details_from_errors("Disk errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot", "serial", "shelf_inv")),
    )
    CORRUPTED_DISK_BY_SERIAL = _RequestPattern(
        "hdd",
        121,
        "corrupted-disk",
        "corrupted disk",
        params_fetcher=mk_params_fetcher({"slot", "serial", "diskperformance"}),
        details_fetcher=mk_details_from_errors("Disk errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot", "serial", "shelf_inv")),
    )
    BAD_DISK_CABLE = _RequestPattern(
        "hdd",
        134,
        "bad-disk-cable",
        "bad disk cable",
        params_fetcher=mk_params_fetcher({"slot", "serial"}),
        details_fetcher=mk_details_from_errors("Disk errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot", "serial", "shelf_inv")),
    )
    BMC_LOW_BATTERY = _RequestPattern("body", 136, "bmc-low-battery", "bmc low battery")
    BMC_LOW_VOLTAGE = _RequestPattern("body", 140, "bmc-low-voltage", "bmc low voltage")
    CPU_FAILED = _RequestPattern(
        "body",
        135,
        "cpu-failed",
        "cpu failure",
        params_fetcher=mk_params_fetcher({"slot"}),
        details_fetcher=mk_details_from_errors("CPU errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot",)),
    )
    CPU_OVERHEATED = _RequestPattern("body", 137, "cpu-overheated", "cpu overheated")
    CPU_CAPPED = _RequestPattern("body", 138, "cpu-capped", "cpu capped")
    MALFUNCTIONING_LINK_RX_CRC_ERRORS = _RequestPattern(
        "check_eth",
        139,
        "malfunctioning-link-rx-crc-errors",
        "malfunctioning link RX CRC errors",
        details_fetcher=details_for_crc_errors,
    )

    GPU_MISSING = _RequestPattern(
        "gpu",
        148,
        "gpu-missing",
        "GPU missing",
        params_fetcher=mk_params_for_gpu,
        details_fetcher=mk_details_from_errors("GPU errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot", "serial")),
    )
    GPU_OVERHEATED = _RequestPattern(
        "gpu",
        149,
        "gpu-overheated",
        "GPU overheated",
        params_fetcher=mk_params_for_gpu,
        details_fetcher=mk_details_from_errors("GPU errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot",)),
    )
    GPU_BANDWIDTH_TOO_LOW = _RequestPattern(
        "gpu",
        153,
        "gpu-bandwidth-too-low",
        "GPU bandwidth too low",
        params_fetcher=mk_params_for_gpu,
        details_fetcher=mk_details_from_errors("GPU errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot",)),
    )
    GPU_CAPPING = _RequestPattern(
        "gpu",
        148,
        "gpu-capped",
        "GPU capped",
        params_fetcher=mk_params_for_gpu,
        details_fetcher=mk_details_from_errors("GPU errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot",)),
    )
    GPU_RETIRED_PAGES = _RequestPattern(
        "gpu",
        148,
        "gpu-retired-pages",
        "GPU retired pages",
        params_fetcher=mk_params_for_gpu,
        details_fetcher=mk_details_from_errors("GPU errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot",)),
    )
    GPU_POWER_CAPPING = _RequestPattern(
        "gpu",
        148,
        "gpu-capped-power",
        "GPU capped power",
        params_fetcher=mk_params_for_gpu,
        details_fetcher=mk_details_from_errors("GPU errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot",)),
    )
    GPU_INFOROM_CORRUPTED = _RequestPattern(
        "gpu",
        148,
        "gpu-inforom-corrupted",
        "GPU inforom corrupted",
        params_fetcher=mk_params_for_gpu,
        details_fetcher=mk_details_from_errors("GPU errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot",)),
    )
    GPU_HANG = _RequestPattern(
        "gpu",
        148,
        "gpu-hang",
        "GPU hang",
        params_fetcher=mk_params_for_gpu,
        details_fetcher=mk_details_from_errors("GPU errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot",)),
    )
    GPU_RETIRED_PAGES_PENDING = _RequestPattern(
        "gpu",
        148,
        "gpu-retired-pages-pending",
        "GPU retired pages pending",
        params_fetcher=mk_params_for_gpu,
        details_fetcher=mk_details_from_errors("GPU errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot",)),
    )
    GPU_P2P_FAILED = _RequestPattern(
        "gpu",
        148,
        "gpu-p2p-failed",
        "GPU p2p failed",
        params_fetcher=mk_params_for_gpu,
        details_fetcher=mk_details_from_errors("GPU errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot",)),
    )
    GPU_POWER_UNKNOWN = _RequestPattern(
        "gpu",
        148,
        "gpu-power-unknown",
        "GPU power unknown",
        params_fetcher=mk_params_for_gpu,
        details_fetcher=mk_details_from_errors("GPU errors"),
        request_id_parts=mk_request_id_parts_fetcher(("slot",)),
    )
    INFINIBAND_MISMATCH = _RequestPattern(
        "check_eth",
        2,
        "infiniband-mismatch",
        "Infiniband mismatch",
    )
    INFINIBAND_ERR = _RequestPattern(
        "check_eth",
        2,
        "infiniband-err",
        "Infiniband error",
    )
    INFINIBAND_INVALID_PHYS_STATE = _RequestPattern(
        "check_eth",
        2,
        "infiniband-invalid-phys-state",
        "Infiniband invalid phys state",
    )
    PCIE_DEVICE_BANDWIDTH_TOO_LOW = _RequestPattern(
        "body",
        154,
        "pcie-device-bandwidth-too-low",
        "PCIE device bandwidth too low",
    )
    SECOND_TIME_NODE = _RequestPattern("body", 154, "report-second-time-node", "reoccurring problems with host")
    REBOOT = _RequestPattern("reboot", 129, "reboot", "Reboot host")
    INFINIBAND_LOW_SPEED = _RequestPattern("check_eth", 2, "infiniband-low-speed", "Infiniband low speed")

    INFINIBAND_INVALID_STATE = _NocRequestPattern(
        "infiniband-invalid-state",
        "https://wiki.yandex-team.ru/users/kitaro/infinibandinvalidstate-repairing/",
    )

    ALL_IPMI = (IPMI_HOST_MISSING, IPMI_UNREACHABLE)

    ALL_NOC = (INFINIBAND_INVALID_STATE,)

    ALL_TYPES = (
        IPMI_HOST_MISSING,
        IPMI_UNREACHABLE,
        MALFUNCTIONING_LINK,
        CORRUPTED_MEMORY,
        INVALID_MEMORY_SIZE,
        INVALID_NUMA_MEMORY_NODES_SIZES,
        LOW_MEMORY_SPEED,
        CORRUPTED_DISK_BY_SLOT,
        CORRUPTED_DISK_BY_SERIAL,
        BMC_LOW_BATTERY,
        BAD_DISK_CABLE,
        CPU_FAILED,
        CPU_OVERHEATED,
        CPU_CAPPED,
        GPU_CAPPING,
        GPU_RETIRED_PAGES,
        GPU_MISSING,
        GPU_OVERHEATED,
        GPU_BANDWIDTH_TOO_LOW,
        GPU_POWER_CAPPING,
        GPU_HANG,
        GPU_INFOROM_CORRUPTED,
        MALFUNCTIONING_LINK_RX_CRC_ERRORS,
        UNCORRECTABLE_ERRORS,
        SECOND_TIME_NODE,
        REBOOT,
        INFINIBAND_MISMATCH,
        INFINIBAND_INVALID_PHYS_STATE,
        PCIE_DEVICE_BANDWIDTH_TOO_LOW,
        INFINIBAND_ERR,
        GPU_RETIRED_PAGES_PENDING,
        GPU_P2P_FAILED,
        GPU_POWER_UNKNOWN,
        INFINIBAND_INVALID_STATE,
        INFINIBAND_LOW_SPEED,
    )
    ALL_TYPE_NAMES = tuple({t.type for t in ALL_TYPES})
    ALL_BY_TYPENAME = {t.type: t for t in ALL_TYPES}

    @classmethod
    def by_typename(cls, typename):
        try:
            return cls.ALL_BY_TYPENAME[typename]
        except KeyError:
            raise LogicalError
