import collections
import datetime
import logging

import humanize
import mongoengine
import simplejson as json
from mongoengine import StringField, LongField, IntField, Q, DoesNotExist

from object_validator import String, validate, ValidationError
from sepelib.core import config
from sepelib.core.constants import WEEK_SECONDS, DAY_SECONDS
from sepelib.core.exceptions import Error
from sepelib.mongo.util import register_model
from walle import audit_log, authorization
from walle.admin_requests.constants import (
    RequestTypes,
    STATUS_PROCESSED,
    STATUS_NOT_EXIST,
    STATUS_IN_PROCESS,
    STATUS_DELETED,
    _NocRequestPattern,
)
from walle.admin_requests.severity import admin_request_severity_tag_factory, RequestSource, BotTag
from walle.authorization import ISSUER_WALLE
from walle.clients import bot, startrek
from walle.clients.utils import strip_api_error
from walle.hosts import Host, HostStatus
from walle.locks import HostInterruptableLock
from walle.models import Document, timestamp
from walle.util import notifications
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.misc import format_time
from walle.util.template_loader import JinjaTemplateRenderer
from walle.util.validation import ApiDictScheme, StringToInteger


_REQUEST_PROCESSING_TIMEOUT = WEEK_SECONDS
"""Time that we expect is enough for DC engineers to process any request."""

_REQUEST_RESOLUTION_TIMEOUT = DAY_SECONDS
"""If problem repeats before the timeout has elapsed after request processing it's considered as permanent."""

_NOCREQUESTS_ST_QUEUE = "NOCREQUESTS"

log = logging.getLogger(__name__)

Result = collections.namedtuple('Result', ['bot_id', 'ticket'])


@register_model
class _AdminRequest(Document):
    id = StringField(primary_key=True, required=True, help_text="Request ID")
    time = LongField(required=True, help_text="Time when the request has been created")
    type = StringField(choices=RequestTypes.ALL_TYPE_NAMES, required=True, help_text="Request type")
    bot_id = IntField(required=True, help_text="Bot request ID")
    host_inv = IntField(required=True, help_text="Target host inventory number (when applicable)")
    host_name = StringField(help_text="Target host name (when applicable and available)")
    host_uuid = StringField(required=True, help_text="Target host UUID")

    meta = {"collection": "admin_requests"}

    def cancel(self):
        _cancel_request(self.bot_id)

        # We should delete the objects carefully, since we use save() for adding new requests to the database
        _AdminRequest.objects(id=self.id, bot_id=self.bot_id).delete()


def create_admin_request(host, request_type, reason, **decision_params):
    """
    :type host: Host
    :type request_type: _RequestPattern
    :type reason: str|unicode
    """

    with audit_log.on_admin_request(
        host.project,
        host.inv,
        host.name,
        host.uuid,
        request_type.type,
        reason,
        decision_params,
        scenario_id=host.scenario_id,
    ) as audit_entry:
        detailed_reason = request_type.details(host, reason, decision_params)
        params = request_type.params(decision_params)
        request_id_extra_parts = request_type.request_id_extra_parts(dict(decision_params, **params))
        severity_tag = admin_request_severity_tag_factory(host, RequestSource.BOT)

        result = _create(
            ISSUER_WALLE,
            request_type,
            host,
            params=params,
            reason=detailed_reason,
            request_id_extra_parts=request_id_extra_parts,
            shelf_inv=decision_params.get("shelf_inv"),
            severity_tag=severity_tag,
        )

        audit_entry.modify(payload={"ticket": result.ticket})

    return result.bot_id, result.ticket


def get_last_request_status(request_type, host_inv, request_id_extra_parts=tuple()):
    try:
        admin_request = _AdminRequest.objects.only("bot_id", "time").get(
            id=_request_id(request_type, host_inv, *request_id_extra_parts)
        )
    except mongoengine.DoesNotExist:
        return

    request = get_request_status(admin_request.bot_id)

    if request["status"] == STATUS_PROCESSED and timestamp() - request["close_time"] >= _REQUEST_RESOLUTION_TIMEOUT:
        return

    request["create_time"] = admin_request.time
    return request


def get_request_status(request_id):
    info = bot.raw_request("/api/status.php", params={"id": request_id, "out": "json"}, authenticate=True)
    if info in ("UNKNOWN", "FALSE"):
        return {"request_id": request_id, "status": STATUS_NOT_EXIST, "info": info}

    try:
        info = json.loads(info)
    except ValueError:
        log.error("Got an invalid response from BOT for #%s request status: %s.", request_id, info)
        raise bot.BotInternalError("Got an invalid response.")

    try:
        info = validate(
            "result",
            info,
            ApiDictScheme(
                {
                    "Status": String(),
                    "Initiator": String(),
                    "TimeCreate": StringToInteger(),
                    "TimeFinish": StringToInteger(optional=True),
                    "StNum": String(optional=True),
                },
                drop_none=True,
            ),
        )
    except ValidationError as e:
        log.error("Got an invalid JSON response from BOT for #%s request status (%s): %s.", request_id, info, e)
        raise bot.BotInternalError("The server returned an invalid JSON response.")

    status = info["Status"]
    request = {
        "request_id": request_id,
        "initiator": info["Initiator"],
        "info": status,
        "ticket": info["StNum"],
    }

    if status in ("NEW", "WORK"):
        request["status"] = STATUS_IN_PROCESS

        in_process_time = timestamp() - info["TimeCreate"]
        if in_process_time >= _REQUEST_PROCESSING_TIMEOUT:
            log.warn(
                "#%s admin request is being processed too much time: it was created %s.",
                request_id,
                humanize.naturaltime(datetime.timedelta(seconds=in_process_time)),
            )

    elif status == "OK":
        if "TimeFinish" not in info:
            raise bot.BotInternalError("Got a processed #{} request without TimeFinish field.", request_id)

        request.update(
            {
                "status": STATUS_PROCESSED,
                "close_time": info["TimeFinish"],
            }
        )
    elif status in ("FAULT", "DELETED"):
        request["status"] = STATUS_DELETED
    else:
        raise bot.BotInternalError("Got an unknown #{} request status: {}", request_id, status)

    return request


def get_request_owners(project_id, issuer, initiated_by_issuer):
    # Request initiator login
    initiator = config.get_value("bot.request_initiator")
    if initiated_by_issuer and authorization.is_user(issuer):
        initiator = authorization.get_issuer_login(issuer)

    # To email addresses
    to = set(config.get_value("bot.request_reply_to"))

    # CC email addresses
    cc = set(config.get_value("bot.request_cc", default=[]))

    # Add to CC user that created the task
    issuer_email = authorization.get_user_email(issuer)
    if issuer_email is not None:
        (to if initiated_by_issuer else cc).add(issuer_email)

    recipients_config = notifications.get_recipients_config(project_id)

    # We have a special separated severity for all requests to Bot
    cc.update(recipients_config.get(notifications.SEVERITY_BOT, []))

    # Also add to CC all recipients for AUDIT severity
    cc.update(notifications.get_recipients_by_severity(notifications.SEVERITY_AUDIT, recipients_config))

    return initiator, to, cc


def cancel_all_by_host(inv, name=None, types=None):
    query = Q(host_inv=inv)
    if name is not None:
        query |= Q(host_name=name)

    if types is not None:
        query &= Q(type__in=[t.type for t in types])

    for request in _AdminRequest.objects(query):
        request.cancel()


def _create(
    initiator,
    request_type,
    host,
    params,
    reason,
    request_id_extra_parts=tuple(),
    shelf_inv=None,
    severity_tag=BotTag.MEDIUM,
):
    log.warning("Creating '%s' admin request for #%s host...", request_type.type, host.inv)

    try:
        # We should not pass host's fqdn for the shelves.
        (inv, fqdn) = (host.inv, host.name) if shelf_inv is None else (shelf_inv, None)
        result = _create_in_bot(
            initiator,
            request_type,
            host.project,
            host_inv=inv,
            host_name=fqdn,
            params=params,
            reason=reason,
            severity_tag=severity_tag,
        )

        # We store host.inv in the document so we can link to the request from the host.
        # We don't store shelf's inv explicitly, but it goes into request_id_extra_parts.
        request = _AdminRequest(
            id=_request_id(request_type, host.inv, *request_id_extra_parts),
            time=timestamp(),
            type=request_type.type,
            bot_id=result.bot_id,
            host_inv=host.inv,
            host_uuid=host.uuid,
            host_name=host.name,
        )
        request.save()
    except Exception as e:
        raise Error("Failed to create an admin request: {}", e)

    return result


def _create_in_bot(
    initiator, request_type, project_id, host_inv, params, reason, host_name=None, severity_tag=BotTag.MEDIUM
):

    admin_requests = bot.get_admin_requests(host_inv, request_type.operation, 90 * DAY_SECONDS, limit=5)
    if admin_requests:
        reason += "\n\nPrevious tickets for {description} for #{inv}:".format(
            description=request_type.description, inv=host_inv
        )
        for admin_request in admin_requests:
            reason += "\n* {time} by {issuer}".format(
                time=format_time(admin_request["time"], "%Y.%m.%d %H:%M"), issuer=admin_request["issuer"]
            )
            if "url" in admin_request:
                reason += " ({url})".format(url=admin_request["url"])

    request_initiator, to, cc = get_request_owners(project_id, initiator, initiated_by_issuer=False)

    request = dict(
        params,
        **{
            "initiator": request_initiator,
            "operation": request_type.operation,
            "problem": request_type.problem,
            "email": "yes",
            "replyto": ",".join(to),
            "cc": ",".join(cc),
            "severity": severity_tag,
        },
    )

    # Create request for host name if available to stress the fact for which host we create the request in case if
    # host inventory number has been changed without our knowledge.
    if host_name is None:
        request["inv"] = host_inv
    else:
        request["name"] = host_name

    if reason:
        request["comment"] = reason

    response = bot.json_request(
        path="/api/v3/dc/request",
        params=request,
        authenticate=True,
        scheme=ApiDictScheme(
            {
                "result": ApiDictScheme({"Id": StringToInteger(), "Status": String(), "StNum": String()}),
            }
        ),
    )

    result = response["result"]
    if result["Status"] not in ("NEW", "WORK"):
        log.error("Failed to create an admin request in Bot: %s.", response)
        raise Error("Failed to create an admin request in Bot.")

    return Result(bot_id=result['Id'], ticket=result['StNum'])


def _cancel_request(request_id):
    """
    Cancels the specified admin request.

    /api/feedback.php?operation=cancel cancels ticket by switching it to DELETED status, but it has a side effect - it
    sets Wall-E as resolver of the ticket which breaks DC statistics, so DC guys ask us to use
    /api/feedback.php?operation=comment instead which only comments to ticket and don't close it.
    """

    log.debug("Cancelling #%s admin request...", request_id)

    bot_request = get_request_status(request_id)

    if bot_request["status"] != STATUS_IN_PROCESS:
        log.debug("Skip cancellation of #%s admin request: it's already closed.", request_id)
        return

    if bot_request["status"] == STATUS_PROCESSED:
        log.debug("Skip cancellation of #%s admin request: it's processed.", request_id)
        return

    # BOT may return us someone else's request instead creating a new one when we create requests, but we must
    # cancel only our requests.
    if bot_request["initiator"] != config.get_value("bot.request_initiator"):
        log.debug("Skip cancellation of #%s admin request: it doesn't belong to Wall-E.", request_id)
        return

    log.warning("Cancelling #%s admin request...", request_id)

    result = bot.raw_request(
        "/api/feedback.php",
        params={
            "id": request_id,
            "initiator": config.get_value("bot.request_initiator"),
            "operation": "cancel",
            "oauth_token": config.get_value("bot.access_token"),
        },
    )

    if result in ("OK", "ERROR: This ticket is not new"):
        return

    log.error("Got an unexpected response from BOT: %r.", result)

    bot_request = get_request_status(request_id)
    if bot_request != STATUS_IN_PROCESS:
        return

    raise bot.BotInternalError("Failed to cancel #{} admin request: {}", request_id, strip_api_error(result))


def _request_id(request_type, host_inv, *extra_parts):
    return "/".join(str(part) for part in (request_type.type, host_inv) + extra_parts)


def _gc_admin_requests():
    for request in gevent_idle_iter(_AdminRequest.objects):
        if _is_abandoned_request(request.host_uuid):
            host = Host.objects(uuid=request.host_uuid).only("tier").first()
            if host:
                tier = host.tier
            else:
                log.info(f"Can't find host {request.host_uuid} for Admin requests GC")
                tier = None
            with HostInterruptableLock(request.host_uuid, tier):
                if _is_abandoned_request(request.host_uuid):
                    request.cancel()


def _is_abandoned_request(uuid):
    try:
        host = Host.objects.only("status").get(uuid=uuid)
    except DoesNotExist:
        return True

    if host.status not in HostStatus.ALL_TASK:
        return True

    return False


def create_noc_request(host: Host, request_type: _NocRequestPattern, reason: str, **decision_params):
    with audit_log.on_admin_request(
        host.project,
        host.inv,
        host.name,
        host.uuid,
        request_type.type,
        reason,
        decision_params,
        scenario_id=host.scenario_id,
    ) as audit_entry:
        try:
            client = startrek.get_client()
            response = client.create_issue(
                {
                    "queue": _NOCREQUESTS_ST_QUEUE,
                    "type": "incident",
                    "summary": f"Ошибка на хосте {host.name}: {decision_params['eine_code']}",
                    "description": _create_noc_request_desc(host, request_type, reason),
                }
            )
            ticket_id = response["key"]
            audit_entry.modify(payload={"ticket": ticket_id})
        except Exception as e:
            raise Error("Failed to create NOC ticket: {}", e)

    return ticket_id


def _create_noc_request_desc(host: Host, request_type: _NocRequestPattern, reason: str):
    return JinjaTemplateRenderer().render_template(
        "noc_request_ticket_description.txt", host=host, request_type=request_type, reason=reason
    )
