"""Wall-E audit logs."""

import logging
import time
import uuid

from mongoengine import StringField, IntField, FloatField, DictField, DoesNotExist

from sepelib.core.exceptions import Error
from sepelib.mongo.util import register_model
from walle import authorization
from walle.admin_requests.constants import RequestTypes
from walle.errors import ApiError
from walle.models import Document
from walle.util import notifications
from walle.util.misc import drop_none

log = logging.getLogger(__name__)

TYPE_CHANGE_GLOBAL_SETTINGS = "change-global-settings"
TYPE_CHANGE_CHECKS_PERCENTAGE_OVERRIDES = "change-checks-percentage-overrides"
TYPE_CHANGE_GLOBAL_TIMED_LIMITS_OVERRIDES = "change-global-timed-limits-overrides"

TYPE_ADD_DMC_RULE = "add-dmc-rule"
TYPE_DELETE_DMC_RULE = "delete-dmc-rule"

TYPE_ENABLE_AUTOMATION = "enable-automation"
TYPE_DISABLE_AUTOMATION = "disable-automation"

TYPE_ENABLE_AUTOMATED_HEALING = "enable-automated-healing"
TYPE_DISABLE_AUTOMATED_HEALING = "disable-automated-healing"

TYPE_ENABLE_DNS_AUTOMATION = "enable-dns-automation"
TYPE_DISABLE_DNS_AUTOMATION = "disable-dns-automation"

TYPE_ADD_PROJECT = "add-project"
TYPE_UPDATE_PROJECT = "update-project"
TYPE_PROJECT_OWNERS_UPDATE_REQUEST = "project-owners-update-request"
TYPE_PROJECT_ROLE_UPDATE_REQUEST = "project-role-update-request"
TYPE_CLONE_PROJECT = "clone-project"
TYPE_DELETE_PROJECT = "delete-project"
TYPE_PROJECT_ADD_OWNER = "project-add-owner"
TYPE_PROJECT_REMOVE_OWNER = "project-remove-owner"
TYPE_PROJECT_REQUEST_TOGGLE_REBOOT_VIA_SSH = "project-request-toggle-reboot-via-ssh"
TYPE_PROJECT_TOGGLE_REBOOT_VIA_SSH = "project-toggle-reboot-via-ssh"
TYPE_PROJECT_ROLE_ADD_MEMBER = "project-role-add-member"
TYPE_PROJECT_ROLE_REMOVE_MEMBER = "project-role-remove-member"


TYPE_FSM_HANDBRAKE = "fsm-handbrake"
TYPE_SCENARIO_FSM_HANDBRAKE = "scenario-fsm-handbrake"
TYPE_EXTEND_FSM_HANDBRAKE = "extend-fsm-handbrake"

TYPE_ADD_PREORDER = "add-preorder"
TYPE_PROCESS_PREORDER = "process-preorder"
TYPE_RESTART_PREORDER = "restart-preorder"
TYPE_DELETE_PREORDER = "delete-preorder"

TYPE_ADD_AUTOMATION_PLOT = "add-automation-plot"
TYPE_UPDATE_AUTOMATION_PLOT = "update-automation-plot"
TYPE_DELETE_AUTOMATION_PLOT = "delete-automation-plot"

TYPE_ADD_MAINTENANCE_PLOT = "add-maintenance-plot"
TYPE_UPDATE_MAINTENANCE_PLOT = "update-maintenance-plot"
TYPE_DELETE_MAINTENANCE_PLOT = "delete-maintenance-plot"

TYPE_ADD_HOST = "add-host"
TYPE_UPDATE_HOST = "update-host"
TYPE_SET_HOST_MAINTENANCE = "set-host-maintenance"
TYPE_CHANGE_HOST_MAINTENANCE = "change-host-maintenance"
TYPE_SET_HOST_STATE = "set-host-state"
TYPE_FORCE_HOST_STATUS = "force-host-status"
TYPE_INVALIDATE_HOST = "invalidate-host"
TYPE_CANCEL_HOST_TASK = "cancel-host-task"
TYPE_DELETE_HOST = "delete-host"
TYPE_HOST_INV_CHANGED = "host-inv-changed"
TYPE_HOST_FQDN_CHANGED = "host-fqdn-changed"
TYPE_HOST_CAUTH_SETTINGS_UPDATED = "host-cauth-settings-updated"

TYPE_ADD_SCENARIO = "add-scenario"
TYPE_DELETE_SCENARIO = "delete-scenario"
TYPE_CANCEL_SCENARIO = "cancel-scenario"
TYPE_PAUSE_SCENARIO = "pause-scenario"
TYPE_MODIFY_SCENARIO = "modify-scenario"
TYPE_STARTED_SCENARIO = "start-scenario"
TYPE_FINISHED_SCENARIO = "finished-scenario"

TYPE_SWITCH_VLAN = "switch-vlan"
TYPE_SWITCH_PROJECT = "switch-project"
TYPE_RELEASE_HOST = "release-host"
TYPE_FIX_DNS_RECORDS = "fix-dns-records"
TYPE_CLEAR_DNS_RECORDS = "clear-dns-records"
TYPE_CHECK_DNS_RECORDS = "check-dns-records"

TYPE_PREPARE_HOST = "prepare-host"
TYPE_POWER_ON_HOST = "power-on-host"
TYPE_POWER_OFF_HOST = "power-off-host"
TYPE_REBOOT_HOST = "reboot-host"
TYPE_KEXEC_REBOOT_HOST = "kexec-reboot-host"
TYPE_PROFILE_HOST = "profile-host"
TYPE_REDEPLOY_HOST = "redeploy-host"
TYPE_REPORT_HOST_FAILURE = "report-host-failure"
TYPE_REPAIR_RACK_FAILURE = "repair-rack-failure"
TYPE_DEACTIVATE_HOST = "deactivate-host"
TYPE_HARDWARE_REPAIR = "hardware-repair"
TYPE_REPAIR_LINK = "repair-link"
TYPE_CHANGE_MEMORY = "change-memory"
TYPE_CHANGE_DISK = "change-disk"
TYPE_RESET_BMC = "reset-bmc"
TYPE_REPAIR_BMC = "repair-bmc"
TYPE_REPAIR_OVERHEAT = "repair-overheat"
TYPE_REPAIR_CAPPING = "repair-capping"
TYPE_BOT_PROJECT_SYNC = "bot-project-sync"
TYPE_BOT_WAIT_FOR_HOST_ACQUIREMENT = "bot-wait-for-host-acquirement"
TYPE_FOOBAR_HOST = "foobar-host"
TYPE_FQDN_DEINVALIDATION = "fqdn-deinvalidation"

TYPE_NO_ACTION = "no-action"
"""
This type indicates that failure was registered but automation was turned off
and no action was taken in order to heal the host.
"""

TYPE_REPORT = "report"
"""This type stores ticket created for any reason for a host or a group of hosts"""


TYPE_ADMIN_REQUEST_IPMI_HOST_MISSING = "admin-request-ipmi-host-missing"
TYPE_ADMIN_REQUEST_IPMI_UNREACHABLE = "admin-request-ipmi-unreachable"
TYPE_ADMIN_REQUEST_CORRUPTED_DISK = "admin-request-corrupted-disk"
TYPE_ADMIN_REQUEST_CORRUPTED_MEMORY = "admin-request-corrupted-memory"
TYPE_ADMIN_REQUEST_INVALID_MEMORY_SIZE = "admin-request-invalid-memory-size"
TYPE_ADMIN_REQUEST_INVALID_NUMA_MEMORY_NODES_SIZES = "admin-request-invalid-numa-memory-nodes-sizes"
TYPE_ADMIN_REQUEST_LOW_MEMORY_SPEED = "admin-request-low-memory-speed"
TYPE_ADMIN_REQUEST_MALFUNCTIONING_LINK = "admin-request-malfunctioning-link"
TYPE_ADMIN_REQUEST_MALFUNCTIONING_LINK_RX_CRC_ERRORS = "admin-request-malfunctioning-link-rx-crc-errors"
TYPE_ADMIN_REQUEST_BMC_LOW_BATTERY = "admin-request-bmc-low-battery"
TYPE_ADMIN_REQUEST_CPU_OVERHEATED = "admin-request-cpu-overheated"
TYPE_ADMIN_REQUEST_CPU_CAPPED = "admin-request-cpu-capped"


def _type_admin_request(request_type_name):
    return "admin-request-{}".format(request_type_name)


ADMIN_REQUEST_TYPES = [_type_admin_request(_request_type_name) for _request_type_name in RequestTypes.ALL_TYPE_NAMES]

TYPE_HOST_MACS_CHANGED = "host-macs-changed"
TYPE_ACTIVE_MAC_CHANGE_DETECTED = "active-mac-change-detected"
TYPE_SWITCH_PORT_CHANGE_DETECTED = "switch-port-change-detected"
TYPE_IPS_CHANGE_DETECTED = "ips-change-detected"

AUTOMATION_TYPES = [
    TYPE_REBOOT_HOST,
    TYPE_PROFILE_HOST,
    TYPE_REDEPLOY_HOST,
    TYPE_REPAIR_RACK_FAILURE,
    TYPE_REPORT_HOST_FAILURE,
    TYPE_DEACTIVATE_HOST,
    TYPE_REPAIR_LINK,
    TYPE_CHANGE_MEMORY,
    TYPE_CHANGE_DISK,
    TYPE_HARDWARE_REPAIR,
    TYPE_RESET_BMC,
    TYPE_REPAIR_BMC,
    TYPE_REPAIR_OVERHEAT,
    TYPE_REPAIR_CAPPING,
    TYPE_FIX_DNS_RECORDS,
]

INSTANT_TYPES = [
    TYPE_CHANGE_GLOBAL_SETTINGS,
    TYPE_ENABLE_AUTOMATION,
    TYPE_DISABLE_AUTOMATION,
    TYPE_ENABLE_AUTOMATED_HEALING,
    TYPE_DISABLE_AUTOMATED_HEALING,
    TYPE_ENABLE_DNS_AUTOMATION,
    TYPE_DISABLE_DNS_AUTOMATION,
    TYPE_EXTEND_FSM_HANDBRAKE,
    #
    TYPE_ADD_PROJECT,
    TYPE_UPDATE_PROJECT,
    TYPE_PROJECT_OWNERS_UPDATE_REQUEST,
    TYPE_CLONE_PROJECT,
    TYPE_DELETE_PROJECT,
    TYPE_PROJECT_ADD_OWNER,
    TYPE_PROJECT_REMOVE_OWNER,
    TYPE_PROJECT_ROLE_UPDATE_REQUEST,
    TYPE_PROJECT_ROLE_ADD_MEMBER,
    TYPE_PROJECT_ROLE_REMOVE_MEMBER,
    TYPE_PROJECT_REQUEST_TOGGLE_REBOOT_VIA_SSH,
    TYPE_PROJECT_TOGGLE_REBOOT_VIA_SSH,
    TYPE_ADD_AUTOMATION_PLOT,
    TYPE_UPDATE_AUTOMATION_PLOT,
    TYPE_DELETE_AUTOMATION_PLOT,
    TYPE_ADD_PREORDER,
    TYPE_DELETE_PREORDER,
    TYPE_ADD_MAINTENANCE_PLOT,
    TYPE_UPDATE_MAINTENANCE_PLOT,
    TYPE_DELETE_MAINTENANCE_PLOT,
    #
    TYPE_UPDATE_HOST,
    TYPE_CHANGE_HOST_MAINTENANCE,
    TYPE_FORCE_HOST_STATUS,
    TYPE_INVALIDATE_HOST,
    TYPE_CANCEL_HOST_TASK,
    TYPE_HOST_INV_CHANGED,
    TYPE_HOST_FQDN_CHANGED,
    TYPE_HOST_MACS_CHANGED,
    TYPE_ACTIVE_MAC_CHANGE_DETECTED,
    TYPE_FIX_DNS_RECORDS,
    TYPE_CLEAR_DNS_RECORDS,
    TYPE_NO_ACTION,
    TYPE_CHANGE_CHECKS_PERCENTAGE_OVERRIDES,
    TYPE_CHANGE_GLOBAL_TIMED_LIMITS_OVERRIDES,
    TYPE_SWITCH_PORT_CHANGE_DETECTED,
    TYPE_IPS_CHANGE_DETECTED,
    #
    TYPE_ADD_SCENARIO,
    TYPE_DELETE_SCENARIO,
    TYPE_MODIFY_SCENARIO,
    TYPE_STARTED_SCENARIO,
    TYPE_CANCEL_SCENARIO,
    TYPE_FINISHED_SCENARIO,
    TYPE_PAUSE_SCENARIO,
    #
    TYPE_ADD_DMC_RULE,
    TYPE_DELETE_DMC_RULE,
    #
] + ADMIN_REQUEST_TYPES

CONTINUOUS_TYPES = [
    # project, global
    TYPE_FSM_HANDBRAKE,
    TYPE_SCENARIO_FSM_HANDBRAKE,
    TYPE_PROCESS_PREORDER,
    TYPE_RESTART_PREORDER,
    # hosts
    TYPE_ADD_HOST,
    TYPE_DELETE_HOST,
    TYPE_SWITCH_PROJECT,
    TYPE_RELEASE_HOST,
    TYPE_SET_HOST_MAINTENANCE,
    TYPE_SET_HOST_STATE,
    TYPE_HOST_CAUTH_SETTINGS_UPDATED,
    TYPE_POWER_ON_HOST,
    TYPE_POWER_OFF_HOST,
    TYPE_REBOOT_HOST,
    TYPE_KEXEC_REBOOT_HOST,
    TYPE_PREPARE_HOST,
    TYPE_PROFILE_HOST,
    TYPE_REDEPLOY_HOST,
    TYPE_REPAIR_RACK_FAILURE,
    TYPE_REPORT_HOST_FAILURE,
    TYPE_DEACTIVATE_HOST,
    TYPE_REPAIR_LINK,
    TYPE_CHANGE_MEMORY,
    TYPE_CHANGE_DISK,
    TYPE_HARDWARE_REPAIR,
    TYPE_RESET_BMC,
    TYPE_REPAIR_BMC,
    TYPE_REPAIR_OVERHEAT,
    TYPE_REPAIR_CAPPING,
    TYPE_SWITCH_VLAN,
    TYPE_CHECK_DNS_RECORDS,
    TYPE_REPORT,
    TYPE_BOT_PROJECT_SYNC,
    TYPE_BOT_WAIT_FOR_HOST_ACQUIREMENT,
    TYPE_FOOBAR_HOST,
    TYPE_FQDN_DEINVALIDATION,
]
TYPES = INSTANT_TYPES + CONTINUOUS_TYPES


STATUS_UNKNOWN = "unknown"
STATUS_ACCEPTED = "accepted"
STATUS_REJECTED = "rejected"
STATUS_COMPLETED = "completed"
STATUS_CANCELLED = "cancelled"
STATUS_FAILED = "failed"
STATUSES = [STATUS_UNKNOWN, STATUS_ACCEPTED, STATUS_REJECTED, STATUS_COMPLETED, STATUS_CANCELLED, STATUS_FAILED]


@register_model
class LogEntry(Document):
    """Represents an audit log record."""

    id = StringField(primary_key=True, required=True, help_text="A unique ID")
    time = FloatField(required=True, help_text="Time when the request has been received")
    issuer = StringField(required=True, help_text="Request issuer")
    type = StringField(choices=TYPES, required=True, help_text="Request type")

    project = StringField(help_text="Target project ID (when applicable)")
    automation_plot = StringField(help_text="Target automation plot ID (when applicable)")
    maintenance_plot = StringField(help_text="Target maintenance plot ID (when applicable)")
    preorder = IntField(help_text="Target preorder ID (when applicable)")
    scenario_id = IntField(help_text="Target scenario ID (when applicable)")
    host_inv = IntField(help_text="Target host inventory number (when applicable)")
    host_uuid = StringField(help_text="Target host UUID (when applicable)")
    host_name = StringField(help_text="Target host name (when applicable and available)")
    reason = StringField(help_text="Optional request reason.")
    decision = DictField(help_text="Serialized decision for task")
    payload = DictField(
        help_text="An arbitrary payload. The schema not fixed nor defined. Meant exclusively for debugging."
    )

    status = StringField(choices=STATUSES, required=True, help_text="Current request status")
    status_time = FloatField(required=True, help_text="Time of the current status")
    error = StringField(help_text="Error string for failed requests")

    instant = None  # Used to force instantaneity for the event

    default_api_fields = (
        "time",
        "issuer",
        "type",
        "status",
        "status_time",
        "host_inv",
        "host_name",
        "host_uuid",
        "reason",
        "error",
    )
    api_fields = default_api_fields + (
        "id",
        "automation_plot",
        "project",
        "preorder",
        "payload",
        "scenario_id",
        "decision",
        "maintenance_plot",
    )
    task_fields = {
        "issuer",
        "type",
        "project",
        "host_inv",
        "host_name",
        "host_uuid",
        "scenario_id",
        "reason",
        "decision",
    }

    meta = {
        "collection": "audit_log",
        "indexes": [
            {"name": "host_name_only", "fields": ["host_name"]},
            # Speed up iteration without filters
            {"name": "time", "fields": ["-time"]},
            # Speed up iteration with filtering for all fields that can store arbitrary values (which increases
            # probability of full DB scan)
            {"name": "issuer", "fields": ["issuer", "time"]},
            {"name": "automation_plot", "fields": ["automation_plot", "time"]},
            {"name": "project", "fields": ["project", "time"]},
            {"name": "preorder", "fields": ["preorder", "time"]},
            {"name": "host_inv", "fields": ["host_inv", "time"]},
            {"name": "host_uuid", "fields": ["host_uuid", "time"]},
            {"name": "host_name", "fields": ["host_name", "time"]},
            {"name": "host_name_reversed", "fields": ["host_name", "-time"]},
            {"name": "entry_type", "fields": ["type", "time"]},
            {"name": "scenario_id", "fields": ["scenario_id", "time"]},
            {"name": "maintenance_plot", "fields": ["maintenance_plot", "time"]},
        ],
    }

    def cancel(self, reason):
        cancel_request(self, reason)

    def fail(self, error):
        fail_request(self, error)

    def complete(self, extra_payload=None):
        complete_request(self, extra_payload)

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        from_statuses = (STATUS_UNKNOWN,)
        if self.status not in from_statuses:
            return

        if exc_val is None:
            instant = self.type in INSTANT_TYPES if self.instant is None else self.instant
            status = STATUS_COMPLETED if instant else STATUS_ACCEPTED
            error = None
        else:
            if isinstance(exc_val, ApiError):
                status = STATUS_REJECTED
            else:
                status = STATUS_FAILED

            error = str(exc_val)

        try:
            _change_status(self.id, from_statuses, status, error=error, check_updated=False)
        except Exception as e:
            log.critical("Failed to change status of audit log entry '%s' to '%s': %s", self.id, status, e)


def get_task_issuer(task):
    try:
        return LogEntry.objects.only("issuer").get(id=task.audit_log_id).issuer
    except DoesNotExist:
        pass


def on_change_global_settings(issuer, new_settings, reason=None):
    return create(issuer=issuer, type=TYPE_CHANGE_GLOBAL_SETTINGS, payload=new_settings, reason=reason)


def on_change_checks_percentage_overrides(issuer, new_override, reason=None):
    return create(issuer=issuer, type=TYPE_CHANGE_CHECKS_PERCENTAGE_OVERRIDES, payload=new_override, reason=reason)


def on_change_global_timed_limits_overrides(issuer, new_override, reason=None):
    return create(issuer=issuer, type=TYPE_CHANGE_GLOBAL_TIMED_LIMITS_OVERRIDES, payload=new_override, reason=reason)


def on_add_dmc_rule(issuer, rule_query, reason=None):
    return create(issuer=issuer, type=TYPE_ADD_DMC_RULE, payload=dict(rule_query=rule_query), reason=reason)


def on_delete_dmc_rule(issuer, rule_id, reason=None):
    return create(issuer=issuer, type=TYPE_DELETE_DMC_RULE, payload={"rule_id": rule_id}, reason=reason)


def on_change_automation_status(issuer, enable, project_id=None, reason=None):
    entry_type = TYPE_ENABLE_AUTOMATION if enable else TYPE_DISABLE_AUTOMATION
    return create(issuer=issuer, type=entry_type, project=project_id, reason=reason)


def on_change_healing_status(issuer, enable, project_id=None, credit=None, credit_time=None, reason=None):
    entry_type = TYPE_ENABLE_AUTOMATED_HEALING if enable else TYPE_DISABLE_AUTOMATED_HEALING
    payload = drop_none({"credit": credit, "credit_time": credit_time}) if credit else {}
    return create(issuer=issuer, type=entry_type, project=project_id, reason=reason, payload=payload)


def on_change_dns_automation_status(issuer, enable, project_id=None, credit=None, credit_time=None, reason=None):
    entry_type = TYPE_ENABLE_DNS_AUTOMATION if enable else TYPE_DISABLE_DNS_AUTOMATION
    payload = drop_none({"credit": credit, "credit_time": credit_time}) if credit else {}
    return create(issuer=issuer, type=entry_type, project=project_id, reason=reason, payload=payload)


def on_fsm_handbrake(issuer, project_id, timeout_time, ticket_key, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_FSM_HANDBRAKE,
        project=project_id,
        reason=reason,
        payload={
            "timeout_time": timeout_time,
            "ticket_key": ticket_key,
        },
    )


def on_scenario_fsm_handbrake(issuer, scenario_id, timeout_time, ticket_key, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_SCENARIO_FSM_HANDBRAKE,
        reason=reason,
        payload={
            "timeout_time": timeout_time,
            "ticket_key": ticket_key,
        },
    )


def on_extend_fsm_handbrake(issuer, project_id, timeout_time, ticket_key, audit_log_id, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_EXTEND_FSM_HANDBRAKE,
        project=project_id,
        reason=reason,
        payload={
            "timeout_time": timeout_time,
            "ticket_key": ticket_key,
            "audit_log_id": audit_log_id,
        },
    )


def on_add_project(issuer, project_id, user_project, reason=None):
    return create(
        issuer=issuer, type=TYPE_ADD_PROJECT, project=project_id, payload={"user_project": user_project}, reason=reason
    )


def on_update_project(issuer, project_id, user_update, reason=None):
    return create(
        issuer=issuer, type=TYPE_UPDATE_PROJECT, project=project_id, payload={"user_update": user_update}, reason=reason
    )


def on_project_owners_update_request(issuer, project_id, user_update, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_PROJECT_OWNERS_UPDATE_REQUEST,
        project=project_id,
        payload={"user_update": user_update},
        reason=reason,
    )


def on_project_role_update_request(issuer, project_id, role, members_update, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_PROJECT_ROLE_UPDATE_REQUEST,
        project=project_id,
        payload={"role": role, "members_update": members_update},
        reason=reason,
    )


def on_clone_project(issuer, project_id, orig_project_id, user_project, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_CLONE_PROJECT,
        project=project_id,
        payload={"orig_project_id": orig_project_id, "user_project": user_project},
        reason=reason,
    )


def on_delete_project(issuer, project_id, reason=None):
    return create(issuer=issuer, type=TYPE_DELETE_PROJECT, project=project_id, reason=reason)


def on_project_add_owner(project_id, login):
    return create(
        issuer=authorization.ISSUER_IDM, type=TYPE_PROJECT_ADD_OWNER, project=project_id, payload={"login": login}
    )


def on_project_remove_owner(project_id, login):
    return create(
        issuer=authorization.ISSUER_IDM, type=TYPE_PROJECT_REMOVE_OWNER, project=project_id, payload={"login": login}
    )


def on_project_role_add_member(project_id, role, member):
    return create(
        issuer=authorization.ISSUER_IDM,
        type=TYPE_PROJECT_ROLE_ADD_MEMBER,
        project=project_id,
        payload={"role": role, "member": member},
    )


def on_project_role_remove_member(project_id, role, member):
    return create(
        issuer=authorization.ISSUER_IDM,
        type=TYPE_PROJECT_ROLE_REMOVE_MEMBER,
        project=project_id,
        payload={"role": role, "member": member},
    )


def on_project_request_toggle_reboot_via_ssh(issuer, project_id, is_enabled, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_PROJECT_REQUEST_TOGGLE_REBOOT_VIA_SSH,
        project=project_id,
        reason=reason,
        payload={"is_enabled": is_enabled},
    )


def on_project_toggle_reboot_via_ssh(project_id, is_enabled):
    return create(
        issuer=authorization.ISSUER_IDM,
        type=TYPE_PROJECT_TOGGLE_REBOOT_VIA_SSH,
        project=project_id,
        payload={"is_enabled": is_enabled},
    )


def on_add_preorder(issuer, project_id, preorder, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_ADD_PREORDER,
        project=project_id,
        preorder=preorder["id"],
        payload={"preorder": preorder},
        reason=reason,
    )


def on_process_preorder(issuer, project_id, preorder, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_PROCESS_PREORDER,
        project=project_id,
        preorder=preorder["id"],
        payload={"preorder": preorder},
        reason=reason,
    )


def on_restart_preorder(issuer, project_id, preorder, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_RESTART_PREORDER,
        project=project_id,
        preorder=preorder["id"],
        payload={"preorder": preorder},
        reason=reason,
    )


def on_delete_preorder(issuer, project_id, preorder, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_DELETE_PREORDER,
        project=project_id,
        preorder=preorder["id"],
        payload={"preorder": preorder},
        reason=reason,
    )


# Scenario


def on_add_scenario(issuer, scenario_id, scenario_name, scenario_type, ticket_key, script_args, status, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_ADD_SCENARIO,
        scenario_id=scenario_id,
        reason=reason,
        payload={
            "scenario_name": scenario_name,
            "scenario_type": scenario_type,
            "ticket_key": ticket_key,
            "script_args": script_args,
            "status": status,
        },
    )


def on_delete_scenario(issuer, scenario_id, reason=None):
    return create(issuer=issuer, type=TYPE_DELETE_SCENARIO, scenario_id=scenario_id, reason=reason)


def on_cancel_scenario(issuer, scenario_id, reason=None):
    return create(issuer=issuer, type=TYPE_CANCEL_SCENARIO, scenario_id=scenario_id, reason=reason)


def on_pause_scenario(issuer, scenario_id, reason=None):
    return create(issuer=issuer, type=TYPE_PAUSE_SCENARIO, scenario_id=scenario_id, reason=reason)


def on_modify_scenario(issuer, scenario_id, updated_fields, reason=None):
    return create(
        issuer=issuer, type=TYPE_MODIFY_SCENARIO, scenario_id=scenario_id, payload=updated_fields, reason=reason
    )


def on_set_maintenance_status(issuer, scenario_id, status, reason):
    payload = {"work_maintenance_status": status}
    return on_modify_scenario(issuer, scenario_id, payload, reason=reason)


def on_start_scenario(issuer, scenario_id, updated_fields, reason=None):
    return create(
        issuer=issuer, type=TYPE_STARTED_SCENARIO, scenario_id=scenario_id, payload=updated_fields, reason=reason
    )


def on_finish_scenario(issuer, scenario_id, reason=None):
    return create(issuer=issuer, type=TYPE_FINISHED_SCENARIO, scenario_id=scenario_id, reason=reason)


# Check schema loggers


def on_add_automation_plot(issuer, plot_id, request, reason=None):
    return create(
        issuer=issuer, type=TYPE_ADD_AUTOMATION_PLOT, automation_plot=plot_id, payload={"plot": request}, reason=reason
    )


def on_update_automation_plot(issuer, plot_id, request, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_UPDATE_AUTOMATION_PLOT,
        automation_plot=plot_id,
        payload={"plot update": request},
        reason=reason,
    )


def on_delete_automation_plot(issuer, plot_id, reason=None):
    return create(issuer=issuer, type=TYPE_DELETE_AUTOMATION_PLOT, automation_plot=plot_id, reason=reason)


def on_add_maintenance_plot(issuer, plot_id, request, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_ADD_MAINTENANCE_PLOT,
        maintenance_plot=plot_id,
        payload={"plot": request},
        reason=reason,
    )


def on_update_maintenance_plot(issuer, plot_id, request, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_UPDATE_MAINTENANCE_PLOT,
        maintenance_plot=plot_id,
        payload={"plot update": request},
        reason=reason,
    )


def on_delete_maintenance_plot(issuer, plot_id, reason=None):
    return create(issuer=issuer, type=TYPE_DELETE_MAINTENANCE_PLOT, maintenance_plot=plot_id, reason=reason)


def on_add_host(
    issuer,
    project_id,
    inv,
    name,
    host_uuid,
    preorder_id,
    user_host,
    added_host,
    instant=None,
    dns=None,
    ignore_cms=None,
    disable_admin_requests=None,
    with_auto_healing=None,
    error=None,
    reason=None,
    scenario_id=None,
    status=None,
    state=None,
):

    log_entry = create(
        issuer=issuer,
        type=TYPE_ADD_HOST,
        project=project_id,
        preorder=preorder_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        scenario_id=scenario_id,
        payload=drop_none(
            {
                "user_host": user_host,
                "added_host": added_host,
                "status": status,
                "state": state,
                "dns": dns,
                "ignore_cms": ignore_cms,
                "with_auto_healing": with_auto_healing,
                "disable_admin_requests": disable_admin_requests,
                "error": error,
            }
        ),
    )
    log_entry.instant = instant
    return log_entry


def on_update_host(issuer, project_id, inv, name, host_uuid, user_update, reason=None, scenario_id=None):
    return create(
        issuer=issuer,
        type=TYPE_UPDATE_HOST,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        payload={"user_update": user_update},
        reason=reason,
        scenario_id=scenario_id,
    )


def on_switch_project_for_host(issuer, old_project_id, new_project_id, inv, name, host_uuid, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_SWITCH_PROJECT,
        project=old_project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        payload={"old_project_id": old_project_id, "new_project_id": new_project_id},
        reason=reason,
    )


def on_set_host_maintenance(
    issuer,
    project_id,
    inv,
    name,
    host_uuid,
    timeout_time=None,
    timeout_status=None,
    reason=None,
    scenario_id=None,
    **payload
):

    return create(
        issuer=issuer,
        type=TYPE_SET_HOST_MAINTENANCE,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        scenario_id=scenario_id,
        payload=drop_none(dict(payload, timeout_time=timeout_time, timeout_status=timeout_status)),
    )


def on_change_host_maintenance(
    issuer,
    project_id,
    inv,
    name,
    host_uuid,
    timeout_time=None,
    timeout_status=None,
    operation_state=None,
    reason=None,
    scenario_id=None,
    **payload
):

    return create(
        issuer=issuer,
        type=TYPE_CHANGE_HOST_MAINTENANCE,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        scenario_id=scenario_id,
        payload=drop_none(
            dict(payload, timeout_time=timeout_time, timeout_status=timeout_status, operation_state=operation_state)
        ),
    )


def on_set_host_state(issuer, project_id, inv, name, host_uuid, state, reason=None, scenario_id=None, **payload):
    return create(
        issuer=issuer,
        type=TYPE_SET_HOST_STATE,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        scenario_id=scenario_id,
        payload=drop_none(dict(payload, state=state)),
    )


def on_force_host_status(
    issuer,
    project_id,
    inv,
    name,
    host_uuid,
    status,
    timeout_time=None,
    timeout_status=None,
    reason=None,
    scenario_id=None,
):
    return create(
        issuer=issuer,
        type=TYPE_FORCE_HOST_STATUS,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        payload=drop_none(dict(status=status, timeout_time=timeout_time, timeout_status=timeout_status)),
        reason=reason,
        scenario_id=scenario_id,
    )


def on_invalidate_host(issuer, project_id, inv, name, host_uuid, reason=None, scenario_id=None, **payload):
    return create(
        issuer=issuer,
        type=TYPE_INVALIDATE_HOST,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        payload=payload,
        reason=reason,
        scenario_id=scenario_id,
    )


def on_task_cancelled(
    issuer, project_id, inv, name, host_uuid, status, task_audit_log_id, reason=None, scenario_id=None
):
    return create(
        issuer=issuer,
        type=TYPE_CANCEL_HOST_TASK,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        scenario_id=scenario_id,
        payload=drop_none(dict(status=status, task_audit_log_id=task_audit_log_id)),
    )


def on_delete_host(
    issuer,
    project_id,
    inv,
    name,
    host_uuid,
    lui=None,
    ignore_cms=None,
    disable_admin_requests=None,
    ignore_maintenance=None,
    reason=None,
    instant=None,
    scenario_id=None,
):
    log_entry = create(
        issuer=issuer,
        type=TYPE_DELETE_HOST,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        scenario_id=scenario_id,
        payload=drop_none(
            {
                "lui": lui,
                "ignore_cms": ignore_cms,
                "ignore_maintenance": ignore_maintenance,
                "disable_admin_requests": disable_admin_requests,
            }
        ),
    )
    log_entry.instant = instant
    return log_entry


def on_check_dns_records(
    issuer,
    project_id,
    inv,
    name,
    host_uuid,
    ignore_maintenance=None,
    disable_admin_requests=None,
    check=None,
    with_auto_healing=None,
    reason=None,
    scenario_id=None,
):
    return create(
        issuer=issuer,
        type=TYPE_CHECK_DNS_RECORDS,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        scenario_id=scenario_id,
        payload=drop_none(
            {
                "disable_admin_requests": disable_admin_requests,
                "check": check,
                "ignore_maintenance": ignore_maintenance,
                "with_auto_healing": with_auto_healing,
            }
        ),
    )


def on_fix_dns_records(
    project_id,
    inv,
    name,
    host_uuid,
    switch,
    switch_source,
    mac,
    mac_source,
    records,
    operations,
    ips=None,
    reason=None,
    scenario_id=None,
):
    return create(
        issuer=authorization.ISSUER_WALLE,
        type=TYPE_FIX_DNS_RECORDS,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        scenario_id=scenario_id,
        payload={
            "switch": switch,
            "switch_source": switch_source,
            "mac": mac,
            "mac_source": mac_source,
            "ips": ips,
            "records": records,
            "operations": operations,
        },
    )


def on_clear_dns_records(project_id, inv, name, host_uuid, operations, reason=None, scenario_id=None):
    return create(
        issuer=authorization.ISSUER_WALLE,
        type=TYPE_CLEAR_DNS_RECORDS,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        scenario_id=scenario_id,
        payload={"operations": operations},
    )


def on_profile_host(
    issuer,
    project_id,
    inv,
    name,
    host_uuid,
    profile,
    profile_tags,
    redeploy,
    deploy_configuration,
    ignore_cms,
    disable_admin_requests,
    check,
    with_auto_healing=None,
    profile_modes=None,
    reason=None,
    scenario_id=None,
):

    deploy_configuration_kwargs = {}
    if deploy_configuration is not None:
        deploy_configuration_kwargs = {
            "provisioner": deploy_configuration.provisioner,
            "deploy_config": deploy_configuration.config,
            "deploy_tags": deploy_configuration.tags,
            "deploy_network": deploy_configuration.network,
            "deploy_certificate": deploy_configuration.certificate,
            "deploy_config_policy": deploy_configuration.deploy_config_policy,
        }

    return create(
        issuer=issuer,
        type=TYPE_PROFILE_HOST,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        scenario_id=scenario_id,
        payload=drop_none(
            dict(
                deploy_configuration_kwargs,
                profile=profile,
                profile_tags=profile_tags,
                profile_modes=profile_modes,
                redeploy=redeploy,
                disable_admin_requests=disable_admin_requests,
                ignore_cms=ignore_cms,
                with_auto_healing=with_auto_healing,
                check=check,
            )
        ),
    )


def on_hardware_repair(issuer, project_id, inv, name, host_uuid, reason=None, scenario_id=None, **payload):
    return create(
        issuer=issuer,
        type=TYPE_HARDWARE_REPAIR,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        scenario_id=scenario_id,
        payload=payload,
        reason=reason,
    )


def on_repair_link(issuer, project_id, inv, name, host_uuid, reason, scenario_id=None, **payload):
    return create(
        issuer=issuer,
        type=TYPE_REPAIR_LINK,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        scenario_id=scenario_id,
        payload=payload,
        reason=reason,
    )


def on_bmc_reset(issuer, project_id, inv, name, host_uuid, reason, scenario_id=None, **payload):
    return create(
        issuer=issuer,
        type=TYPE_RESET_BMC,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        scenario_id=scenario_id,
        payload=payload,
        reason=reason,
    )


def on_repair_bmc(issuer, project_id, inv, name, host_uuid, payload, reason):
    return create(
        issuer=issuer,
        type=TYPE_REPAIR_BMC,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        payload=payload,
        reason=reason,
    )


def on_power_off_host(issuer, project_id, inv, name, host_uuid, payload, reason):
    return create(
        issuer=issuer,
        type=TYPE_POWER_OFF_HOST,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        payload=payload,
        reason=reason,
    )


def on_repair_cpu_overheat(issuer, project_id, inv, name, host_uuid, payload, reason):
    return create(
        issuer=issuer,
        type=TYPE_REPAIR_OVERHEAT,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        payload=payload,
        reason=reason,
    )


def on_repair_cpu_capping(issuer, project_id, inv, name, host_uuid, payload, reason):
    return create(
        issuer=issuer,
        type=TYPE_REPAIR_CAPPING,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        payload=payload,
        reason=reason,
    )


def on_change_memory(issuer, project_id, inv, name, host_uuid, payload, reason):
    return create(
        issuer=issuer,
        type=TYPE_CHANGE_MEMORY,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        payload=payload,
        reason=reason,
    )


def on_change_disk(issuer, project_id, inv, name, host_uuid, payload, reason):
    return create(
        issuer=issuer,
        type=TYPE_CHANGE_DISK,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        payload=payload,
        reason=reason,
    )


def on_redeploy_host(
    issuer,
    project_id,
    inv,
    name,
    host_uuid,
    deploy_configuration,
    ignore_cms,
    disable_admin_requests,
    check,
    with_auto_healing,
    set_provisioner=None,
    set_config=None,
    set_tags=None,
    set_network=None,
    set_deploy_config_policy=None,
    profile=None,
    profile_tags=None,
    profile_modes=None,
    reason=None,
    scenario_id=None,
):

    deploy_configuration_kwargs = {}
    if deploy_configuration is not None:
        deploy_configuration_kwargs = {
            "provisioner": deploy_configuration.provisioner,
            "deploy_config": deploy_configuration.config,
            "deploy_tags": deploy_configuration.tags,
            "deploy_network": deploy_configuration.network,
            "ipxe_supported": deploy_configuration.ipxe,
            "deploy_certificate": deploy_configuration.certificate,
            "deploy_config_policy": deploy_configuration.deploy_config_policy,
        }

    return create(
        issuer=issuer,
        type=TYPE_REDEPLOY_HOST,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        scenario_id=scenario_id,
        payload=drop_none(
            dict(
                deploy_configuration_kwargs,
                profile=profile,
                profile_tags=profile_tags,
                profile_modes=profile_modes,
                set_provisioner=set_provisioner,
                set_config=set_config,
                set_tags=set_tags,
                set_network=set_network,
                set_deploy_config_policy=set_deploy_config_policy,
                ignore_cms=ignore_cms,
                disable_admin_requests=disable_admin_requests,
                check=check,
                with_auto_healing=with_auto_healing,
            )
        ),
    )


def on_repair_rack(issuer, project_id, inv, name, host_uuid, location_path, reason=None, scenario_id=None):
    return create(
        issuer=issuer,
        type=TYPE_REPAIR_RACK_FAILURE,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        scenario_id=scenario_id,
        payload={"location": location_path},
        reason=reason,
    )


def on_deactivate_host(issuer, project_id, inv, name, host_uuid, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_DEACTIVATE_HOST,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
    )


def on_admin_request(project_id, inv, name, host_uuid, request_type_name, reason=None, payload=None, scenario_id=None):
    return create(
        issuer=authorization.ISSUER_WALLE,
        type=_type_admin_request(request_type_name),
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        payload=payload,
        scenario_id=scenario_id,
    )


def on_report_host_failure(issuer, project_id, inv, name, host_uuid, checks=None, reason=None, scenario_id=None):
    return create(
        issuer=issuer,
        type=TYPE_REPORT_HOST_FAILURE,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        scenario_id=scenario_id,
        payload=drop_none({"checks": checks}),
    )


def on_report(issuer, project_id, inv, name, host_uuid, ticket=None, reason=None):
    return create(
        issuer=issuer,
        type=TYPE_REPORT,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        payload=drop_none({"ticket": ticket}),
    )


def on_bot_project_sync(issuer, project_id, inv, name, host_uuid, reason=None, **payload):
    return create(
        issuer=issuer,
        type=TYPE_BOT_PROJECT_SYNC,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        payload=payload,
    )


def on_host_inv_changed(issuer, project_id, old_inv, new_inv, name, host_uuid, reason):
    return create(
        issuer=issuer,
        type=TYPE_HOST_INV_CHANGED,
        project=project_id,
        host_inv=old_inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=reason,
        payload={"old_inv": old_inv, "new_inv": new_inv},
    )


def on_host_fqdn_changed(issuer, project_id, old_fqdn, new_fqdn, inv, host_uuid, reason):
    return create(
        issuer=issuer,
        type=TYPE_HOST_FQDN_CHANGED,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=old_fqdn,
        reason=reason,
        payload={"old_fqdn": old_fqdn, "new_fqdn": new_fqdn},
    )


# some other events


def on_failure_processing_cancelled(project_id, inv, name, host_uuid, failure_reason, cancel_reason):
    return create(
        issuer=authorization.ISSUER_WALLE,
        type=TYPE_NO_ACTION,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        reason=failure_reason,
        error=cancel_reason,
    )


def on_active_mac_changed(
    project_id,
    inv,
    name,
    host_uuid,
    mac,
    actualization_time,
    source,
    prev_mac=None,
    prev_actualization_time=None,
    prev_source=None,
):
    payload = {"new": {"mac": mac, "actualization_time": actualization_time, "source": source}}

    # Notice: `prev_mac` may be None which means that we've learned host's active MACs, but it had a few active MACs.
    if prev_actualization_time is not None and prev_source is not None:
        payload["prev"] = {"mac": prev_mac, "actualization_time": prev_actualization_time, "source": prev_source}

    return create(
        issuer=authorization.ISSUER_WALLE,
        type=TYPE_ACTIVE_MAC_CHANGE_DETECTED,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        payload=payload,
    )


def on_host_macs_changed(
    project_id,
    inv,
    name,
    host_uuid,
    new_macs,
    new_active_mac,
    new_active_mac_time,
    new_active_mac_source,
    old_macs,
    old_active_mac,
    old_active_mac_time,
    old_active_mac_source,
):

    payload = {
        "new": {
            "macs": new_macs,
            "active_mac": new_active_mac,
            "active_mac_time": new_active_mac_time,
            "active_mac_source": new_active_mac_source,
        },
        "old": {
            "macs": old_macs,
            "active_mac": old_active_mac,
            "active_mac_time": old_active_mac_time,
            "active_mac_source": old_active_mac_source,
        },
    }

    return create(
        issuer=authorization.ISSUER_WALLE,
        type=TYPE_HOST_MACS_CHANGED,
        project=project_id,
        host_inv=inv,
        host_name=name,
        host_uuid=host_uuid,
        payload=payload,
    )


def on_switch_port_changed(
    project_id,
    inv,
    name,
    host_uuid,
    switch,
    port,
    actualization_time,
    source,
    prev_switch=None,
    prev_port=None,
    prev_actualization_time=None,
    prev_source=None,
):
    payload = {"new": {"switch": switch, "port": port, "actualization_time": actualization_time, "source": source}}

    if (
        prev_switch is not None
        and prev_port is not None
        and prev_actualization_time is not None
        and prev_source is not None
    ):
        payload["prev"] = {
            "switch": prev_switch,
            "port": prev_port,
            "actualization_time": prev_actualization_time,
            "source": prev_source,
        }

    return create(
        issuer=authorization.ISSUER_WALLE,
        type=TYPE_SWITCH_PORT_CHANGE_DETECTED,
        project=project_id,
        host_inv=inv,
        host_uuid=host_uuid,
        host_name=name,
        payload=payload,
    )


def on_ips_changed(host, new_ips):
    payload = {"old": host.ips, "new": new_ips}

    return create(
        issuer=authorization.ISSUER_WALLE,
        type=TYPE_IPS_CHANGE_DETECTED,
        project=host.project,
        host_inv=host.inv,
        host_uuid=host.uuid,
        host_name=host.name,
        payload=payload,
    )


def complete_request(entry_or_id, extra_payload=None, check_updated=True):
    _change_status(
        entry_or_id,
        (STATUS_UNKNOWN, STATUS_ACCEPTED),
        STATUS_COMPLETED,
        extra_payload=extra_payload,
        suppress_errors=True,
        check_updated=check_updated,
    )


def complete_task(task):
    complete_request(task.audit_log_id)


def cancel_request(entry_or_id, error=None, check_updated=True):
    _change_status(
        entry_or_id,
        (STATUS_UNKNOWN, STATUS_ACCEPTED),
        STATUS_CANCELLED,
        check_updated=check_updated,
        error=error,
        suppress_errors=True,
    )


def cancel_task(task, error=None):
    cancel_request(task.audit_log_id, error)


def fail_request(entry_or_id, error):
    _change_status(entry_or_id, (STATUS_UNKNOWN, STATUS_ACCEPTED), STATUS_FAILED, error=error, suppress_errors=True)


def fail_task(task, error):
    fail_request(task.audit_log_id, error)


def create(**kwargs):
    entry = LogEntry(id=_uuid(), time=_time(), status=STATUS_UNKNOWN, status_time=_time(), **kwargs)
    entry.save(force_insert=True)
    return entry


def _get_update_fields(extra_payload):
    update = {}
    for key, value in (extra_payload or {}).items():
        if "." in key:
            raise Error("Invalid payload key: {}.", key)

        update[LogEntry.payload.db_field + "." + key] = value
    return update


def update_payload(entry_or_id, extra_payload):
    if isinstance(entry_or_id, str):
        entry = LogEntry(id=entry_or_id)
    else:
        entry = entry_or_id
    entry.modify(__raw__={"$set": _get_update_fields(extra_payload)})


def _change_status(
    entry_or_id, from_statuses, status, error=None, extra_payload=None, check_updated=True, suppress_errors=False
):
    if isinstance(entry_or_id, str):
        entry = LogEntry(id=entry_or_id)
    else:
        entry = entry_or_id

    try:
        update = drop_none(
            {
                LogEntry.status.db_field: status,
                LogEntry.status_time.db_field: _time(),
                LogEntry.error.db_field: error,
            }
        )

        for key, value in (extra_payload or {}).items():
            if "." in key:
                raise Error("Invalid payload key: {}.", key)

            update[LogEntry.payload.db_field + "." + key] = value

        updated = entry.modify(dict(status__in=from_statuses), __raw__={"$set": update})
    except Exception as e:
        if not suppress_errors:
            raise
        log.critical("Failed to set '%s' status for audit log entry '%s': %s", status, entry.id, e)
    else:
        if updated:
            try:
                notifications.on_event(entry)
            except Exception:
                log.critical("Failed to send notification for %s entry:", entry.id, exc_info=True)
        elif check_updated:
            log.critical(
                "Failed to set '%s' status for audit log entry '%s': it doesn't exist or changed its state.",
                status,
                entry.id,
            )


def _uuid():
    return uuid.uuid4().hex


def _time():
    return time.time()
