"""User authorization logic."""

import logging
from collections import namedtuple

import cachetools
import six

import object_validator
from blackbox import Blackbox, BlackboxError, FIELD_LOGIN
from sepelib.core import config, constants
from sepelib.core.exceptions import Error, LogicalError
from walle.clients.staff import is_member, is_group
from walle.errors import UnauthenticatedError, UnauthorizedError
from walle.util.validation import ApiDictScheme
from . import user_info

log = logging.getLogger(__name__)

AuthInfo = namedtuple("AuthInfo", ("issuer", "session_id"))

HOT_CACHE_MAX_SIZE = 50
HOT_CACHE_TTL = constants.NETWORK_TIMEOUT
_AUTHENTICATION_HOT_CACHE = cachetools.TTLCache(maxsize=HOT_CACHE_MAX_SIZE, ttl=HOT_CACHE_TTL)
_SESSION_ID_HOT_CACHE = cachetools.TTLCache(maxsize=HOT_CACHE_MAX_SIZE, ttl=HOT_CACHE_TTL)

FALLBACK_CACHE_MAX_SIZE = 500
FALLBACK_CACHE_TTL = constants.DAY_SECONDS
_AUTHENTICATION_FALLBACK_CACHE = cachetools.TTLCache(maxsize=FALLBACK_CACHE_MAX_SIZE, ttl=FALLBACK_CACHE_TTL)
_SESSION_ID_FALLBACK_CACHE = cachetools.TTLCache(maxsize=FALLBACK_CACHE_MAX_SIZE, ttl=FALLBACK_CACHE_TTL)

BLACKBOX_RETRIES = 3


class _BlackboxString(object_validator.String):
    def validate(self, obj):
        return super().validate(six.ensure_text(obj, "utf-8"))


# blackbox methods return odict instead of dict objects
class _BlackboxApiDictScheme(ApiDictScheme):
    def validate(self, obj):
        if type(obj) is not dict and isinstance(obj, dict):
            obj = dict(obj)

        return super().validate(obj)


class BlackBoxCommunicationError(Error):
    def __init__(self, message, *args):
        message = "Error in communication with blackbox: {}".format(message)
        super().__init__(message, *args)


def authenticate(oauth_client_id):
    return _authenticate_cached(oauth_client_id)


def _authenticate_cached(oauth_client_id):
    """Authenticate the request.
    Use a hot cache with short TTL to reduce amount of requests to BlackBox.
    Use fallback-cache to provide some reliability in severe conditions.
    """
    # Import these dependencies only when they are really required
    from flask import request

    if "Authorization" in request.headers:
        hot_cache = _AUTHENTICATION_HOT_CACHE
        fallback_cache = _AUTHENTICATION_FALLBACK_CACHE
        cache_key = request.headers["Authorization"]
    elif "Session_id" in request.cookies:
        hot_cache = _SESSION_ID_HOT_CACHE
        fallback_cache = _SESSION_ID_FALLBACK_CACHE
        cache_key = request.cookies["Session_id"]
    else:
        raise UnauthenticatedError("You must be authenticated to perform this request.")

    try:
        auth_info = hot_cache[cache_key]
    except KeyError:
        try:
            auth_info = _authenticate(oauth_client_id, request), True
            hot_cache[cache_key] = fallback_cache[cache_key] = auth_info
        except UnauthenticatedError as unauthenticated:
            auth_info = str(unauthenticated), False
            hot_cache[cache_key] = fallback_cache[cache_key] = auth_info
        except BlackBoxCommunicationError as blackbox_error:
            try:
                # Promote fall-back value to the hot cache to prevent wall-e from
                # making slow requests to blackbox which is not responding at the time.
                # Hot cache have a very short TTL.
                auth_info = hot_cache[cache_key] = fallback_cache[cache_key]
                log.critical("BlackBox communication error", exc_info=True)
            except KeyError:
                raise blackbox_error

    info, authenticated = auth_info
    if not authenticated:
        raise UnauthenticatedError(info)

    return info


def _authenticate(oauth_client_id, request):
    """Authenticate the request by requesting credentials from BlackBox."""
    # Import these dependencies only when they are really required
    session_id = None
    client = Blackbox(
        url=config.get_value("blackbox.url", Blackbox.URLS['intranet']['production']),
        retry_count=BLACKBOX_RETRIES,  # BlackBox makes one retry by default, we need more.
    )

    try:
        if "Authorization" in request.headers:
            authorization_header = _validate_authorization_header(request.headers["Authorization"])
            result = client.oauth({"Authorization": authorization_header}, request.remote_addr, dbfields=[FIELD_LOGIN])
        elif "Session_id" in request.cookies:
            session_id = request.cookies["Session_id"]
            result = client.sessionid(
                session_id, request.remote_addr, request.host.split(":")[0], dbfields=[FIELD_LOGIN]
            )
        else:
            # the caller must prevent this from happening
            raise LogicalError()
    except BlackboxError as e:
        raise BlackBoxCommunicationError(f"{str(e)}.")

    try:
        result = object_validator.validate("result", result, _BlackboxApiDictScheme({"status": _BlackboxString()}))
        if result["status"] != "VALID":
            raise UnauthenticatedError(
                "{} is expired or invalid.", ("API access token" if session_id is None else "The session")
            )

        scheme = {"fields": _BlackboxApiDictScheme({"login": _BlackboxString(min_length=1)})}

        if session_id is None:
            scheme["oauth"] = _BlackboxApiDictScheme(
                {"client_id": _BlackboxString(), "client_name": _BlackboxString(), "scope": _BlackboxString()}
            )

        result = object_validator.validate("result", result, _BlackboxApiDictScheme(scheme))
    except object_validator.ValidationError as e:
        raise BlackBoxCommunicationError("The server returned an invalid JSON response: {}", e)

    if session_id is None:
        if not oauth_client_id:
            raise LogicalError()

        provided_client_id = result["oauth"]["client_id"]
        has_another_client_id = provided_client_id != oauth_client_id
        has_proper_scope = user_info.ISSUER_SCOPE in (result["oauth"]["scope"] or "").split(" ")
        if has_another_client_id and not has_proper_scope:
            raise UnauthenticatedError("API access token was obtained for other application - not for Wall-E.")

    return AuthInfo(issuer=result["fields"]["login"] + "@", session_id=session_id)


def _validate_authorization_header(authorization_header):
    try:
        auth_type, token = authorization_header.split()
        if not token:
            raise UnauthenticatedError("Authorization token is empty")
    except ValueError:
        raise UnauthenticatedError("Authorization token is empty")

    return authorization_header


def authorize(
    issuer,
    error_message,
    owners=tuple(),
    authorize_admins=False,
    authorize_admins_with_sudo=False,
    authorize_by_group=True,
):
    """Authorizes the request."""

    if not config.get_value("authorization.enabled"):
        return

    login = user_info.get_issuer_login(issuer)

    if authorize_admins_with_sudo:
        if user_info.is_admin(login):
            return

        raise UnauthorizedError("You must have admin privileges in Wall-E to use sudo.")

    if login in owners or authorize_admins and user_info.is_admin(login):
        return

    if authorize_by_group and is_member(login, list(filter(is_group, owners))):
        return

    raise UnauthorizedError(error_message)
