import abc
import functools
import operator
import uuid

import flask
import mongoengine

from sepelib.core import config
from walle.audit_log import LogEntry
from walle.clients import iam as iam_client
from walle.errors import BadRequestError, UnauthorizedError
from walle.expert.automation_plot import AutomationPlot
from walle.hosts import Host
from walle.projects import Project
from walle.util.misc import InvOrUUIDOrName
from walle.util.patterns import parse_pattern

BEARER_HEADER_PREFIX = "Bearer "


class IamHandlerDisabled(BadRequestError):
    def __init__(self):
        super().__init__("Handler was disabled for IAM based WALLE installation")


class UnknownIamTargetFolder(BadRequestError):
    """Can't find request argument for IAM auth"""

    def __init__(self, details):
        super().__init__(f"Unknown iam target folder. {details}")


class BaseApiIamPermission:
    """Base class for typing"""


class NoOneApiIamPermission(BaseApiIamPermission):
    """Handler disabled for Cloud IAM based contour"""


class AnyoneApiIamPermission(BaseApiIamPermission):
    """Handler without authentication"""

    # NOTE(rocco66): waiting for IAM in juggler and walle agent, https://st.yandex-team.ru/WALLE-4385


class ApiIamPermission(BaseApiIamPermission):
    """Base class for typing"""

    @abc.abstractmethod
    def get_actions(self, query_args: dict, request_obj: dict) -> list[iam_client.IamAction]:
        """Get folder name from request"""


def _get_request_arg(query_args, request_obj, arg_name):
    return flask.request.view_args.get(arg_name) or query_args.get(arg_name, request_obj.get(arg_name))


def _get_many_request_args(*args, **kwargs) -> list[str]:
    if value := _get_request_arg(*args, **kwargs):
        if isinstance(value, list):
            return value
        return [value]
    else:
        return []


class SingleActionApiIamPermission(ApiIamPermission, abc.ABC):
    @property
    @abc.abstractmethod
    def iam_permission(self):
        pass


class GizmoApiIamPermission(SingleActionApiIamPermission, abc.ABC):
    def get_actions(self, query_args, request_obj):
        return [iam_client.IamAction(self.iam_permission, iam_client.GIZMO_RESOURCE)]


class FolderFromArgApiIamPermission(SingleActionApiIamPermission, abc.ABC):
    def __init__(self, arg_name):
        self._arg_name = arg_name

    @abc.abstractmethod
    def get_folder_ids(self, query_args, request_obj) -> list[str]:
        pass

    def get_actions(self, query_args, request_obj):
        if folders := self.get_folder_ids(query_args, request_obj):
            return [iam_client.IamAction(self.iam_permission, f) for f in folders]
        else:
            raise UnknownIamTargetFolder(f"Can't find any folders for '{self._arg_name}' argument")


class BindProjectFolderApiIamPermission(FolderFromArgApiIamPermission):
    iam_permission = iam_client.ProjectsPermissions.update

    def get_folder_ids(self, query_args, request_obj):
        if folder := _get_request_arg(query_args, request_obj, self._arg_name):
            return [folder]
        else:
            raise UnknownIamTargetFolder(f"Request should have folder '{self._arg_name}' argument")


class BindAutomationPlotFolderApiIamPermission(FolderFromArgApiIamPermission):
    iam_permission = iam_client.AutomationPlotsPermissions.update

    def get_folder_ids(self, query_args, request_obj):
        if folder := _get_request_arg(query_args, request_obj, self._arg_name):
            return [folder]
        else:
            raise UnknownIamTargetFolder(f"Request should have folder '{self._arg_name}' argument")


class ProjectApiIamPermission(FolderFromArgApiIamPermission, abc.ABC):
    def get_folder_ids(self, query_args, request_obj):
        result = []
        if project_ids := _get_many_request_args(query_args, request_obj, self._arg_name):
            for project_id in project_ids:
                if project := Project.objects.only("yc_iam_folder_id").get(id=project_id):
                    if project.yc_iam_folder_id:
                        result.append(project.yc_iam_folder_id)
                    else:
                        raise UnknownIamTargetFolder(f"Project '{project_id}' does not have binded YC folder")
                else:
                    raise UnknownIamTargetFolder(f"Can't find '{project_id}' project")
        else:
            raise UnknownIamTargetFolder(f"Request should have '{self._arg_name}' argument")
        return result


class AutomationPlotApiIamPermission(FolderFromArgApiIamPermission, abc.ABC):
    def get_folder_ids(self, query_args, request_obj):
        result = []
        if plot_ids := _get_many_request_args(query_args, request_obj, self._arg_name):
            for plot_id in plot_ids:
                if automation_plot := AutomationPlot.objects.only("yc_iam_folder_id").get(id=plot_id):
                    if automation_plot.yc_iam_folder_id:
                        result.append(automation_plot.yc_iam_folder_id)
                    else:
                        raise UnknownIamTargetFolder(
                            f"AutomationPlot '{automation_plot}' does not have binded YC folder"
                        )
                else:
                    raise UnknownIamTargetFolder(f"Can't find automation plot '{plot_id}'")
        else:
            raise UnknownIamTargetFolder(f"Request should have '{self._arg_name}' argument")
        return result


class CreateProjectApiIamPermission(GizmoApiIamPermission):
    iam_permission = iam_client.ProjectsPermissions.create


class AgentApiIamPermission(GizmoApiIamPermission):
    iam_permission = iam_client.ApiPermissions.juggler


class UpdateProjectApiIamPermission(ProjectApiIamPermission):
    iam_permission = iam_client.ProjectsPermissions.update


class GetProjectApiIamPermission(ProjectApiIamPermission):
    iam_permission = iam_client.ProjectsPermissions.get


class DeleteProjectApiIamPermission(ProjectApiIamPermission):
    iam_permission = iam_client.ProjectsPermissions.delete


class UpdateAutomationPlotApiIamPermission(AutomationPlotApiIamPermission):
    iam_permission = iam_client.AutomationPlotsPermissions.update


class GetAutomationPlotApiIamPermission(AutomationPlotApiIamPermission):
    iam_permission = iam_client.AutomationPlotsPermissions.get


class OptionalGetAutomationPlotApiIamPermission(GetAutomationPlotApiIamPermission):
    def get_actions(self, query_args, request_obj):
        try:
            folders = self.get_folder_ids(query_args, request_obj)
        except UnknownIamTargetFolder:
            return []
        else:
            if folders:
                return [iam_client.IamAction(self.iam_permission, f) for f in folders]
            else:
                return []


class DeleteAutomationPlotApiIamPermission(AutomationPlotApiIamPermission):
    iam_permission = iam_client.AutomationPlotsPermissions.delete


class HostApiIamPermission(FolderFromArgApiIamPermission, abc.ABC):
    def get_folder_ids(self, query_args, request_obj):
        # TODO(rocco66): iam_client.HostsPermissions after service X

        result = []
        if host_ids := _get_many_request_args(query_args, request_obj, self._arg_name):
            for host_id in host_ids:
                if host := Host.get_by_host_id_query(InvOrUUIDOrName(host_id)):
                    if project := Project.objects.only("yc_iam_folder_id").get(id=host.project):
                        if project.yc_iam_folder_id:
                            result.append(project.yc_iam_folder_id)
                        else:
                            raise UnknownIamTargetFolder(f"Project '{host.project}' does not have binded YC folder")
                    else:
                        raise UnknownIamTargetFolder(f"Can't find project '{host.project_id}'")
                else:
                    raise UnknownIamTargetFolder(f"Can't find project for '{host_id}'")
        else:
            raise UnknownIamTargetFolder(f"Request should have '{self._arg_name}' argument")
        return result


class UpdateHostApiIamPermission(HostApiIamPermission):
    iam_permission = iam_client.ProjectsPermissions.update


class DeleteHostApiIamPermission(HostApiIamPermission):
    iam_permission = iam_client.ProjectsPermissions.update


class GetHostApiIamPermission(HostApiIamPermission):
    iam_permission = iam_client.ProjectsPermissions.get


class GetHostsApiIamPermission(ApiIamPermission):
    def __init__(
        self, projects_arg_name=None, hosts_arg_name=None, hosts_arg_inv=None, hosts_arg_uuid=None, host_patterns=None
    ):
        assert hosts_arg_name or hosts_arg_inv or hosts_arg_uuid or projects_arg_name or host_patterns
        self._projects_arg_name = projects_arg_name
        self._hosts_arg_name = hosts_arg_name
        self._hosts_arg_inv = hosts_arg_inv
        self._hosts_arg_uuid = hosts_arg_uuid
        self._host_patterns = host_patterns

    def get_actions(self, query_args, request_obj) -> list[iam_client.IamAction]:
        project_ids = set()
        if self._projects_arg_name:
            project_ids |= set(_get_many_request_args(query_args, request_obj, self._projects_arg_name))

        qs = []
        if self._hosts_arg_name:
            if host_names := _get_many_request_args(query_args, request_obj, self._hosts_arg_name):
                qs.append(mongoengine.Q(name__in=host_names))
        if self._hosts_arg_inv:
            if host_invs := _get_many_request_args(query_args, request_obj, self._hosts_arg_inv):
                qs.append(mongoengine.Q(inv__in=[int(i) for i in host_invs]))
        if self._hosts_arg_uuid:
            if host_uuids := _get_many_request_args(query_args, request_obj, self._hosts_arg_uuid):
                qs.append(mongoengine.Q(uuid__in=host_uuids))
        if self._host_patterns:
            if host_pattrerns := _get_many_request_args(query_args, request_obj, self._host_patterns):
                qs.append(mongoengine.Q(name__in=[parse_pattern(p) for p in host_pattrerns]))
        if qs:
            project_ids |= set(Host.objects(functools.reduce(operator.or_, qs)).distinct("project"))

        if project_ids:
            projects_query = Project.objects(id__in=list(project_ids))
        else:
            projects_query = Project.objects()
        return [
            iam_client.IamAction(iam_client.ProjectsPermissions.get, f)
            for f in projects_query.distinct("yc_iam_folder_id")
        ]


class LogEntryApiIamPermission(ApiIamPermission):
    def __init__(self, entry_id_arg_name):
        self._entry_id_arg_name = entry_id_arg_name

    def get_actions(self, query_args: dict, request_obj: dict) -> list[iam_client.IamAction]:
        if entry_ids := _get_many_request_args(query_args, request_obj, self._entry_id_arg_name):
            log_entry = LogEntry.objects.only("project", "host_name").get(id__in=entry_ids)
            project = Project.objects.get(id=log_entry.project)
            if not project:
                raise UnknownIamTargetFolder(f"Can't find any folders for '{project.id}' project")
            return [iam_client.IamAction(iam_client.ProjectsPermissions.get, project.yc_iam_folder_id)]
        else:
            return []


def check(permissions: list[BaseApiIamPermission], query_args: dict, request_obj: dict):
    actions = []
    for permission in permissions:
        if isinstance(permission, AnyoneApiIamPermission):
            return "someone@"
        elif isinstance(permission, NoOneApiIamPermission):
            raise IamHandlerDisabled()
        elif isinstance(permission, ApiIamPermission):
            actions += permission.get_actions(query_args, request_obj)
        else:
            raise RuntimeError(f"unknown IAM permission type '{permission}'")
    iam_as_client = iam_client.AsClient(config.get_value("iam.access_service_endpoint"))
    iam_token = _get_iam_token()
    request_id = _get_request_id()
    try:
        if not actions:
            subject = iam_as_client.authenticate(iam_token, request_id=request_id)
        else:
            try:
                subject = iam_as_client.authorize(iam_token, actions, request_id=request_id)
            except iam_client.TooComplexAuthError as exc:
                raise BadRequestError(str(exc))
    except iam_client.IamAuthError as exc:
        raise UnauthorizedError(str(exc))
    else:
        return iam_client.get_name_by_token(iam_token, subject)


def _get_iam_token():
    if auth_header := flask.request.headers.get("Authorization"):
        if auth_header.startswith(BEARER_HEADER_PREFIX):
            if token := auth_header[len(BEARER_HEADER_PREFIX) :]:
                return token
    raise UnauthorizedError("Request does not have required IAM token")


def get_projects_with_access():
    iam_token = _get_iam_token()
    projects_with_access = []
    iam_as_client = iam_client.AsClient(config.get_value("iam.access_service_endpoint"))
    for project in Project.objects.only("id", "yc_iam_folder_id"):
        if not project.yc_iam_folder_id:
            continue
        action = iam_client.IamAction(iam_client.ProjectsPermissions.get, project.yc_iam_folder_id)
        try:
            iam_as_client.authorize(iam_token, [action], _get_request_id())
        except iam_client.IamAuthError:
            continue
        projects_with_access.append(project.id)
    return projects_with_access


def get_iam_user_info() -> str:
    iam_as_client = iam_client.AsClient(config.get_value("iam.access_service_endpoint"))
    return str(iam_as_client.authenticate(_get_iam_token(), _get_request_id()))


def _get_request_id():
    return flask.request.headers.get("x-request-id", str(uuid.uuid4()))
