"""Finite-state machine for long-running asks."""

import logging
import re
import time
from abc import abstractmethod

import gevent.event
import gevent.pool
from gevent.lock import RLock

# Load stage processing handlers
# noinspection PyUnresolvedReferences
import walle.fsm_stages.handlers  # noqa
from sepelib.core import config
from sepelib.core.constants import MINUTE_SECONDS
from sepelib.core.exceptions import Error
from walle.models import timestamp
from walle.stats import stats_manager as stats
from walle.util.misc import StopWatch

_SETTINGS_CHECKER_PERIOD = MINUTE_SECONDS

log = logging.getLogger(__name__)

"""Maximum task check interval.

Note: mocked in tests.
"""


class BaseFsm:
    """Finite-state machine for long-running host tasks."""

    _name = "Finite-state machine"

    def __init__(self, settings, shards_num_config_key, max_concurrency, partitioner):
        self._stop = False
        self._next_check = 0
        self._started = False
        self._settings = settings
        self._on_handbrake = _on_handbrake(settings)

        self._partitioner = partitioner
        self._main_event = gevent.event.Event()
        self._settings_event = gevent.event.Event()
        self._total_shards_count = config.get_value(shards_num_config_key)

        # TODO(rocco66): do we need it?
        self._all_shards = list(range(self._total_shards_count))

        self._pool = gevent.pool.Pool(config.get_value(max_concurrency))
        self._daemon_greenlet = None
        self._settings_checker_greenlet = None

        self._snake_class_name = self._get_snake_case_name()

    def _get_snake_case_name(self):
        pattern = re.compile(r'(?<!^)(?=[A-Z])')
        return pattern.sub('_', self.__class__.__name__).lower()

    def __enter__(self):
        return self

    def __exit__(self, exc_type, exc_val, exc_tb):
        self.stop()

    def start(self):
        if self._started:
            raise Error("{} is already started.", self._name)

        log.info("Starting %s...", self._name)

        self._stop = False
        self._main_event.clear()
        self._settings_event.clear()
        self._next_check = 0
        self._started = True

        try:
            self._partitioner.start()
            self._daemon_greenlet = gevent.spawn(self._daemon)
            self._settings_checker_greenlet = gevent.spawn(self._settings_checker)
        except Exception as e:
            self.stop()
            log.info("%s has crashed with exception: %s", self._name, e)

    def stop(self):
        if not self._started:
            return

        log.info("Stopping %s...", self._name)

        self._stop = True
        self._main_event.set()
        self._settings_event.set()

        log.info("Stopping partitioner for %s...", self._name)
        if self._partitioner is not None:
            self._partitioner.stop()
        log.info("Partitioner for %s is stopped", self._name)

        log.info("Stopping daemon greenlet for %s...", self._name)
        if self._daemon_greenlet is not None:
            self._daemon_greenlet.kill()
        log.info("Daemon greenlet for %s is stopped", self._name)

        log.info("Stopping settings greenlet for %s...", self._name)
        if self._settings_checker_greenlet is not None:
            self._settings_checker_greenlet.kill()
        log.info("Settings greenlet for %s is stopped", self._name)

        log.info("Stopping pool for %s...", self._name)
        if self._pool is not None:
            self._pool.kill()
        log.info("Pool for %s is stopped", self._name)

        log.info("%s has stopped.", self._name)
        self._started = False

    def notify(self):
        self._next_check = 0
        self._main_event.set()

    def trigger_settings_check(self):
        self._settings_event.set()

    def _check_settings(self):
        # no need to re-app
        with RLock():
            self._settings.reload()

            prev = self._on_handbrake
            self._on_handbrake = _on_handbrake(self._settings)
            if self._on_handbrake != prev:
                if self._on_handbrake:
                    log.info("Handbrake set, pausing tasks processing in %s.", self._name)
                else:
                    log.info("Handbrake lifted, resuming tasks processing in %s.", self._name)
                    self.notify()

    def _settings_checker(self):
        while not self._stop:
            try:
                self._check_settings()

                if self._stop:
                    break

                # we get notified on important settings change
                # check periodically to make sure we didn't miss the notification
                # use stop event instead of sleep to not miss the shutdown
                self._settings_event.clear()
                self._settings_event.wait(_SETTINGS_CHECKER_PERIOD)
            except:
                log.exception("%s daemon has crashed:", self._name)

    def _daemon(self):
        try:
            while not self._stop:
                if self._on_handbrake:
                    self._next_check = time.time() + _SETTINGS_CHECKER_PERIOD
                else:
                    if time.time() >= self._next_check:
                        stopwatch = StopWatch()
                        try:
                            self._state_machine()
                        finally:
                            stats.add_sample(("fsm", self._snake_class_name, "iteration_time"), stopwatch.get())

                if self._stop:
                    break

                wait_time = self._next_check - time.time()
                if wait_time > 0:
                    self._main_event.clear()
                    log.debug("Next check of %s in %.1f seconds.", self._name, wait_time)
                    self._main_event.wait(timeout=wait_time)
        except:
            log.exception("%s daemon has crashed:", self._name)

    @abstractmethod
    def _state_machine(self):
        raise NotImplementedError


def _on_handbrake(settings):
    return settings.fsm_handbrake is not None and settings.fsm_handbrake.timeout_time >= timestamp()
