import dataclasses
import enum
import typing as tp

from sepelib.core import config
from walle.clients import eine, deploy
from walle.yc.types import DnsApiType, YcDnsZone

DNS_API_BOXES_SECTION = "dns_api.boxes"
DEPLOY_BOXES_SECTION = "lui.boxes"
IPMIPROXY_BOXES_SECTION = "ipmiproxy.boxes"
EINE_BOXES_SECTION = "eine.boxes"

CONFIG_BOX_MAPPING_SECTION = "project_box_mapping"


class BoxType(str, enum.Enum):
    eine = "eine"
    dns = "dns"


class UnknownBox(Exception):
    """Raises when there is no config for box"""


# TODO(rocco66): use it in eine/lui/ipmiproxy clients
@dataclasses.dataclass
class EineBoxConfig:
    deploy_host: str
    ipmiproxy_host: str
    eine_host: str

    @staticmethod
    def from_box_id(box_id: str):
        # TODO(rocco66): refactor with ipmiproxy.get_yandex_internal_provider, ipmiproxy.get_yandex_box_provider_if_exists
        ipmiproxy_params = config.get_value(f"{IPMIPROXY_BOXES_SECTION}.{box_id}", None)
        if not ipmiproxy_params or not ipmiproxy_params.get("host"):
            raise UnknownBox(f"ipmiproxy '{box_id}'")
        ipmiproxy_host = ipmiproxy_params["host"]

        eine_provider = eine.get_yandex_box_provider_if_exists(box_id)
        if not eine_provider:
            raise UnknownBox(f"eine '{box_id}'")

        deploy_provider = deploy.get_yandex_box_provider_if_exists(box_id)
        if not deploy_provider:
            raise UnknownBox(f"eine '{box_id}'")

        return EineBoxConfig(
            deploy_host=deploy_provider.get_host(),
            ipmiproxy_host=ipmiproxy_host,
            eine_host=eine_provider.get_host(),
        )

    def to_api(self):
        return dataclasses.asdict(self)


@dataclasses.dataclass
class DnsBoxConfig:
    type: DnsApiType
    dns_endpoint: str
    iam_endpoint: str
    console_url: str
    service_account_id: str
    key_id: str
    private_key: str = dataclasses.field(repr=False)

    def get_dns_zone_link(self, dns_zone: YcDnsZone) -> str:
        return f"{self.console_url}/folders/{dns_zone.folder_id}/dns/zone/{dns_zone.id}/recordsets"

    def should_use_rurikk_dns(self):
        return self.type == DnsApiType.rurikk_dns

    def to_api(self):
        return {k: v for k, v in dataclasses.asdict(self).items() if k != "private_key"}


def get_dns_box_config(box_id: tp.Optional[str]) -> tp.Optional[DnsBoxConfig]:
    if box_id is None:
        return
    raw_config = config.get_value(f"{DNS_API_BOXES_SECTION}.{box_id}", None)
    if raw_config is None:
        return
    else:
        return DnsBoxConfig(**raw_config)


def get_box(project_id: str, box_type: BoxType) -> tp.Optional[str]:
    mapping = config.get_value(CONFIG_BOX_MAPPING_SECTION, {})
    if project_id not in mapping:
        return
    return mapping.get(project_id).get(box_type)


def get_all_dns_boxes() -> dict[str, DnsBoxConfig]:
    res = {}
    for box_name, raw_box_config in config.get_value(DNS_API_BOXES_SECTION, {}).items():
        res[box_name] = DnsBoxConfig(**raw_box_config)
    return res


def get_all_eine_boxes() -> dict[str, EineBoxConfig]:
    res = {}
    for box_name in config.get_value(EINE_BOXES_SECTION, {}):
        res[box_name] = EineBoxConfig.from_box_id(box_name)
    return res
