"""
https://abc-back.yandex-team.ru/api/v4/ API wrapper.
Docs: https://wiki.yandex-team.ru/intranet/abc/api/#aktualnoenovoev4api
"""
import typing as tp

from requests.exceptions import RequestException
from six.moves.urllib.parse import urlparse, parse_qs

from object_validator import List, String, DictScheme, Integer, Bool
from sepelib.core import config
from walle.clients.utils import json_request
from walle.errors import RecoverableError
from walle.util.misc import drop_none
from walle.util.validation import AnyScheme, NoneValue

PAGE_SIZE = 1000


HOST_SERVICE_SLUG = "host_service"
TVM_APP_RESOURCE_TYPE = 47


class Role:
    HARDWARE_RESOURCES_OWNER = "hardware_resources_owner"
    HARDWARE_RESOURCES_MANAGER = "hardware_resources_manager"
    PRODUCT_HEAD = "product_head"


class RoleScopeSlug:
    ADMINISTRATION = "administration"


class ABCInternalError(RecoverableError):
    def __init__(self, message, **kwargs):
        super().__init__("Error in communication with ABC: " + message, **kwargs)


def _api_request(method, path, params=None, data=None, scheme=None, cursor=None, page_size=None, **kwargs):
    """
    `page_size` and `cursor` are connected: passed page_size means we want paginated result, cursor value is
    calculated after the first call
    """
    url = "https://{host}/api/v4/{path}".format(host=config.get_value("abc.host"), path=path)
    headers = {"Authorization": "OAuth {}".format(config.get_value("abc.access_token", ""))}

    if page_size is not None:
        if params is None:
            params = {}
        params["page_size"] = page_size
        if cursor is not None:
            params["cursor"] = cursor

    try:
        return json_request("abc", method, url, params=params, data=data, headers=headers, scheme=scheme, **kwargs)
    except RequestException as e:
        raise ABCInternalError("{}".format(e), path=path, params=params, data=data, scheme=scheme, **kwargs)


def _paginated_api_request(*args, **kwargs):
    result = []
    cursor = None

    while True:
        request_kwargs = drop_none(dict(kwargs, page_size=PAGE_SIZE, cursor=cursor))
        resp = _api_request(*args, **request_kwargs)

        page = resp["results"]
        result.extend(page)
        if len(page) < PAGE_SIZE:
            break

        cursor = _extract_cursor_param(resp["next"])

    return result


def get_services(fields=(), **filters):
    """
    https://wiki.yandex-team.ru/intranet/abc/api/#get/api/v4/services/
    """
    item_scheme = DictScheme(
        {"id": Integer(), "slug": String(), "path": String(), "state": String()}, ignore_unknown=True
    )
    scheme = DictScheme({"results": List(item_scheme)}, ignore_unknown=True)

    filters = _fix_filter_kwargs(filters)

    params = dict(filters, fields=_services_fields_parameter(fields))

    result = _paginated_api_request("GET", path="services/", params=params, scheme=scheme)
    return result


def _sort_uniq(*seqs):
    uniq = set()
    for seq in seqs:
        uniq.update(seq)
    return sorted(uniq)


def _services_fields_parameter(fields):
    required_fields = ("id", "path", "slug", "state")
    return ",".join(_sort_uniq(required_fields, fields))


def _fix_filter_kwargs(filters):
    # st.yandex-team.ru/ABC-7127: ABC API doesn't support list of ids/slugs, fix it
    for filter_name in ("id", "slug", "state", "resource", "category"):
        if isinstance(filters.get(filter_name), (list, tuple)):
            filters["{}__in".format(filter_name)] = ','.join(str(id_) for id_ in filters.pop(filter_name))
    return filters


def get_service_by_id(service_id, fields=()):
    return get_services(id=service_id, fields=fields)[0]


def get_service_by_slug(service_slug, fields=()):
    return get_services(slug=service_slug, fields=fields)[0]


def get_service_slug(service_id):
    return get_service_by_id(service_id, fields=("slug",))["slug"]


def get_service_members(service_id=None, service_slug=None, role_codes=None, role_scope_slugs=None):
    """
    https://wiki.yandex-team.ru/intranet/abc/api/#postapi/v4/services/members/
    Multiple filters are treated like 'filter_1 AND filter_2`.
    """

    if not (service_id or service_slug):
        raise TypeError("Provide service_id or service_slug")

    params = {"fields": "person,role"}

    if service_id:
        params["service"] = service_id
    else:
        params["service__slug"] = service_slug
    if role_codes:
        params["role__code__in"] = ",".join(role_codes)
    if role_scope_slugs:
        params["role__scope__slug__in"] = ",".join(role_scope_slugs)

    item_scheme = DictScheme(
        {
            "person": DictScheme({"login": String(), "is_robot": Bool()}, delete_unknown=True),
            "role": DictScheme(
                {"scope": DictScheme({"slug": String()}, delete_unknown=True), "code": String()}, delete_unknown=True
            ),
        },
        delete_unknown=True,
    )
    scheme = DictScheme({"results": List(item_scheme)}, ignore_unknown=True)

    result = _paginated_api_request("GET", path="services/members/", params=params, scheme=scheme)
    return result


def _resources_fields_parameter(fields):
    required_fields = ("id", "resource", "state")
    return ",".join(_sort_uniq(required_fields, fields))


def get_resources_consumed_by_service(fields=(), **filters):
    """
    https://wiki.yandex-team.ru/intranet/abc/api/#get/api/v4/resources/consumers/
    """
    item_scheme = DictScheme(
        {
            "id": Integer(),
            "state": String(),
            "resource": DictScheme(
                {
                    "id": Integer(),
                    "external_id": AnyScheme([String(), NoneValue()]),
                },
                ignore_unknown=True,
            ),
        },
        ignore_unknown=True,
    )
    scheme = DictScheme({"results": List(item_scheme)}, ignore_unknown=True)

    filters = _fix_filter_kwargs(filters)

    params = dict(filters, fields=_resources_fields_parameter(fields))

    result = _paginated_api_request("GET", path="resources/consumers", params=params, scheme=scheme)
    return result


def get_service_tvm_app_ids(service_id):
    resp = get_resources_consumed_by_service(service=service_id, type=TVM_APP_RESOURCE_TYPE, state="granted")
    return [int(res["resource"]["external_id"]) for res in resp]


def _extract_cursor_param(next_):
    if next_ is None:
        return None

    parsed = urlparse(next_)
    query = parse_qs(parsed.query)
    return query["cursor"][0]


def get_service_on_duty_logins(service_slug: str, duty_schedule_slugs: tp.Optional[tp.List[str]] = None) -> [str]:
    """
    https://wiki.yandex-team.ru/intranet/abc/api/#get/api/v4/duty/onduty/
    """

    duty_schedule_str = ",".join(duty_schedule_slugs) if duty_schedule_slugs else None

    params = drop_none(
        {
            "service__slug": service_slug,
            "schedule__slug__in": duty_schedule_str,
            "fields": "person.login",
        }
    )

    item_scheme = DictScheme(
        {
            "person": DictScheme(
                {
                    "login": String(),
                },
                ignore_unknown=True,
            )
        },
        ignore_unknown=True,
    )
    scheme = List(item_scheme)

    response = _api_request("GET", path="duty/on_duty/", params=params, scheme=scheme)
    return [item["person"]["login"] for item in response]


def get_service_parent_slug(service_slug: str) -> str:
    result = get_service_by_slug(service_slug, fields=("parent",))
    return result["parent"]["slug"]
