"""Bot client.

Production: https://bot.yandex-team.ru/
Development: https://dev.bot.yandex-team.ru/

API requires acquired "Ответственный администратор" role on:
Production: https://idm.yandex-team.ru/#roleRequestForm=system:bot;role:role
Development: https://idm.test.yandex-team.ru/#roleRequestForm=system:bot;role:role

Attention: Inventory numbers look like integers, but in theory there might be inventory numbers with non-number
characters - they may be used for VMs and licenses, so it worth to be careful and handle such inventory numbers
properly.
"""

import logging
import operator
import re
import time
import typing as tp
from collections import namedtuple, OrderedDict
from contextlib import contextmanager

import requests
import simplejson as json
import six
from cachetools.func import ttl_cache
from requests import Response
from requests.exceptions import RequestException

import object_validator
import walle.clients.utils
import walle.util.misc
from object_validator import Integer, String, List
from sepelib.core import config, constants
from sepelib.core.exceptions import Error, LogicalError
from walle import constants as walle_constants
from walle.clients import otrs, startrek, tvm
from walle.clients.utils import iter_csv_response, strip_api_error
from walle.errors import (
    NoInformationError,
    InvalidHostConfiguration,
    ResourceAlreadyExistsError,
    RecoverableError,
    FixableError,
    ResourceConflictError,
)
from walle.models import timestamp
from walle.util import net
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.misc import drop_none, format_time, get_location_path, InvOrUUIDOrName
from walle.util.net import format_mac
from walle.util.validation import ApiDictScheme, StringToInteger

log = logging.getLogger(__name__)
bot_request_log = logging.getLogger(__name__ + "_requests")
slot_to_unit_re = re.compile(r"\s*(\d+)\w*\s*")


NO_PERMISSON_PART_MESSAGE = "have no permissions to change service"


class PreorderStatus:
    OPEN = "OPEN"
    CLOSED = "CLOSE"

    ALL = [OPEN, CLOSED]


class SetupRequestStatus:
    INVENTORIED = "inventoried"
    WAIT_TEST = "wait_test"
    IN_TEST = "in_test"
    TEST_FAIL = "test_fail"
    TEST_OK = "test_ok"
    DELIVERING = "delivering"
    DELIVERED = "delivered"
    DISTRIBUTING = "distributing"
    DISTRIBUTED = "distributed"
    CANCELED = "canceled"

    # our internal status for marking cases that are not covered by statuses used by BOT
    NOT_PREORDERED = ".not_preordered"  # host has never been preordered, no status available

    # these statuses should be considered "okay": hosts have not been preordered or distributed but are in production
    ALL_NO_NEED_TO_WAIT = [NOT_PREORDERED]

    ALL_BOT = [
        INVENTORIED,
        WAIT_TEST,
        IN_TEST,
        TEST_FAIL,
        TEST_OK,
        DELIVERING,
        DELIVERED,
        DISTRIBUTING,
        DISTRIBUTED,
        CANCELED,
    ]


class ServerStatus:
    NEW = "new"
    TAKEN = "taken"

    ALL = [NEW, TAKEN]


class BotRequestStatus:
    NEW = "NEW"
    """Request is not processed"""

    WORK = "WORK"
    """Request is being processed"""

    OK = "OK"
    """Request processed"""

    DELETED = "DELETED"
    """Request deleted"""


class BotRequest:
    RESULT_OK = 1
    RESULT_ERR = 2
    RESULT_FIELD = "res"
    SYSTEM_MODEL_FIELD = "attribute12"
    SYSTEM_MODEL_FALLBACK_FIELD = "attribute23"
    COMPONENT_TYPE_FIELD = "item_segment3"
    COMPONENT_MODEL_NAME_FIELD = "item_segment1"
    COMPONENTS_FIELD = "Components"
    CONNECTED_FIELD = "Connected"
    SHARED_FIELD = "Shared"
    INSTANCE_INV_FIELD = "instance_number"
    REQUEST_DATA = "data"
    DISK_SERIAL_NUMBER_FIELD = "XXCSI_SERIAL_NUMBER"
    DISK_CAPACITY_GB_FIELD = "attribute14"
    DISK_INTERFACE_FIELD = "attribute15"
    DISK_TYPE_FIELD = "attribute16"


class BotDiskType:
    HDD = "HDD"
    SSD = "SSD"

    ALL = [HDD, SSD]


class BotDiskInterface:
    SATA = "SATA"
    SAS = "SAS"
    U2 = "U.2"
    M2 = "M.2"

    ALL = [SATA, SAS, U2, M2]


class DiskKind:
    HDD = "hdds"
    SSD = "ssds"
    NVME = "nvmes"

    ALL = [HDD, SSD, NVME]

    @staticmethod
    def convert_from_bot(bot_disk_iface, bot_disk_type):
        descr_to_kind = {
            (BotDiskInterface.SATA, BotDiskType.HDD): DiskKind.HDD,
            (BotDiskInterface.SAS, BotDiskType.HDD): DiskKind.HDD,
            (BotDiskInterface.SATA, BotDiskType.SSD): DiskKind.SSD,
            (BotDiskInterface.U2, BotDiskType.SSD): DiskKind.NVME,
            (BotDiskInterface.M2, BotDiskType.SSD): DiskKind.NVME,
        }
        disk_descr = bot_disk_iface, bot_disk_type
        if disk_descr in descr_to_kind:
            return descr_to_kind[disk_descr]
        else:
            log.error("Got unsupported combination of disk properties: %s", disk_descr)
            raise LogicalError()


class StorageType:
    STORAGES = "STORAGES"
    NODE_STORAGE = "NODE-STORAGE"

    ALL = [STORAGES, NODE_STORAGE]


class BotInternalError(RecoverableError):
    def __init__(self, message, *args, **kwargs):
        super().__init__("Error in communication with BOT: {}".format(message), *args, **kwargs)


class BotProjectIdNotFound(ResourceConflictError):
    pass


class BotPersistentError(FixableError):
    pass


class InvalidPreorderIdError(BotPersistentError):
    def __init__(self, preorder_id, **kwargs):
        super().__init__("Preorder #{} doesn't exist.".format(preorder_id), **kwargs)


class InvalidInventoryNumber(BotPersistentError):
    def __init__(self, inv, **kwargs):
        super().__init__("Host with #{} inventory number is not registered in BOT.".format(inv, **kwargs))


class PlannerIdNotFound(FixableError):
    def __init__(self, planner_id, **kwargs):
        super().__init__("Could not find matching bot project id for Planner id {}".format(planner_id), **kwargs)


DiskInfo = namedtuple("DiskInfo", ["kind", "instance_number", "serial_number", "capacity_gb", "from_node_storage"])
HostDiskConf = namedtuple("HostDiskConf", DiskKind.ALL)
StorageInfo = namedtuple("StorageInfo", ("type", "inv"))


_RequestIterator = namedtuple("_RequestIterator", ["timestamp", "iterator"])
_HardwareInfo = namedtuple("HardwareInfo", ["inv", "status", "type", "group", "location_id", "name"])
_HardwareLocation = namedtuple("HardwareLocation", ["country", "city", "datacenter", "queue", "rack", "unit"])


class HardwareLocation(_HardwareLocation):
    def get_path(self):
        return get_location_path(self)


HostLocationInfo = namedtuple("HostLocationInfo", ["inv", "name", "location"])
HostPlatform = namedtuple("HostPlatform", ["system", "board"])


def get_preorder_info(preorder_id, authenticate=True):
    """Returns information about the specified preorder.

    If server has been added to preorder by a mistake and it hasn't been taken yet, it could be silently replaced with a
    right one.

    Closed preorder means that all requested servers has been added to the preorder.
    """

    # API returns Bad Request for id=0
    if preorder_id < 1:
        raise InvalidPreorderIdError(preorder_id)

    ok_scheme = ApiDictScheme(
        {
            "id": StringToInteger(optional=True),  # May be null for some preorders by some reason
            "info": ApiDictScheme(
                {
                    "status": String(choices=PreorderStatus.ALL),
                    "author": String(regex=walle_constants.LOGIN_RE),
                    "service_id": StringToInteger(),
                }
            ),
            "servers": List(ApiDictScheme({"inv": StringToInteger(min=0), "status": String(choices=ServerStatus.ALL)})),
        },
        drop_none=True,
    )

    response, result = json_request(
        "/api/preorders.php",
        params={"id": preorder_id},
        check_status=True,
        allow_errors=(requests.codes.bad_request, requests.codes.not_found),
        with_response=True,
        scheme=ok_scheme,
        authenticate=authenticate,
        error_scheme=ApiDictScheme({"errmsg": String()}),
    )

    if response.status_code == requests.codes.not_found:
        raise InvalidPreorderIdError(preorder_id)

    if response.status_code != requests.codes.ok:
        raise BotInternalError(
            "The server returned an error: {}".format(strip_api_error(result["errmsg"])),
            error=strip_api_error(result["errmsg"]),
            status_code=response.status_code,
            result=result,
            preorder_id=preorder_id,
        )

    servers = {server["inv"]: server["status"] for server in result["servers"]}
    if len(servers) != len(result["servers"]):
        raise BotInternalError("Got duplicated server entries.", preorder_id=preorder_id, result=result)

    return {
        "id": preorder_id,
        "status": result["info"]["status"],
        "owner": result["info"]["author"],
        "servers": servers,
        "bot_project_id": result["info"]["service_id"],
    }


def get_host_preorder_status(inv: int) -> tuple[str, int]:
    """Fetches information about host's status in BOT.
    API schema from https://wiki.yandex-team.ru/dca/services/hwr/api/v1/
    """
    ok_scheme = ApiDictScheme(
        {
            "data": List(
                ApiDictScheme(
                    {
                        "status": ApiDictScheme(
                            {
                                "code": String(choices=SetupRequestStatus.ALL_BOT),
                            }
                        ),
                        "order": ApiDictScheme(
                            {
                                "id": Integer(),
                            }
                        ),
                    }
                )
            )
        }
    )

    response, result = json_request(
        "/v1/compute-node-setups",
        params={
            "fields": "status.code,order.id",
            "filter": "inv:\"{}\"".format(inv),
            "limit": 1,
            "offset": 0,
            "sort": "inv",
        },
        check_status=True,
        with_response=True,
        scheme=ok_scheme,
        authenticate=True,
        error_scheme=ApiDictScheme({"message": String()}),
        tvm=True,
        host=config.get_value("bot.hwr.host"),
    )

    if response.status_code != requests.codes.ok:
        raise BotInternalError(
            "The server returned an error: {}".format(strip_api_error(result["message"])),
            error=strip_api_error(result["message"]),
            status_code=response.status_code,
            result=result,
            inv=inv,
        )
    if len(result["data"]) == 0:
        # This is an old host, was commissioned without preorder.
        return SetupRequestStatus.NOT_PREORDERED, None

    return result["data"][0]["status"]["code"], result["data"][0]["order"]["id"]


def acquire_preordered_host(order_id, inv, name):
    """Acquires a preordered host.
    API schema from https://wiki.yandex-team.ru/dca/services/hwr/api/v1/
    """

    host_info = get_host_info(inv)
    if host_info is None:
        raise InvalidInventoryNumber(inv)

    result = raw_request(
        "/compute-node-setups/{}/_take".format(order_id),
        stream=True,
        authenticate=True,
        method="POST",
        data={int(inv): "{}".format(name)},
        check_status=False,
        tvm=True,
        host=config.get_value("bot.hwr.host"),
    )

    if result.status_code != 200:
        error = result.json().get('message', 'unknown error')
        raise BotInternalError(
            "Failed to acquire #{} from BOT: {}".format(inv, error),
            error=error,
            status_code=result.status_code,
            inv=inv,
        )


@ttl_cache(maxsize=1, ttl=15 * constants.MINUTE_SECONDS)
def missed_preordered_hosts():
    response = json_request(
        "/api/missed.php",
        params={"days": 0},
        authenticate=True,
        scheme=List(
            ApiDictScheme(
                {
                    "inv": StringToInteger(),
                    "order": StringToInteger(),
                }
            )
        ),
    )

    return {host["inv"]: host for host in response}


def get_inv_by_name(name):
    """Returns host inventory number by its name or None if host is not registered in Bot."""

    response = raw_request("/api/inv-by-name.php", params={"name": name})
    reply = response.strip().split("\n")

    result = reply[0].strip()
    if result == "ERROR":
        return

    if result == "OK" and len(reply) == 2:
        try:
            return int(reply[1].strip())
        except ValueError:
            pass

    log.error("Got an invalid response from Bot: %r", response)
    raise BotInternalError("Got an invalid response: {!r}".format(response))


def get_host_info(inv_or_name, with_macs=False, with_platform=False):
    """Returns OEBS info for the specified host or None if host is not registered in Bot.

    See https://wiki.yandex-team.ru/bot/api/osinfo for details.
    """
    inv_or_name = InvOrUUIDOrName(inv_or_name)
    result = json_request(
        "/api/osinfo.php",
        params=inv_or_name.kwargs(
            name_arg="fqdn",
            output="|".join(
                (
                    "instance_id",
                    "GROUP_OWNED_id",
                    "instance_number",
                    "XXCSI_FQDN",
                    "status_name",
                    "XXCSI_IPMI",
                    "XXCSI_MACADDRESS1",
                    "XXCSI_MACADDRESS2",
                    "XXCSI_MACADDRESS3",
                    "XXCSI_MACADDRESS4",
                    "loc_segment1",
                    "loc_segment2",
                    "loc_segment3",
                    "loc_segment4",
                    "loc_segment5",
                    "loc_segment6",
                )
            ),
        ),
        scheme=ApiDictScheme(
            {
                # Can't validate against the full scheme here: when `res` == RESULT_ERR, `os` == "not found" (not a dict).
                "res": Integer(),
            }
        ),
    )

    if result["res"] == BotRequest.RESULT_ERR:
        return

    if result["res"] == BotRequest.RESULT_OK:
        scheme = {
            "os": List(
                ApiDictScheme(
                    {
                        # All optional fields may be None or ""
                        "instance_id": String(min_length=1),
                        "instance_number": String(min_length=1),
                        "GROUP_OWNED_id": String(optional=True),
                        "status_name": String(min_length=1),
                        "XXCSI_FQDN": String(optional=True),
                        "XXCSI_IPMI": String(optional=True),
                        "XXCSI_MACADDRESS1": String(optional=True),
                        "XXCSI_MACADDRESS2": String(optional=True),
                        "XXCSI_MACADDRESS3": String(optional=True),
                        "XXCSI_MACADDRESS4": String(optional=True),
                        "loc_segment1": String(min_length=1),
                        "loc_segment2": String(min_length=1),
                        "loc_segment3": String(min_length=1),
                        "loc_segment4": String(min_length=1),
                        "loc_segment5": String(min_length=1),
                        "loc_segment6": String(min_length=1),
                    },
                    drop_none=True,
                )
            ),
        }
    else:
        scheme = {"msg": String()}

    try:
        result = object_validator.validate("result", result, ApiDictScheme(scheme))
    except object_validator.ValidationError as e:
        log.error("Got an invalid JSON response from BOT: %s (%s).", result, e)
        raise BotInternalError(
            "The server returned an invalid JSON response: {!r} {}".format(result, e), inv_or_name=inv_or_name
        )

    if result["res"] != BotRequest.RESULT_OK:
        error = strip_api_error(result["msg"])
        raise BotInternalError("The server returned an error: {}".format(error), error=error, inv_or_name=inv_or_name)

    for entry in result["os"]:
        oebs_id, inv = entry["instance_id"], entry["instance_number"]
        bot_project_id, name = (
            entry.get("GROUP_OWNED_id", "").strip() or None,
            entry.get("XXCSI_FQDN", "").strip() or None,
        )

        try:
            oebs_id, inv = int(oebs_id), int(inv)
            if oebs_id < 0 or inv < 0:
                raise ValueError
        except ValueError:
            raise BotInternalError(
                "Got an invalid host OEBS ID or inventory number: {}, {}".format(oebs_id, inv), inv_or_name=inv_or_name
            )

        if bot_project_id is not None:
            try:
                bot_project_id = int(bot_project_id)
                if bot_project_id < 0:
                    raise ValueError
            except ValueError:
                raise BotInternalError(
                    "Got invalid OEBS project id for the host: {}".format(bot_project_id), inv_or_name=inv_or_name
                )

        if inv_or_name in (inv, name):
            break
    else:
        return

    ipmi_mac = entry.get("XXCSI_IPMI", "").strip() or None
    if ipmi_mac is not None:
        try:
            ipmi_mac = format_mac(ipmi_mac)
        except Error as e:
            raise BotInternalError("{}".format(e), inv_or_name=inv_or_name, data=entry, mac=ipmi_mac)

    try:
        location = _get_hardware_location(
            *(_get_value(entry["loc_segment{}".format(loc_id)]) for loc_id in range(1, 7))
        )
    except ValueError:
        raise BotInternalError(
            "Got an invalid host location.",
            inv_or_name=inv_or_name,
            data=entry,
        )

    return drop_none(
        {
            "id": oebs_id,
            "inv": inv,
            # May be missing
            "name": name.lower() if name else None,
            "ipmi_mac": ipmi_mac,
            "macs": _get_host_macs(inv, entry) if with_macs else None,
            "location": location,
            "oebs_status": entry["status_name"],
            "bot_project_id": bot_project_id,
            "platform": _get_host_platform(inv_or_name) if with_platform else None,
        }
    )


def _consistof_request(params, data_scheme):
    params = dict(format="json", **params)
    scheme = ApiDictScheme(
        {
            BotRequest.RESULT_FIELD: Integer(choices=[BotRequest.RESULT_OK, 0]),
            BotRequest.REQUEST_DATA: ApiDictScheme(data_scheme, optional=True),
            "errmsg": String(optional=True),
        },
        mutually_exclusive_required=(BotRequest.REQUEST_DATA, "errmsg"),
    )

    resp = json_request("/api/consistof.php", params=params, scheme=scheme)

    errmsg = resp.get("errmsg")
    if errmsg is not None:
        msg = "Failed to get host's hardware components info (params: {}): The server returned an error: {}"
        raise BotInternalError(
            msg.format(params, strip_api_error(errmsg)), params=params, error=strip_api_error(errmsg)
        )
    else:
        return resp[BotRequest.REQUEST_DATA]


def _get_host_platform(inv_or_name):
    """Get host system and motherboard models"""
    if not isinstance(inv_or_name, InvOrUUIDOrName):
        inv_or_name = InvOrUUIDOrName(inv_or_name)

    body_scheme = {
        BotRequest.INSTANCE_INV_FIELD: String(min_length=1),
        BotRequest.SYSTEM_MODEL_FIELD: String(min_length=1),
        BotRequest.SYSTEM_MODEL_FALLBACK_FIELD: String(min_length=1, optional=True),
        BotRequest.COMPONENTS_FIELD: List(
            ApiDictScheme(
                {
                    BotRequest.COMPONENT_MODEL_NAME_FIELD: String(min_length=1),
                    BotRequest.COMPONENT_TYPE_FIELD: String(min_length=1),
                }
            )
        ),
    }
    result_data = _consistof_request(inv_or_name.kwargs(name_arg="name"), body_scheme)

    # there could be no MOTHERBOARD for host in BOT, so we'll set it to None
    board = None
    for component in result_data[BotRequest.COMPONENTS_FIELD]:
        if component[BotRequest.COMPONENT_TYPE_FIELD] == "MOTHERBOARD":
            board = component[BotRequest.COMPONENT_MODEL_NAME_FIELD]
            break

    system = result_data[BotRequest.SYSTEM_MODEL_FIELD]
    if system == "N/A":
        system = result_data.get(BotRequest.SYSTEM_MODEL_FALLBACK_FIELD, "N/A")

    return HostPlatform(system=system, board=board)


def _get_host_macs(inv, os_info):
    all_macs = _get_mac_addresses(os_info)

    data_scheme = {
        "Components": List(
            ApiDictScheme(
                {
                    # All optional fields may be None or ""
                    "item_segment2": String(optional=True),
                    "item_segment3": String(optional=True),
                    "XXCSI_MACADDRESS1": String(optional=True),
                }
            ),
            optional=True,
        )
    }
    # OS info contains MAC addresses for motherboard's NICs only.
    result_data = _consistof_request({"inv": inv}, data_scheme)

    for component in result_data.get("Components", []):
        if not (
            component.get("item_segment3") == "EXPANSIONCARDS" and component.get("item_segment2") == "ETHERNETCARDS"
        ):
            continue
        macs_for_non_nic = _get_mac_addresses(component)
        all_macs |= macs_for_non_nic

    return sorted(all_macs)


def get_host_disk_configuration(host_inv):
    component_schema = ApiDictScheme(
        {
            BotRequest.COMPONENT_TYPE_FIELD: String(optional=True),
            BotRequest.INSTANCE_INV_FIELD: StringToInteger(optional=True),
            BotRequest.DISK_SERIAL_NUMBER_FIELD: String(optional=True),
            BotRequest.DISK_CAPACITY_GB_FIELD: String(optional=True),
            BotRequest.DISK_INTERFACE_FIELD: String(optional=True),
            BotRequest.DISK_TYPE_FIELD: String(optional=True),
        }
    )
    connected_component_schema = ApiDictScheme(
        {
            BotRequest.COMPONENT_TYPE_FIELD: String(optional=True),
            BotRequest.INSTANCE_INV_FIELD: StringToInteger(optional=True),
            BotRequest.COMPONENTS_FIELD: List(component_schema, optional=True),
        }
    )
    data_scheme = {
        BotRequest.COMPONENTS_FIELD: List(component_schema, optional=True),
        BotRequest.CONNECTED_FIELD: List(connected_component_schema, optional=True),
        BotRequest.SHARED_FIELD: List(component_schema, optional=True),
    }
    result_data = _consistof_request({"inv": host_inv, "shared": 1}, data_scheme)

    disk_info_map = {dt: [] for dt in DiskKind.ALL}

    def fill_disk_info(components, from_node_storage=False):
        for component in components:
            component_type = component.get(BotRequest.COMPONENT_TYPE_FIELD)
            if component_type == "NODE-STORAGE":
                fill_disk_info(component.get(BotRequest.COMPONENTS_FIELD, []), from_node_storage=True)
            elif component_type == "DISKDRIVES":
                disk_kind = DiskKind.convert_from_bot(
                    component[BotRequest.DISK_INTERFACE_FIELD], component[BotRequest.DISK_TYPE_FIELD]
                )
                disk_info_map[disk_kind].append(
                    DiskInfo(
                        kind=disk_kind,
                        instance_number=component[BotRequest.INSTANCE_INV_FIELD],
                        serial_number=component[BotRequest.DISK_SERIAL_NUMBER_FIELD],
                        capacity_gb=int(component[BotRequest.DISK_CAPACITY_GB_FIELD]),
                        from_node_storage=from_node_storage,
                    )
                )

    fill_disk_info(result_data.get(BotRequest.COMPONENTS_FIELD, []))
    fill_disk_info(result_data.get(BotRequest.CONNECTED_FIELD, []))
    fill_disk_info(result_data.get(BotRequest.SHARED_FIELD, []), from_node_storage=True)

    return HostDiskConf(**disk_info_map)


def get_storages(host_inv: int) -> tp.List[StorageInfo]:
    connected_component_schema = ApiDictScheme(
        {BotRequest.COMPONENT_TYPE_FIELD: String(optional=True), BotRequest.INSTANCE_INV_FIELD: String(optional=True)}
    )

    data_scheme = {BotRequest.CONNECTED_FIELD: List(connected_component_schema, optional=True)}

    result_data = _consistof_request({"inv": host_inv, "shared": 1}, data_scheme)
    storages = []

    for component in result_data.get(BotRequest.CONNECTED_FIELD, []):
        component_type = component.get(BotRequest.COMPONENT_TYPE_FIELD)
        if component_type in StorageType.ALL:
            storages.append(StorageInfo(type=component_type, inv=component[BotRequest.INSTANCE_INV_FIELD]))

    return storages


def _get_mac_addresses(data):
    mac_addresses = set()
    for mac_id in "XXCSI_MACADDRESS1", "XXCSI_MACADDRESS2", "XXCSI_MACADDRESS3", "XXCSI_MACADDRESS4":
        mac = data.get(mac_id, "").strip()
        if not mac:
            continue

        try:
            mac = format_mac(mac)
        except Error as e:
            raise BotInternalError("{}".format(e), data=data, mac=mac)

        mac_addresses.add(mac)

    return mac_addresses


def iter_hosts_info():
    # See http://bot.yandex-team.ru/api/view.php?name=view_oops_hardware&mode=describe for field description
    _HostInfoData = namedtuple(
        "HostInfo",
        (
            "inv",
            "fqdn",
            "status",
            "country",
            "city",
            "datacenter",
            "queue",
            "rack",
            "unit",
            "location",
            "mac1",
            "mac2",
            "mac3",
            "mac4",
            "ExMACs",
            "service",
            "segment1",
            "segment2",
            "segment3",
            "segment4",
            "motherboard",
            "ipmi_mac",
            "planner_id",
            "connected_slot",
        ),
    )

    response = _view_iterator("view_oops_hardware", row_type=_HostInfoData, timeout=120).iterator

    for data in response:
        try:
            inv = int(data.inv)
            planner_id = data.planner_id if data.planner_id else None
            macs = _get_host_info_macs(data)
            ipmi_mac = _get_ipmi_mac(data)
            location = _get_hardware_location(
                data.country,
                data.city,
                data.datacenter,
                data.queue,
                _modify_rack(data.rack),
                _connected_slot_to_unit(data.connected_slot),
            )
        except ValueError as e:
            log.error("Got an invalid data from BOT, error: {}, data: {!r}.".format(e, data))
            raise BotInternalError("Got an invalid data for host #{}: {}.".format(data.inv, e), inv=inv, data=data)

        yield drop_none(
            {
                "inv": inv,
                "macs": macs,
                "location": location,
                "oebs_status": data.status,
                # Optional fields
                "name": data.fqdn.lower() or None,
                "ipmi_mac": ipmi_mac,
                "planner_id": planner_id,
            }
        )


def _connected_slot_to_unit(slot):
    match = slot_to_unit_re.match(slot)
    if match:
        return match.group(1)
    return "0"


def _modify_rack(rack):
    if rack:
        rack = rack.replace("#", "--")
    return rack


def _get_ipmi_mac(host_info):
    if host_info.ipmi_mac:
        return net.format_mac(host_info.ipmi_mac)

    return None


def _get_host_info_macs(host_info):
    onboard_macs = [host_info.mac1, host_info.mac2, host_info.mac3, host_info.mac4]
    expansion_macs = host_info.ExMACs.split(",")

    return sorted(set(map(net.format_mac, filter(None, onboard_macs + expansion_macs))))


def rename_host(inv, new_name):
    """Renames host in BOT.

    Attention: This is an asynchronous operation. Usually it takes 15 seconds.
               IPMI FQDNs updating is fully independent task that runs under cron and periodically syncs DNS with BOT.

    :raises BotInternalError, InvalidHostConfiguration, ResourceAlreadyExistsError
    """

    host_info = get_host_info(inv)
    if host_info is None:
        raise InvalidHostConfiguration("The host doesn't exist in BOT: {}".format(inv), inv=inv)

    if host_info.get("name") == new_name:
        log.info("No need to rename host #%s: BOT already have actual information", inv)
        return

    request = {
        str(host_info["id"]): {
            "inv": str(inv),
            "new_fqdn": new_name,
        }
    }

    response = json_request(
        "/adm/rename.php",
        method="POST",
        data={
            "data": json.dumps(request),
            "oauth_token": config.get_value("bot.access_token"),
        },
        scheme=ApiDictScheme(
            {"Ok": String(optional=True), "Warning": String(optional=True), "Error": String(optional=True)}
        ),
        check_content_type=False,
    )

    ok, warning, error = response.get("Ok", ""), response.get("Warning", ""), response.get("Error", "")

    if not ok:
        if not error:
            log.error("Rename operation failed with an empty error message: %s", response)
            raise BotInternalError("Rename operation failed with an empty error message.")

        error_message = strip_api_error(error, strip_html=True)
        error_code = error_message.lower()

        if "не является корректным fqdn" in error_code:
            error_class = InvalidHostConfiguration
        elif "уже используется" in error_code:
            error_class = ResourceAlreadyExistsError
        else:
            error_class = BotInternalError

        raise error_class("Rename operation failed with error: {}", error_message)

    if warning:
        log.error(
            "Got the following warning during renaming #%s to %s in BOT: %s",
            inv,
            new_name,
            strip_api_error(warning, strip_html=True),
        )


def assign_project_id(inv, new_project_id):
    """Change OEBS project id for the host.

    Attention: This is an asynchronous operation. Usually it takes 15 seconds.
    :raises BotInternalError, InvalidHostConfiguration
    """

    host_info = get_host_info(inv)
    if host_info is None:
        raise InvalidHostConfiguration("The host doesn't exist in BOT: {}".format(inv))

    if host_info.get("bot_project_id") == new_project_id:
        log.info("No need to change OEBS project id on #%s host: BOT already have actual information", inv)
        return

    request = {
        str(host_info["id"]): {
            "inv": str(inv),
            "project": new_project_id,
        }
    }

    response = json_request(
        "/adm/rename.php",
        method="POST",
        data={
            "data": json.dumps(request),
            "oauth_token": config.get_value("bot.access_token"),
        },
        scheme=ApiDictScheme(
            {"Ok": String(optional=True), "Warning": String(optional=True), "Error": String(optional=True)}
        ),
        check_content_type=False,
        verify=False,
    )

    ok, warning, error = response.get("Ok", ""), response.get("Warning", ""), response.get("Error", "")

    if not ok:
        if not error:
            log.error("Change OEBS project operation failed with an empty error message: %s", response)
            raise BotInternalError("Change OEBS project operation failed with an empty error message.")

        error_message = strip_api_error(error, strip_html=True)
        raise BotInternalError(f"Change OEBS project operation failed with error: {error_message}")

    if warning:
        warning_template = "Got the following warning during change OEBS project to %s operation on host #%s BOT: %s"
        api_warning = strip_api_error(warning, strip_html=True)
        log.error(warning_template, new_project_id, inv, api_warning)
        if NO_PERMISSON_PART_MESSAGE in api_warning:
            warning_message = warning_template % (new_project_id, inv, api_warning)
            raise BotInternalError(f"Change OEBS project operation failed with warning: {warning_message}")


def is_excepted_status(host_status):
    return host_status not in walle_constants.BotHostStatus.get_production_host_statuses()


def get_host_location_info():
    for info in iter_hosts_info():
        yield HostLocationInfo(info["inv"], info.get("name", None), info["location"])


def get_locations():
    _GolemLocation = namedtuple("GolemLocation", "location_id country city building queue rack unit description")
    locations = {}

    for data in gevent_idle_iter(_view_iterator("view_golem_locations", row_type=_GolemLocation).iterator):
        try:
            location_id = int(data.location_id)
            location = _get_hardware_location(data.country, data.city, data.building, data.queue, data.rack, data.unit)
        except ValueError as e:
            log.error("Got an invalid data from Bot, error: {}, data: {!r}".format(data, e))
            raise BotInternalError("Got an invalid data.")

        locations[location_id] = location

    return locations


def get_rt_location(location):
    """Converts BOT location to RackTables location.

    At this time there is no direct mapping from BOT location scheme to RackTables location scheme.

    :raises NoInformationError
    """

    location_name = None

    if location.country == "RU":
        if location.city in ("IVA", "MYT", "SAS", "VLADIMIR"):
            location_name = location.city
        elif location.city == "MOW" and location.datacenter in ("FOL", "UGRB"):
            location_name = location.datacenter
    elif location.country == "FI" and location.city == "MANTSALA":
        location_name = location.city

    if location_name is None:
        raise NoInformationError("Can't translate BOT location {} into RackTables location.", location.get_path())

    return location_name[:3]


def get_known_hosts():
    """Returns inv->name and name->inv mappings for all hardware assigned to production.

    Attention: Hosts excepted from production and hosts that haven't been acquired by their owners are not returned. Use
    get_host_info() to get information about such hosts.
    """

    inv_name = {}
    name_inv = {}

    response_timestamp, hardware_info = iter_hardware_info()
    for info in gevent_idle_iter(hardware_info):
        inv, name = info.inv, info.name

        if inv in inv_name:
            log.error("Got conflicting data from Bot: #%s->%s vs #%s->%s.", inv, name, inv, inv_name[inv])
        elif name is not None and name in name_inv:
            log.error("Got conflicting data from Bot: #%s->%s vs #%s->%s.", inv, name, name_inv[name], name)

        inv_name[inv] = name
        if name is not None:
            name_inv[name] = inv

    if not inv_name or not name_inv:
        raise BotInternalError("Got an empty hardware list.")

    return inv_name, name_inv, response_timestamp


def get_ipmi_macs():
    """Returns Inventory number -> IPMI MAC mapping for all hardware assigned to production.

    Attention:
    * Hosts excepted from production and hosts that haven't been acquired by their owners aren't returned. Use
    get_host_info() to get information about such hosts.
    * get_ipmi_macs() (http://bot.yandex-team.ru/api/view.php?name=view_ipmi_list) returns more info than
    iter_hosts_info() (http://bot.yandex-team.ru/api/view.php?name=view_oops_hardware) for some reason, so it's
    preferable to be used for obtaining IPMI MAC info.
    """

    _IpmiData = namedtuple("IpmiData", "inv motherboard ipmi_mac")
    inv_mac = {}

    for data in gevent_idle_iter(_view_iterator("view_ipmi_list", row_type=_IpmiData, timeout=90).iterator):
        try:
            inv = int(data.inv)
            if inv < 0:
                raise ValueError

            ipmi_mac = format_mac(data.ipmi_mac)
        except Exception as e:
            log.error("Got an invalid data from Bot, error: {}, data: {!r}".format(data, e))
            raise BotInternalError("Got an invalid data.")

        inv_mac[inv] = ipmi_mac

    return inv_mac


def iter_hardware_info():
    """Returns basic information for all hardware assigned to production.

    Attention: Hosts excepted from production and hosts that haven't been acquired by their owners are not returned.
    Use get_host_info() to get information about such hosts.
    """

    response_timestamp, response_data = _view_iterator("view_golem_hardware", row_type=_HardwareInfo)
    return _RequestIterator(response_timestamp, _iter_golem_data(response_data))


def _iter_golem_data(response_data):
    """This is the  worker method for iter_hardware_info.
    Returns basic information for all hardware assigned to production.

    Attention: Hosts excepted from production and hosts that haven't been acquired by their owners are not returned.
    Use get_host_info() to get information about such hosts.
    """
    for data in response_data:
        if data.type != "SRV" or data.group not in ("NODES", "SERVERS"):
            continue

        try:
            inv = int(data.inv)
            name = data.name.lower() if data.name else None
            location_id = int(data.location_id)
        except (ValueError, AttributeError) as e:
            log.error("Got an invalid data from Bot, error: {}, data: {!r}".format(data, e))
            raise BotInternalError(
                "Got an invalid data.",
            )

        yield data._replace(inv=inv, name=name, location_id=location_id)


def get_admin_requests(inv, request_type, period, limit):
    bot_date_format = "%d.%m.%y"

    end_time = time.time()
    start_time = end_time - period

    bot_requests = json_request(
        "/adm/requestshow.php",
        check_content_type=False,
        params={
            "OsInv": inv,
            "RequestType": request_type,
            "TimeCreate": format_time(start_time, bot_date_format) + "-" + format_time(end_time, bot_date_format),
            "Limit": limit,
            "export": "json",
            "oauth_token": config.get_value("bot.access_token"),
        },
        scheme=List(
            ApiDictScheme(
                {
                    "TimeStampCreate": StringToInteger(),
                    "Initiator": String(),
                    "OtrsNum": String(optional=True),
                    "StNum": String(optional=True),
                    "Status": String(),
                },
                drop_none=True,
            )
        ),
    )

    admin_requests = []

    for bot_request in sorted(bot_requests, key=operator.itemgetter("TimeStampCreate"), reverse=True)[:limit]:
        admin_request = {
            "time": bot_request["TimeStampCreate"],
            "issuer": bot_request["Initiator"] + "@",
            "status": bot_request["Status"],
        }

        otrs_num = bot_request.get("OtrsNum")
        st_num = bot_request.get("StNum")

        if st_num is not None:
            admin_request["url"] = startrek.get_ticket_url_by_id(st_num)
        elif otrs_num is not None:
            admin_request["url"] = otrs.get_ticket_url_by_id(otrs_num)
        else:
            if bot_request["Status"] != BotRequestStatus.DELETED:
                log.error(
                    "#%s: Got unexpected data from BOT:"
                    " no startrek or OTRS ticket id and some strange status for request %s: %s",
                    inv,
                    bot_request["Status"],
                    bot_request["Id"],
                )
            # BOT failed to automatically create a proper ticket, skip it
            continue

        admin_requests.append(admin_request)

    return admin_requests


@ttl_cache(maxsize=1, ttl=constants.HOUR_SECONDS)
def get_oebs_projects():
    projects = OrderedDict()
    scheme = List(
        ApiDictScheme(
            {
                "group_id": StringToInteger(),
                "ru_description": String(),
                "us_description": String(),
                "s3_group_id": StringToInteger(nullable=True),
                "s2_group_id": StringToInteger(nullable=True),
                "s1_group_id": StringToInteger(nullable=True),
                "planner_id": StringToInteger(nullable=False),
            }
        )
    )
    for project in gevent_idle_iter(
        json_request("/api/view.php", scheme=scheme, params={"name": "view_oebs_services", "format": "json"})
    ):
        project_id = project["group_id"]
        parent_project_id = (
            list(filter(None, (project[level] for level in ("s3_group_id", "s2_group_id", "s1_group_id")))) + [None]
        )[1]

        projects[project_id] = {
            "project_id": project_id,
            "ru_description": project["ru_description"],
            "en_description": project["us_description"],
            "parent_project_id": parent_project_id if parent_project_id != project_id else None,
            "planner_id": project["planner_id"],
            "subprojects": [],
        }

    for project_id, project in projects.items():
        parent_project_id = project["parent_project_id"]
        if parent_project_id is not None:
            projects[parent_project_id]["subprojects"].append(project)

    return projects


def is_valid_oebs_project(project_id):
    return project_id in get_oebs_projects()


def is_allowed_bot_project(project_id) -> (bool, str):
    allowed_bot_project_ids = config.get_value("bot.allowed_bot_project_ids", [])
    if allowed_bot_project_ids and project_id not in allowed_bot_project_ids:
        return False, (
            "Bot project id '{}' not allowed. List of allowed Bot project IDs: '{}.'",
            project_id,
            sorted(allowed_bot_project_ids),
        )
    return True, None


def get_oebs_projects_tree():
    return [project for project in get_oebs_projects().values() if project["parent_project_id"] is None]


def get_bot_project_id_by_planner_id(planner_id):
    for bot_project_id, project in get_oebs_projects().items():
        if project["planner_id"] == planner_id:
            return bot_project_id
    raise PlannerIdNotFound(planner_id)


def _api_url(path, host=""):
    if not host:
        host = config.get_value("bot.host")
    return "https://" + host + path


def _view_iterator(name, row_type, timeout=constants.NETWORK_TIMEOUT):
    """Iterate over a bot view, yielding rows of data.
    :param name: view name
    :param row_type: data structure that should represent a row, preferably a namedtuple
    :param timeout: request timeout
    :return: tuple of (response_cache_time, iterator)
    """

    path = "/api/view.php"
    params = {"name": name}
    fields_number = len(row_type._fields)
    make_row = row_type._make

    response = raw_request(path, params=params, stream=True, timeout=timeout)

    def iterator():
        try:
            for row in iter_csv_response(response, fields_number, ignore_extra_fields=True):
                yield make_row(_get_value(column) for column in row)
        except Exception as e:
            raise BotInternalError("{}".format(e), name=name, row_type=row_type)

    try:
        response_cache_time = int(response.headers.get('X-BOT-Created-Time', timestamp()))
    except ValueError:
        log.error(
            "Got invalid response cache time from BOT: %s (expected integer) for %s request.",
            response.headers.get('X-BOT-Created-Time'),
            path,
        )
        response_cache_time = timestamp()

    return _RequestIterator(response_cache_time, iterator())


def _get_value(value):
    return None if value == "-" else value


def _get_hardware_location(*args):
    if all(arg == "" for arg in args):
        raise ValueError

    return HardwareLocation(*args)


def _log_request(func):
    @six.wraps(func)
    def decorated(path, **kwargs):
        method = kwargs.get("method", "GET")
        stream = kwargs.get("stream")
        drop_params = ["scheme", "error_scheme"]
        try:
            bot_request_log.debug(
                "BOT request: %s %s %s%r",
                method,
                _api_url(path),
                "(stream) " if stream else "",
                {k: v for k, v in kwargs.items() if k not in drop_params},
            )
            response = func(path, **kwargs)
            if kwargs.get("with_response"):
                bot_request_log.debug(
                    "BOT response: %s %s", response[0].status_code, "" if stream else response[0].text.strip()
                )
            elif isinstance(response, Response):
                bot_request_log.debug(
                    "BOT response: %s %s", response.status_code, "" if stream else response.text.strip()
                )
            else:
                bot_request_log.debug("BOT response: %s", response)
        except RequestException as e:
            # Use e.message here: unicode(RequestException("some non-ascii text")) raises UnicodeEncodeError
            bot_request_log.exception("BOT request error: %s", str(e))
            if e.request:
                bot_request_log.debug(
                    "BOT error %s: %s, response body: %s",
                    e.request.status_code,
                    e.request.reason,
                    e.request.text.strip(),
                )
            raise

        return response

    return decorated


@_log_request
def raw_request(path, stream=False, authenticate=False, **kwargs):
    with _request_context(path, stream=stream, authenticate=authenticate, **kwargs) as request:
        response = walle.clients.utils.request(**request)

    return response if stream else response.text.strip()


@_log_request
def json_request(path, data=None, authenticate=False, tvm=False, host="", **kwargs):
    with _request_context(path, form_data=data, authenticate=authenticate, use_tvm=tvm, host=host, **kwargs) as request:
        return walle.clients.utils.json_request(**request)


@contextmanager
def _request_context(path, method="GET", authenticate=False, use_tvm=False, host="", **kwargs):
    if authenticate:
        headers = kwargs.setdefault("headers", {})
        if not use_tvm:
            headers.setdefault("Authorization", "OAuth " + config.get_value("bot.access_token", default=""))
        else:
            tvm_alias = config.get_value("bot.hwr.tvm_alias")
            headers.setdefault("X-Ya-Service-Ticket", tvm.get_ticket_for_service(tvm_alias))

    kwargs["service"] = "bot"
    kwargs["method"] = method
    url = kwargs["url"] = _api_url(path, host=host)

    try:
        yield kwargs
    except RequestException as e:
        kwargs.pop("headers", None)
        raise BotInternalError("{}: {}".format(url, str(e)), **kwargs)


def get_planner_id_by_bot_project_id(bot_project_id):
    return get_oebs_projects()[bot_project_id]["planner_id"]
