from requests.exceptions import RequestException

from object_validator import List, String, DictScheme
from sepelib.core import config
from walle.clients import tvm
from walle.clients.utils import json_request
from walle.errors import RecoverableError


class CalendarInternalError(RecoverableError):
    def __init__(self, message, **kwargs):
        super().__init__("Error in communication with Calendar: " + message, **kwargs)


def _api_request(method: str, path: str, params=None, scheme=None):
    url = "https://{host}/{path}".format(host=config.get_value("calendar.host"), path=path)

    try:
        tvm_alias = config.get_value("calendar.tvm_aliase")
        headers = {"X-Ya-Service-Ticket": tvm.get_ticket_for_service(tvm_alias)}
        return json_request("calendar", method, url, params=params, scheme=scheme, headers=headers)
    except RequestException as e:
        raise CalendarInternalError("{}".format(e), path=path, params=params, scheme=scheme)


def get_holidays(date_from: str, date_to: str, for_countries: list[str], out_mode: str = "holidays"):
    """
    Reference: https://wiki.yandex-team.ru/calendar/api/new-web/#get-holidays

    'date_from' in format 'year-month-day'
    'date_to' in format 'year-month-day'
    'for_countries' in format ['rus', 'ukr']
    """
    item_scheme = DictScheme({"date": String(), "type": String()}, ignore_unknown=True)
    scheme = DictScheme({"holidays": List(item_scheme)}, ignore_unknown=True)
    params = {"from": date_from, "to": date_to, "for": ",".join(for_countries), "outMode": out_mode}

    return _api_request("GET", path="get-holidays", params=params, scheme=scheme)
