"""
Client for CAuth push API
Docs: https://wiki.yandex-team.ru/Intranet/cauth/server-api/
"""
import cgi
import enum
import json
import logging

import six
from requests import RequestException

import object_validator
from object_validator import DictScheme, String
from sepelib.core import config
from walle.clients.utils import request, check_certs_exist
from walle.errors import RecoverableError

log = logging.getLogger(__name__)


class CAuthError(RecoverableError):
    def __init__(self, message, *args, **kwargs):
        super().__init__("Error in communication with CAuth: {}".format(six.ensure_text(message)), *args, **kwargs)


class CauthSource:
    WALLE = "walle"
    YP = "yp"
    IDM = "idm"
    YCLOUD = "ycloud"
    HD = "hd"
    BOT = "bot"
    CONDUCTOR = "conductor"
    IDM_CONDUCTOR = "idm-conductor"
    CMS = "cms"
    IDM_CMS = "idm-cms"

    ALL = [WALLE, YP, IDM, YCLOUD, HD, BOT, CONDUCTOR, IDM_CONDUCTOR, CMS, IDM_CMS]


class CauthFlowType:
    CLASSIC = "classic"
    BACKEND_SOURCES = "backend_sources"

    ALL = [CLASSIC, BACKEND_SOURCES]


class CAuthKeySources(str, enum.Enum):
    STAFF = "staff"
    SECURE = "secure"
    INSECURE = "insecure"


def add_host_via_push(fqdn, project, cauth_settings=None):
    scheme = DictScheme(
        {
            "status": String(choices=["added"]),
            "srv": String(choices=[fqdn]),
        }
    )

    grp = "walle.{}".format(project)
    data = {"srv": fqdn, "grp": grp}
    if cauth_settings:
        data.update(cauth_settings)
    _api_post_request("/add_server/", data, scheme=scheme)


def remove_host_via_push(fqdn):
    scheme = DictScheme(
        {
            "status": String(choices=["removed"]),
            "srv": String(choices=[fqdn]),
        }
    )

    _api_post_request("/remove_server/", data={"srv": fqdn}, scheme=scheme)


def get_certificate():
    cert = config.get_value("cauth.cert_path")
    key = config.get_value("cauth.key_path")
    check_certs_exist(cert, key)
    return cert, key


def _api_post_request(path, data, scheme=None):
    url = "https://" + config.get_value("cauth.api_url") + path

    try:
        response = request("cauth", method="POST", url=url, data=data, cert=get_certificate(), check_status=False)
    except RequestException as e:
        raise CAuthError("{}", e, url=url)

    _raise_if_error(response)

    json_res = json.loads(six.ensure_text(response.content))
    if scheme is not None:
        _validate_response_content(json_res, scheme)

    return json_res


def _raise_if_error(response):
    content_type = _get_response_content_type(response)
    if content_type == "text/plain":  # cauth returns errors in plain text
        raise CAuthError(response.content, response=response)
    elif content_type != "application/json":
        raise CAuthError("Server returned wrong content type {}".format(content_type), response=response)


def _get_response_content_type(response):
    content_type, _ = cgi.parse_header(response.headers.get("Content-Type", ""))
    return content_type


def _validate_response_content(content, scheme):
    try:
        object_validator.validate("result", content, scheme)
    except object_validator.ValidationError as e:
        log.error("Got an invalid JSON response: %r", content)
        raise CAuthError("The server returned an invalid JSON response: {}".format(e), content=content, scheme=scheme)
