"""Certificator API client.

Production: https://crt.yandex-team.ru/api/certificate/
Development: https://crt.test.yandex-team.ru/api/certificate/

API requires "rc-server certificates requester" role for "Сертификатор" on prod and for "Crt (test)" on development:
Production: https://idm.yandex-team.ru/#rf=1,rf-role=4oCKnilC#certificator/group-4;;;,rf-expanded=4oCKnilC
Development: https://idm.test.yandex-team.ru/#rf=1,rf-role=jEXqJ8g6#crt-test/group-4;;;,rf-expanded=jEXqJ8g6
"""

import http.client
import logging
from contextlib import contextmanager

from requests import RequestException

import walle.clients.utils
import walle.util.misc
from object_validator import String
from sepelib.core import config
from walle.errors import RecoverableError
from walle.util.validation import ApiDictScheme

log = logging.getLogger(__name__)


class CertificatorInternalError(RecoverableError):
    def __init__(self, message, *args, **kwargs):
        super().__init__("Error in communication with Certificator: " + message, *args, **kwargs)


class CertificatorPersistentError(RecoverableError):
    pass


class CertificatorClientError(CertificatorPersistentError):
    def __init__(self, message, *args, **kwargs):
        super().__init__("Error in communication with Certificator: " + message, *args, **kwargs)


class CertificatorPathNotFoundError(CertificatorClientError):
    pass


CERTIFICATE_STATUS_ISSUED = "issued"
CERTIFICATE_STATUS_REQUESTED = "requested"
CERTIFICATE_STATUS_REVOKED = "revoked"
ALL_CERTIFICATE_STATUSES = [CERTIFICATE_STATUS_ISSUED, CERTIFICATE_STATUS_REQUESTED, CERTIFICATE_STATUS_REVOKED]

_CERTIFICATE_RESPONSE_SCHEME = ApiDictScheme(
    {
        "url": String(),
        "ca_name": String(),
        "common_name": String(),
        # url for the .pfx form of the certificate. None if certificate is not issued yet.
        "download": String(optional=True),
        # generic download url for the certificate (used to construct a url for a .pem form).
        "download2": String(optional=True),
        "status": String(choices=ALL_CERTIFICATE_STATUSES),
        "type": String(),
    },
    drop_none=True,
)


class Certificate:
    def __init__(self, url, status, download=None):
        self.url = url
        self.status = status
        self.download = download

    @classmethod
    def from_dict(cls, dictionary):
        return cls(url=dictionary["url"], status=dictionary["status"], download=dictionary.get("download"))

    def to_dict(self):
        return {
            "url": self.url,
            "status": self.status,
            "download": self.download,
        }

    @property
    def requested(self):
        return self.status == CERTIFICATE_STATUS_REQUESTED

    @property
    def issued(self):
        return self.status == CERTIFICATE_STATUS_ISSUED

    @property
    def revoked(self):
        return self.status == CERTIFICATE_STATUS_REVOKED

    @classmethod
    def request(cls, fqdn, **kwargs):
        certificate_params = config.get_value("certificator.params")
        certificate_params["common_name"] = fqdn

        result = json_request(_api_url("/certificate/"), "POST", data=certificate_params, **kwargs)

        if result["status"] not in (CERTIFICATE_STATUS_ISSUED, CERTIFICATE_STATUS_REQUESTED):
            raise CertificatorPersistentError(
                "Certificator failed to issue certificate: {} for request {}. {}",
                result["status"],
                certificate_params,
                result.get("url", "No url provided."),
                params=certificate_params,
            )

        return cls(url=result["url"], status=result["status"], download=_get_download_url(result))

    @classmethod
    def get_info(cls, url, **kwargs):
        result = json_request(url, **kwargs)
        return Certificate(url=result["url"], status=result["status"], download=_get_download_url(result))

    def refresh_info(self, **kwargs):
        return self.get_info(self.url, **kwargs)

    def fetch(self, force_primary=False, **kwargs):
        if self.download is None:
            raise CertificatorInternalError(
                "Attempting to fetch a certificate without a download link in {} status", self.status, **kwargs
            )

        if force_primary:
            kwargs.setdefault("headers", {})["X-Replicated-State"] = "master"

        return raw_request(self.download, **kwargs)

    def revoke(self, **kwargs):
        result = json_request(self.url, "DELETE", **kwargs)
        if result["status"] != CERTIFICATE_STATUS_REVOKED:
            raise CertificatorPersistentError(
                "Certificator didn't revoke the certificate: {}. {}", result["status"], self.url
            )

        return Certificate(url=result["url"], status=result["status"], download=_get_download_url(result))

    def __eq__(self, other):
        if not isinstance(other, self.__class__):
            return False
        return self.to_dict() == other.to_dict()


def json_request(url, method="GET", scheme=None, **kwargs):
    """Make JSON request to certificator's API by given URL.
    Return result JSON."""

    if scheme is None:
        scheme = _CERTIFICATE_RESPONSE_SCHEME

    if method == "POST":
        kwargs["allow_errors"] = (http.client.CREATED,)
        kwargs["error_scheme"] = scheme

    kwargs.update(config.get_value("certificator.request_args", {}))  # verify, timeout, etc.
    headers = {"Authorization": "OAuth " + config.get_value("certificator.access_token")}

    with _certificator_error_handler(url):
        return walle.clients.utils.json_request("certificator", method, url, scheme=scheme, headers=headers, **kwargs)


def raw_request(url, method="GET", **kwargs):
    """Make plain HTTP request to certificator's API by given URL.
    Return response body."""

    request_kwargs = dict(config.get_value("certificator.request_args", {}), **kwargs)

    headers = request_kwargs.setdefault("headers", {})
    headers.setdefault("Authorization", "OAuth " + config.get_value("certificator.access_token"))

    with _certificator_error_handler(url):
        return walle.clients.utils.request("certificator", method, url, as_text=True, **request_kwargs)


@contextmanager
def _certificator_error_handler(url):
    try:
        yield
    except RequestException as e:
        # Use e.message here: unicode(RequestException("some non-ascii text")) raises UnicodeEncodeError
        if e.response is not None:
            exception_args = "Failed to fetch {}: {} {}", url, str(e), e.response.text
            if 404 == e.response.status_code:  # not exactly an error
                raise CertificatorPathNotFoundError(*exception_args, url=url, response=e.response)
            if 400 <= e.response.status_code < 500:  # client error:
                raise CertificatorClientError(*exception_args, url=url, response=e.response)
            if 500 <= e.response.status_code:  # server error, retry may help
                raise CertificatorInternalError(*exception_args, url=url, response=e.response)

        # just something unexpected
        raise


def _api_url(path):
    return "https://{}/api/{}".format(config.get_value("certificator.host"), path.lstrip("/"))


def _get_download_url(certificate_data):
    if certificate_data.get("download2"):
        return certificate_data["download2"] + ".pem"
    else:
        return None
