"""Provides a client to Yandex host deployment system."""

import http.client
import logging
import time
from abc import ABC, abstractmethod
from base64 import standard_b64encode
from copy import deepcopy
from typing import Optional

import dateutil.parser
import requests
import simplejson as json

import object_validator
import walle.clients.utils
import walle.util.misc
from object_validator import DictScheme, List, String, Integer
from sepelib.core import config, constants
from sepelib.core.exceptions import Error
from walle import constants as walle_constants
from walle.errors import RecoverableError
from walle.util import db_cache
from walle.util.misc import drop_none
from walle.util.net import format_mac
from walle.util.validation import FlatableList

log = logging.getLogger(__name__)

DEPLOY_CONFIG_EXTERNAL = "external"
"""Special kind of deploy config, allows user to provide configuration  ad-hoc. Wall-E does not support it."""


class DeployClientError(RecoverableError):
    pass


class DeployInternalError(DeployClientError):
    pass


class DeployPersistentError(DeployClientError):
    pass


class HostDoesntExistError(DeployPersistentError):
    pass


class _DeployApiError(DeployInternalError):
    def __init__(self, api_error, *args, **kwargs):
        super().__init__(*args, **kwargs)
        self.error = api_error


class _ApiResult(object_validator.Object):
    def validate(self, obj):
        if type(obj) not in (list, dict):
            raise object_validator.InvalidTypeError(obj)

        return obj


STATUS_PENDING = "pending"
"""Host deployment has been requested but hasn't been initiated yet.

If host stalls in this state it may be due to the following reasons:
* Redeploy command has been issued with an invalid MAC address.
* IPMI reset command returned success, but hasn't actually reset the host.
"""

STATUS_BOOTING = "booting"
"""Host is booting from PXE."""

STATUS_PREPARING = "preparing"
"""Host is preparing for installation."""

STATUS_DEPLOYING = "deploying"
"""Host is deploying now."""

STATUS_COMPLETED = "completed"
"""Host deployment completed."""

STATUS_FAILED = "failed"
"""Host deployment failed."""

STATUS_DISK_FAILED = "disk_failed"
"""Host deployment failed (bad disk)."""

STATUS_CONFIG_INAPPROPRIATE = "config_inappropriate"
"""Deploy config is invalid"""

STATUS_INVALID = "invalid"
"""Deployment process got an unexpected state."""

STATUS_RETRY = "retry"
"""Deployment process must be repeated."""


STATUSES = [
    STATUS_BOOTING,
    STATUS_PREPARING,
    STATUS_DEPLOYING,
    STATUS_COMPLETED,
    STATUS_FAILED,
    STATUS_DISK_FAILED,
    STATUS_CONFIG_INAPPROPRIATE,
    STATUS_INVALID,
    STATUS_RETRY,
]
"""All available statuses."""


COMPLETED_STATUS_TIME_ACCURACY = constants.MINUTE_SECONDS
"""
Completed status is actually set before host rebooting. Consider that one minute is enough for reboot command to
complete.
"""


class DeployProvider(ABC):
    @abstractmethod
    def get_host(self) -> str:
        raise NotImplementedError

    @abstractmethod
    def get_headers(self) -> dict[str, str]:
        raise NotImplementedError


class YandexInternalDeployProvider(DeployProvider):
    def __init__(self, host: str, token: str):
        self._host = host
        self._token = token

    def get_host(self) -> str:
        return self._host

    def get_headers(self) -> dict[str, str]:
        return {"Authorization": "OAuth {}".format(self._token)}


class YandexBoxDeployProvider(DeployProvider):
    def __init__(self, host: str, login: str, password: str):
        self._host = host
        self._login = login
        self._password = password

    def get_host(self) -> str:
        return self._host

    def get_headers(self) -> dict[str, str]:
        cred = "{}:{}".format(self._login, self._password)
        return {"Authorization": "Basic {}".format(standard_b64encode(cred.encode()).decode())}


class DeployClient:
    """A client to Yandex host deployment system.

    It fully abstracts us from a knowledge about LUI/FUI API.
    """

    def __init__(self, provider: DeployProvider):
        self.__client = LuiClient(provider)

    def get_deploy_status(self, hostname):
        """Returns current host deployment status.

        :raises HostDoesntExistError
        """

        lui_info = self.__client.get_lui_info(hostname)
        action, status, modify_time = lui_info["action"], lui_info["status"], lui_info.get("time_modified")

        if action == "BOOT_FROM_HD" and status == "INSTALLED_AND_REBOOTED":
            client_status = STATUS_COMPLETED
        elif action == "INSTALL" and status == "PENDING":
            client_status = STATUS_PENDING
        # Notice: FreeBSD always gets READY_TO_INSTALL status omitting PXE_CONFIG_SENT status
        elif action == "INSTALL" and status == "PXE_CONFIG_SENT":
            client_status = STATUS_BOOTING
        # READY_TO_INSTALL is a state in which every host stalls for 0 - 300 seconds
        # (actual time is determined randomly)
        elif action == "INSTALL" and status == "READY_TO_INSTALL":
            client_status = STATUS_PREPARING
        elif action == "INSTALL" and status == "INSTALLING":
            client_status = STATUS_DEPLOYING
        elif action == "INSTALL" and status == "INSTALLATION_FAILED":
            client_status = STATUS_FAILED
        elif action == "INSTALL" and status == "DISK_FAILED":
            client_status = STATUS_DISK_FAILED
        elif action == "INSTALL" and status == "CONFIG_INAPPROPRIATE":
            client_status = STATUS_CONFIG_INAPPROPRIATE
        elif action == "WAIT" and status == "READY_TO_INSTALL":
            client_status = STATUS_RETRY
        else:
            client_status = STATUS_INVALID

        description = action + "/" + status

        if modify_time is not None:
            try:
                if not modify_time:
                    raise ValueError()

                modify_time = time.mktime(dateutil.parser.parse(modify_time).timetuple())
            except ValueError:
                raise DeployInternalError(
                    "Got an invalid {} host modify time from LUI: {}.", hostname, modify_time, lui_info=lui_info
                )

        return drop_none(
            {
                "macs": [format_mac(mac) for mac in lui_info["mac"]],
                "config": lui_info["config"],
                "lui_status": status,
                "status": client_status,
                "description": description,
                "fail_count": lui_info.get("cnt_failed", 0),
                "modify_time": modify_time,
            }
        )

    def schedule_redeploy(
        self, host, mac, config_name, private_data=None, host_networks=None, project_id=None, config_content=None
    ):
        """Schedules host deployment after a reboot."""

        self.__client.schedule_redeploy(host, mac, config_name, private_data, host_networks, project_id, config_content)

    def setup(self, host, mac, config_name):
        self.__client.setup_host(host, mac, config_name)

    def deactivate(self, host):
        """Deactivates the host."""

        self.__client.deactivate(host)

    def remove(self, host):
        """Removes a host from the deployment system."""

        self.__client.remove(host)


class LuiClient:
    """LUI/FUI(DHCP) API client."""

    __max_tries = 1
    __try_interval = 5

    _STATUS_PENDING = "PENDING"
    _STATUS_UNKNOWN = "UNKNOWN"

    _ACTION_INSTALL = "INSTALL"
    _ACTION_BOOT_FROM_HD = "BOOT_FROM_HD"

    def __init__(self, provider: DeployProvider, timeout=None):
        self._provider = provider
        self.__timeout = constants.NETWORK_TIMEOUT if timeout is None else timeout

    def get_deploy_log(self, hostname, tail_bytes=None):
        if tail_bytes == 0:
            return ""

        headers = self._provider.get_headers()

        log_url = "https://{api_hostname}/api/proxy/files/log/{hostname}.log".format(
            api_hostname=self._provider.get_host(), hostname=hostname
        )

        if tail_bytes is not None:
            response = requests.head(log_url, timeout=constants.NETWORK_TIMEOUT, headers=self._provider.get_headers())
            if response.status_code == http.client.NOT_FOUND:
                return ""

            if response.status_code != http.client.OK:
                raise requests.RequestException(response.reason)

            if response.headers.get("Accept-Ranges") == "bytes" and "Content-Length" in response.headers:
                try:
                    content_length = int(response.headers["Content-Length"])
                except ValueError:
                    raise Error("Invalid Content-Length ({}).", response.headers["Content-Length"])

                if content_length == 0:
                    return ""

                headers["Range"] = "bytes={}-".format(max(0, content_length - tail_bytes))
            else:
                log.error(
                    "Failed to get log of %s from lui provisioner using HTTP range: "
                    "HTTP range is not supported by the server.",
                    hostname,
                )

        response = walle.clients.utils.request(
            "deploy-logs", "GET", log_url, headers=headers, stream=True, check_status=False
        )

        if response.status_code in (http.client.OK, http.client.PARTIAL_CONTENT):
            return response.iter_content(chunk_size=constants.NETWORK_BUFSIZE)
        elif response.status_code == http.client.NOT_FOUND:
            return ""
        else:
            raise requests.RequestException(response.reason)

    def get_deploy_configs(self):
        all_config_names = set(self.lui_api("listConfigNames", {}, scheme=List(String())))
        all_config_names.discard(DEPLOY_CONFIG_EXTERNAL)  # wall-e does not support external configs yet.

        return list(all_config_names)

    def get_deploy_config(self, config_name):
        # pick 'config' field. data, name, metadata, filename fields are ignored
        return self.lui_api("config_get", {"name": config_name})["config"]

    def set_host_config(self, hostname, config_name):
        # Force specifying of OS: if we automatically determine OS every time on method call we cause errors for Linux
        # hosts when DHCP API is not available.

        response = self.lui_api(
            "server_update",
            {
                "query": {"name": hostname},
                "update": {"config": config_name},
            },
        )

        if not response["updatedExisting"]:
            raise HostDoesntExistError(
                "Failed to update config for host '{}': the host doesn't exist in LUI database", hostname
            )

    def get_lui_info(self, hostname):
        """Returns host info from LUI."""

        try:
            info = self.lui_api(
                "server_get",
                {"query": {"name": hostname}},
                scheme=DictScheme(
                    {
                        "mac": FlatableList(String()),
                        "config": String(),
                        "action": String(),
                        "status": String(),
                        "cnt_failed": Integer(optional=True),
                        "time_modified": String(optional=True),
                    },
                    ignore_unknown=True,
                ),
            )
        except _DeployApiError as e:
            if "Can't find server using query" in e.error:
                raise HostDoesntExistError(str(e), hostname=hostname)
            raise

        return info

    def schedule_redeploy(
        self, hostname, mac, config_name, private_data=None, host_networks=None, project_id=None, config_content=None
    ):
        """Schedules host deployment after reboot."""

        return self.setup_host(
            hostname,
            mac,
            config_name,
            action=self._ACTION_INSTALL,
            status=self._STATUS_PENDING,
            private_data=private_data,
            allow_network=host_networks,
            project_id=project_id,
            config_content=config_content,
        )

    def setup_host(
        self,
        hostname,
        mac,
        config_name,
        action=_ACTION_BOOT_FROM_HD,
        status=_STATUS_UNKNOWN,
        config_content=None,
        **kwargs
    ):
        return self.lui_api(
            "setup",
            drop_none(
                dict(
                    name=hostname,
                    mac=mac,
                    config=config_name,
                    action=action,
                    status=status,
                    reboot_manually=True,
                    config_content=config_content,
                    **kwargs
                )
            ),
            DictScheme(
                {
                    "name": String(choices=[hostname]),
                    "mac": FlatableList(String(choices=mac)),
                    "config": String(choices=[config_name]),
                    "status": String(choices=[status]),
                    "action": String(choices=[action]),
                },
                ignore_unknown=True,
            ),
        )

    def deactivate(self, hostname):
        """Deactivates the host."""

        self.lui_api(
            "server_update",
            {
                "query": {"name": hostname},
                "update": {"action": self._ACTION_BOOT_FROM_HD, "status": self._STATUS_UNKNOWN},
            },
        )

    def remove(self, hostname):
        """Removes a host from LUI."""

        self.lui_api("server_delete", {"query": {"name": hostname}})

    def lui_api(self, action, params, scheme=None):
        url = "https://{host}/api".format(host=self._provider.get_host())
        headers = {"Content-Type": "application/json", **self._provider.get_headers()}
        data = json.dumps({"method": action, "args": [], "kwargs": deepcopy(params)})
        params = _hide_private_data_from_request_params(params)
        try:
            response = self.__post(url, data, headers=headers)
            reply = response.json()
        except (requests.RequestException, ValueError) as e:
            raise DeployInternalError("LUI API: {}({}) failed with error: {}.", action, params, e)

        try:
            base_scheme = DictScheme({"status": String(choices=["success"], optional=True)}, ignore_unknown=True)
            reply = object_validator.validate("response", reply, DictScheme({"result": scheme or base_scheme}))
            return reply["result"]

        except object_validator.ValidationError as e:
            self.__raise_error(action, params, reply, e)

    def __raise_error(self, action, params, reply, error):
        try:
            host_error_scheme = DictScheme(
                {
                    "result": DictScheme(
                        {"status": String(choices=["error"]), "message": String(), "err_code": String()},
                        ignore_unknown=True,
                    )
                }
            )
            reply = object_validator.validate("response", reply, host_error_scheme)

            message = reply["result"]["message"].rstrip(".")
            raise _DeployApiError(
                message,
                "LUI API: {}({}) failed with error: {}.",
                action,
                params,
                message,
                action=action,
                params=params,
                error=message,
            )

        except object_validator.ValidationError:
            pass

        try:
            api_error_scheme = DictScheme({"error": DictScheme({"msg": String()}, ignore_unknown=True, optional=True)})
            reply = object_validator.validate("response", reply, api_error_scheme)
            message = reply["error"].get("msg").rstrip(".")

            raise _DeployApiError(
                message,
                "LUI API: {}({}) failed with error: {}.",
                action,
                params,
                message,
                action=action,
                params=params,
                error=message,
            )

        except object_validator.ValidationError:
            pass

        # It was not a valid error message. Report invalid answer.
        log.info("LUI server returned an unexpected response %s", reply)
        raise DeployInternalError("LUI API: {}({}) returned an unexpected response: {}.", action, params, error)

    def __post(self, url, data=None, headers=None):
        try_count = 1

        while True:
            try:
                response = walle.clients.utils.request(
                    "lui", "POST", url, data=data, headers=headers, check_status=False, timeout=self.__timeout
                )

                if response.status_code in (
                    http.client.INTERNAL_SERVER_ERROR,
                    http.client.BAD_GATEWAY,
                    http.client.SERVICE_UNAVAILABLE,
                    http.client.GATEWAY_TIMEOUT,
                ):
                    response.raise_for_status()
            except requests.RequestException as e:
                log.warning("LUI/DHCP API: POST %s failed: %s.", url, e)

                if try_count < self.__max_tries:
                    time.sleep(self.__try_interval)
                    try_count += 1
                else:
                    raise
            else:
                response.raise_for_status()
                return response


def get_client(provider: DeployProvider) -> DeployClient:
    """Returns a client to Yandex host deployment system."""

    return DeployClient(provider)


def get_lui_client(provider: DeployProvider, timeout: Optional[int] = None) -> LuiClient:
    """Returns a client to Yandex host deployment system."""

    return LuiClient(provider, timeout=timeout)


def get_deploy_configs(provider: DeployProvider):
    @db_cache.cached("linux_deploy_configs-of-{}".format(provider.get_host()), 5 * constants.MINUTE_SECONDS)
    def cache_getter():
        log.debug("Updating LUI deploy config names cache for {}...".format(provider.get_host()))
        return get_lui_client(provider, timeout=walle_constants.LOW_LATENCY_NETWORK_TIMEOUT).get_deploy_configs()

    return cache_getter()


def get_deploy_config(provider: DeployProvider, config_name):
    return get_lui_client(provider, timeout=walle_constants.LOW_LATENCY_NETWORK_TIMEOUT).get_deploy_config(config_name)


def get_yandex_internal_provider() -> DeployProvider:
    return YandexInternalDeployProvider(
        config.get_value("lui.host", "localhost"), config.get_value("lui.access_token", "")
    )


def get_yandex_box_provider_if_exists(box_id: str) -> Optional[DeployProvider]:
    params = config.get_value("lui.boxes.{}".format(box_id), None)
    if params is None:
        return None
    return YandexBoxDeployProvider(
        host=params.get("host", "localhost"), login=params.get("login", ""), password=params.get("password", "")
    )


def get_deploy_provider(box_id: Optional[str] = None) -> DeployProvider:
    if box_id is not None:
        provider = get_yandex_box_provider_if_exists(box_id)
        if provider is not None:
            return provider
    return get_yandex_internal_provider()


def _hide_private_data_from_request_params(params):
    if "private_data" in params:
        for item in params["private_data"]:
            if "content" in item:
                item["content"] = "*" * 16

    return params
