import abc
import typing as tp

from walle.errors import RecoverableError
from .operations import DnsApiOperation


class DnsError(RecoverableError):
    pass


class DnsZoneNotFound(DnsError):
    def __init__(self, qname, exc, **kwargs):
        super().__init__("DNS zone {} not found: {}".format(repr(qname), exc), qname=qname, exc=exc, **kwargs)


class DnsMultipleEndpoints(DnsError):
    def __init__(self, qname, endpoints, **kwargs):
        super().__init__(
            "DNS record {} has multiple endpoints: {}.",
            qname,
            ", ".join(endpoints),
            qname=qname,
            endpoints=endpoints,
            **kwargs
        )


class DnsCommunicationError(DnsError):
    def __init__(self, exc, **kwargs):
        errors_messages = [
            error.get("message", "Can not parse error messages from API response")
            for error in exc.response.json().get("errors", {})
        ]
        super().__init__("DNS communication error: {}: {}".format(exc, ", ".join(errors_messages)), exc=exc, **kwargs)


class DnsValueError(DnsError):
    pass


class DnsInvalidZone(DnsError):
    def __init__(self, zone, **kwargs):
        super().__init__("Got invalid zone info: {}.".format(zone), zone=zone, **kwargs)


class DnsClientInterface(abc.ABC):
    @abc.abstractmethod
    def get_aaaa(self, hostname: str) -> tp.Optional[str]:
        pass

    @abc.abstractmethod
    def get_a(self, hostname: str) -> tp.Optional[str]:
        pass

    @abc.abstractmethod
    def get_ptr(self, ip_address: str) -> tp.Optional[str]:
        pass

    @abc.abstractmethod
    def apply_operations(self, operations: list[DnsApiOperation]):
        pass
