from sepelib.core.exceptions import LogicalError

from walle.constants import DEFAULT_DNS_TTL
from walle.util.misc import filter_dict_keys


class DnsApiOperation:
    __slots__ = ["operation", "type", "name", "data"]

    DNS_API_DELETE = "dns-api-delete"
    DNS_API_ADD = "dns-api-add"

    OPERATION_MAP = {
        DNS_API_DELETE: "delete",
        DNS_API_ADD: "add",
    }

    def __init__(self, operation, type, name, data):
        self.operation = operation
        self.type = type
        self.name = name
        self.data = data

    @classmethod
    def add(cls, type, name, data):
        return DnsOperationAdd(cls.DNS_API_ADD, type, name, data)

    @classmethod
    def delete(cls, type, name, data):
        return DnsOperationDelete(cls.DNS_API_DELETE, type, name, data)

    @classmethod
    def from_dict(cls, d):
        kwargs = filter_dict_keys(d, {"type", "name", "data"})

        if d["operation"] == cls.DNS_API_DELETE:
            return cls.delete(**kwargs)

        elif d["operation"] == cls.DNS_API_ADD:
            return cls.add(**kwargs)

        else:
            raise LogicalError()

    def to_slayer_api_primitive(self) -> dict:
        if self.type == "PTR":
            # dns-api-specific...
            name, data = self.data, self.name
        else:
            name, data = self.name, self.data

        return {
            "name": name,
            "type": self.type,
            "data": data,
            "operation": self.OPERATION_MAP[self.operation],
        }

    def to_dict(self):
        return {
            "operation": self.operation,
            "type": self.type,
            "name": self.name,
            "data": self.data,
        }

    def to_statbox(self):
        return {
            "dns_operation_name": self.operation,
            "dns_operation_record_type": self.type,
            "dns_operation_record_name": self.name,
            "dns_operation_record_data": self.data,
        }

    def description(self):
        return "{operation} {name} -> {value} {type} record".format(
            operation=self.OPERATION_MAP[self.operation], name=self.name, value=self.data, type=self.type
        )

    def __eq__(self, other):
        if not isinstance(other, type(self)):
            return False
        return self.to_dict() == other.to_dict()

    def __repr__(self):
        return "{}({})".format(type(self).__name__, repr(self.to_dict()))


class DnsOperationAdd(DnsApiOperation):
    def to_slayer_api_primitive(self):
        return dict(super().to_slayer_api_primitive(), ttl=DEFAULT_DNS_TTL)


class DnsOperationDelete(DnsApiOperation):
    pass
