"""Einstellung client.

Attention: Inventory numbers look like integers, but in theory there might be inventory numbers with non-number
characters - they may be used for VMs and licenses, so it worth to be careful and handle such inventory numbers
properly.

v2 API reference - https://playground.eine.yandex-team.ru/playground/
"""

import http.client
import logging
import re
import typing as tp
from abc import ABC, abstractmethod
from base64 import standard_b64encode
from collections import namedtuple

import requests
from requests.exceptions import RequestException

import walle.clients.utils
from object_validator import List, String, DictScheme, Integer, Bool, Float
from sepelib.core import config, constants
from sepelib.core.exceptions import LogicalError
from walle import util
from walle.authorization import ISSUER_WALLE, ISSUER_ROBOT_WALLE
from walle.clients.utils import strip_api_error
from walle.constants import EINE_PROFILES_WITH_DC_SUPPORT, LOW_LATENCY_NETWORK_TIMEOUT, FLEXY_EINE_PROFILE
from walle.errors import RecoverableError
from walle.util import net, db_cache
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.misc import drop_none, format_time
from walle.util.validation import ApiDictScheme, GeventIdleList, StringToInteger

log = logging.getLogger(__name__)


class TicketType:
    STARTREK = "startrek"
    OTRS = "otrs"


class EineStageState:
    UNKNOWN = "unknown"
    PLANNED = "planned"
    RUNNING = "running"
    FAILED = "failed"
    SKIPPED = "skipped"
    COMPLETED = "completed"

    # NB: Order is important, this list is used as an index. Check the actual order here:
    # https://github.yandex-team.ru/einstellung/eine-server-web-flask/blob/master/app/api/computers/mapping.py#L11-L28
    ALL = [UNKNOWN, PLANNED, RUNNING, FAILED, SKIPPED, COMPLETED]


class EineComputerStatus:
    QUEUED = "queued"

    # The host is queued for profile or profiling now
    #
    # Attention: Host gets "queued" status on adding, so it not always means that host is profiling now - only when it
    # has "queued" status and profile is not null.
    ALL_IN_PROCESS = [QUEUED]

    FAILED = "failed"
    ACTION_NEEDED = "actioneed"
    IN_RESERV = "inreserv"

    # Failed statuses: failed -> actioneed -> inreserv
    ALL_FAILED = [FAILED, ACTION_NEEDED, IN_RESERV]

    COMPLETED = "completed"
    PRODUCTION = "production"

    # Completed statuses: completed -> production
    ALL_COMPLETED = [COMPLETED, PRODUCTION]

    ALL = ALL_IN_PROCESS + ALL_FAILED + ALL_COMPLETED


class EineProfileStatus:
    # The host is queued for profile or profiling now
    QUEUED = "queued"
    FAILED = "failed"
    COMPLETED = "completed"
    STOPPED = "stopped"

    # NB: Order is important, this list is used as an index. Check the actual order here:
    # https://github.yandex-team.ru/einstellung/eine-server-web-flask/blob/master/app/api/computers/mapping.py#L11-L28
    ALL = [QUEUED, FAILED, COMPLETED, STOPPED]


class EineProfileTags:
    # General tags
    MEMORY_RELAX = "memory-relax"
    SMART_RELAX = "smart-relax"
    SWP_UP = "swp-up"
    CHASSIS = "chassis"
    SKIP_VLAN_SWITCH = "skip-vlan-switch"
    SKIP_FW_FANTABLE = "skip-fw-fantable"

    # Flexy tags
    FULL_PROFILING = "full_profiling"
    BASIC_LOAD = "basic_load"
    ADVANCED_LOAD = "advanced_load"
    EXTRA_LOAD = "extra_load"
    DANGEROUS_LOAD = "dangerous_load"
    FIRMWARE_UPDATE = "firmware_update"
    DRIVES_WIPE = "drives_wipe"


class ProfileMode:
    """An abstraction over flexy profile tags mess which helps us to classify profile operations."""

    DEFAULT = "default"
    FIRMWARE_UPDATE = "firmware-update"
    BASIC_TEST = "basic-test"
    HIGHLOAD_TEST = "highload-test"
    EXTRA_HIGHLOAD_TEST = "extra-highload-test"
    DANGEROUS_HIGHLOAD_TEST = "dangerous-highload-test"
    DISK_RW_TEST = "disk-rw-test"
    SWP_UP = "swp-up"
    ALL = [
        FIRMWARE_UPDATE,
        BASIC_TEST,
        HIGHLOAD_TEST,
        EXTRA_HIGHLOAD_TEST,
        DANGEROUS_HIGHLOAD_TEST,
        DISK_RW_TEST,
        SWP_UP,
        DEFAULT,
    ]

    DEFAULT_ADD_TAGS = {EineProfileTags.FULL_PROFILING}
    DEFAULT_EXCLUDE_TAGS = {EineProfileTags.SWP_UP}

    FIRMWARE_UPDATE_ADD_TAGS = {
        EineProfileTags.MEMORY_RELAX,
        EineProfileTags.SMART_RELAX,
        EineProfileTags.FIRMWARE_UPDATE,
    }
    FIRMWARE_UPDATE_EXCLUDE_TAGS = {
        EineProfileTags.SWP_UP,
        EineProfileTags.CHASSIS,
        EineProfileTags.FULL_PROFILING,
        EineProfileTags.ADVANCED_LOAD,
        EineProfileTags.DANGEROUS_LOAD,
        EineProfileTags.EXTRA_LOAD,
    }

    BASIC_TEST_ADD_TAGS = {EineProfileTags.BASIC_LOAD}
    BASIC_TEST_EXCLUDE_TAGS = {EineProfileTags.MEMORY_RELAX, EineProfileTags.SWP_UP}

    HIGHLOAD_TEST_ADD_TAGS = {EineProfileTags.FULL_PROFILING, EineProfileTags.ADVANCED_LOAD}
    HIGHLOAD_TEST_EXCLUDE_TAGS = {EineProfileTags.MEMORY_RELAX, EineProfileTags.SWP_UP}

    EXTRA_HIGHLOAD_TEST_ADD_TAGS = HIGHLOAD_TEST_ADD_TAGS | {EineProfileTags.EXTRA_LOAD}

    DANGEROUS_HIGHLOAD_TEST_ADD_TAGS = EXTRA_HIGHLOAD_TEST_ADD_TAGS | {EineProfileTags.DANGEROUS_LOAD}
    DANGEROUS_HIGHLOAD_TEST_EXCLUDE_TAGS = {
        EineProfileTags.MEMORY_RELAX,
        EineProfileTags.SMART_RELAX,
        EineProfileTags.SWP_UP,
    }

    DISK_RW_TEST_ADD_TAGS = {EineProfileTags.FULL_PROFILING, EineProfileTags.DANGEROUS_LOAD}
    DISK_RW_TEST_EXCLUDE_TAGS = {EineProfileTags.MEMORY_RELAX, EineProfileTags.SMART_RELAX, EineProfileTags.SWP_UP}

    SWP_UP_ADD_TAGS = {EineProfileTags.SWP_UP}

    @classmethod
    def get_configuration(cls, profile, mode):
        if mode == cls.DEFAULT:
            modes = [cls.DEFAULT]
            add_tags = cls.DEFAULT_ADD_TAGS
            exclude_tags = cls.DEFAULT_EXCLUDE_TAGS

        elif mode == cls.FIRMWARE_UPDATE:
            modes = [cls.FIRMWARE_UPDATE]
            add_tags = cls.FIRMWARE_UPDATE_ADD_TAGS
            exclude_tags = cls.FIRMWARE_UPDATE_EXCLUDE_TAGS

        # Flexy testing blocks wouldn't work without basic_load (or full_profiling)
        # https://wiki.yandex-team.ru/users/farunda/cfnew/#opisanieblokovvprofile
        elif mode == cls.BASIC_TEST:
            modes = [cls.BASIC_TEST]
            add_tags = cls.BASIC_TEST_ADD_TAGS
            exclude_tags = cls.BASIC_TEST_EXCLUDE_TAGS

        elif mode == cls.HIGHLOAD_TEST:
            modes = [cls.HIGHLOAD_TEST]
            add_tags = cls.HIGHLOAD_TEST_ADD_TAGS
            exclude_tags = cls.HIGHLOAD_TEST_EXCLUDE_TAGS

        # TODO Add tags for extra_load here?
        elif mode == cls.EXTRA_HIGHLOAD_TEST:
            modes = [cls.EXTRA_HIGHLOAD_TEST]
            add_tags = cls.EXTRA_HIGHLOAD_TEST_ADD_TAGS
            exclude_tags = cls.HIGHLOAD_TEST_EXCLUDE_TAGS

        elif mode == cls.DANGEROUS_HIGHLOAD_TEST:
            modes = [cls.DANGEROUS_HIGHLOAD_TEST]
            add_tags = cls.DANGEROUS_HIGHLOAD_TEST_ADD_TAGS
            exclude_tags = cls.DANGEROUS_HIGHLOAD_TEST_EXCLUDE_TAGS

        elif mode == cls.DISK_RW_TEST:
            # disk-dangerous works only with highload-test, so include MODE_HIGHLOAD implicitly but without smart-relax
            modes = [cls.DISK_RW_TEST]
            if profile != FLEXY_EINE_PROFILE:
                modes.append(cls.HIGHLOAD_TEST)
            add_tags = cls.DISK_RW_TEST_ADD_TAGS
            exclude_tags = cls.DISK_RW_TEST_EXCLUDE_TAGS

        elif mode == cls.SWP_UP:
            modes = [cls.SWP_UP]
            add_tags = cls.SWP_UP_ADD_TAGS
            exclude_tags = set()

        else:
            raise LogicalError()

        return sorted(modes), add_tags, exclude_tags

    @classmethod
    def get_modes(cls, tags):
        modes = []

        # advanced_load and dangerous_load work only with basic_load (or full_profiling)
        if (
            EineProfileTags.FULL_PROFILING in tags
            and EineProfileTags.ADVANCED_LOAD in tags
            and EineProfileTags.MEMORY_RELAX not in tags
        ):
            modes.append(cls.HIGHLOAD_TEST)

        if (
            EineProfileTags.DANGEROUS_LOAD in tags
            and EineProfileTags.FULL_PROFILING in tags
            and EineProfileTags.SMART_RELAX not in tags
        ):
            modes.append(cls.DISK_RW_TEST)

        return sorted(modes) or None


EINE_SERVICE_PROFILES = {
    "shell",  # shell в машину
    "farunda-test",  # Профиль Павла Фарунды, в нём может быть что угодно, что поможет дебажить/чинить проблему
    "cdrom-boot",  # быстрый бут и обновление свичпорта в айне
    "fw-lan-cdrom",  # быстрая прошивка 10г карточки
    "iserj-test",  # профиль Сергея Ярмоленко, в нём может быть что угодно, что поможет дебажить/чинить проблему
    "nop",  # заглушка, нужная, чтобы прогнать машину в бете айне
    "toshiba-THNSNJ960PCSZ-fw-flash",  # прошить диски тошиба ссд 960, т.к. айне это не умеет
    "x9drw-fw-fix",  # даунгрейд прошивок х9дрв, обычно машины из ремонта такие приезжают
}
"""Profiles which DC engineers use to debug nodes failed on DC supported profiles."""

EineSwitchInfo = namedtuple("EineSwitchInfo", ["switch", "port", "timestamp"])
EineMacsInfo = namedtuple("EineMacsInfo", ["active", "timestamp"])

_FULL_SCAN_TIMEOUT = 60
_MAINTENANCE_RACKS = ["Diagnostics", "KARANTIN", "-"]


RE_BODY_MATCH = re.compile(r"^.*<body.*?>(.*)</body>.*$", re.MULTILINE | re.DOTALL)
RE_DROP_TAGS = re.compile(r"</?.*?>", re.MULTILINE | re.DOTALL)


class EineInternalError(RecoverableError):
    def __init__(self, message, *args, **kwargs):
        super().__init__("Error in communication with Einstellung: " + message, *args, **kwargs)


class EinePersistentError(RecoverableError):
    def __init__(self, message, *args, **kwargs):
        self.status_code = kwargs.pop("status_code", None)
        self._msg_args = (message,) + args

        super().__init__("Einstellung returned an error: " + message, *args, status_code=self.status_code, **kwargs)

    def host_does_not_exist(self, inv):
        return EineHostDoesNotExistError(inv, *self._msg_args)


class EineHostDoesNotExistError(EinePersistentError):
    def __init__(self, inventory, message, **kwargs):
        super().__init__("Can't find host #{}: " + message, inventory, inv=inventory, **kwargs)


class EineHostStatus:
    def __init__(self, info):
        """Get info as returned from /computers/inventory/{inv}/ and return an appropriate object."""
        self._info = info

    def inventory(self):
        """Return inventory number as an integer."""
        return self._info["inventory"]

    def in_use(self):
        return self._info["in_use"]

    def has_profile(self):
        return self._info.get("einstellung") is not None

    def in_process(self):
        return self.profile_status() == EineProfileStatus.QUEUED

    def profile(self):
        self._assert_has_profile()
        return self._info["einstellung"]["profile_name"]

    def profile_id(self):
        """Return einstellung._id which is more like host<->profile link id."""
        return self._info["einstellung"]["_id"]

    def profile_message(self):
        self._assert_has_profile()
        return self._info["einstellung"].get("message") or "<no message>."

    def profile_assigned_timestamp(self):
        self._assert_has_profile()
        return self._info["einstellung"]["assigned_at"]

    def profile_updated_timestamp(self):
        return self._info["einstellung"]["updated_at"]

    def profile_status(self):
        self._assert_has_profile()
        status_id = self._info["einstellung"]["status"]
        return EineProfileStatus.ALL[status_id]

    def get_stage_description(self):
        self._assert_has_profile()
        current_stage = self._get_current_stage()
        if current_stage is None:
            return "pending"

        status = EineStageState.ALL[current_stage["status"]]
        return "{}:{}".format(current_stage["stage"], status)

    def local_tags(self):
        self._assert_has_profile()
        return self._info["einstellung"].get("tags_local", [])

    def ticket_id(self):
        """Return either ticked id (in startrek form, like ITDC-00000) or None."""
        # Eine polls a list of failed profiles every 30 minutes and creates tickets for a group of failed hosts.
        # Ticket available as computer property. Eine polls a list of created tickets every 5 minutes and deletes
        # all closed tickets from computer properties.
        return self._info.get("props", {}).get("otrs_ticket", {}).get("value")

    def switch(self) -> tp.Optional[EineSwitchInfo]:
        """Return switch name from host properties or None if information can't be obtained."""
        if "switch" in self._info.get("props", {}):
            prop = self._info["props"]["switch"]
            switch_port = [name.strip() for name in prop["value"].split(" ")]

            if len(switch_port) == 2 and all(switch_port):
                return EineSwitchInfo(switch=switch_port[0], port=switch_port[1], timestamp=prop["timestamp"])

            # Don't report maintenance racks, it's totally pointless.
            elif self._info.get("rack") not in _MAINTENANCE_RACKS:
                log.error(
                    "Got an invalid switch/port for host #%s(%s) from Einstellung: %r.",
                    self._info.get("inventory"),
                    self._info.get("hostname"),
                    prop["value"],
                )

    def has_staled_location_info(self, host_network):
        eine_location = self.switch()

        if eine_location is None:
            return True

        if host_network is None or host_network.network_switch is None or host_network.network_port is None:
            return False

        return (
            eine_location.switch != host_network.network_switch or eine_location.port != host_network.network_port
        ) and eine_location.timestamp < host_network.network_timestamp

    def active_mac(self) -> tp.Optional[EineMacsInfo]:
        if "last_active_mac" in self._info.get("props", {}):
            prop = self._info["props"]["last_active_mac"]
            return EineMacsInfo(active=net.format_mac(prop["value"]), timestamp=prop["timestamp"])
        else:
            return None

    def __repr__(self):
        return repr(self._info)

    def __eq__(self, other):
        return isinstance(other, type(self)) and self._info == other._info

    def _assert_has_profile(self):
        # this is a guardian against incorrect usage.
        # 1. Methods with this guardian should be called only after caller ensured that host have a profile assigned.
        # 2. Methods with this guardian should be called only if host info was fetched with `profile` flag.
        if not (self.has_profile() and self._info["einstellung"].get("profile_name")):
            # Always check that host has profile.
            raise LogicalError()

    def _get_current_stage(self):
        stage_id = self._info["einstellung"].get("current_stage")
        if stage_id is None:
            return None
        return self._info["einstellung"]["stages"][stage_id]


class EineProvider(ABC):
    @abstractmethod
    def get_host(self) -> str:
        raise NotImplementedError

    @abstractmethod
    def get_headers(self) -> dict[str, str]:
        raise NotImplementedError


class YandexInternalEineProvider(EineProvider):
    def __init__(self, host: str, token: str):
        self._host = host
        self._token = token

    def get_host(self) -> str:
        return self._host

    def get_headers(self) -> dict[str, str]:
        return {"Authorization": "OAuth {}".format(self._token)}


class YandexBoxEineProvider(EineProvider):
    def __init__(self, host: str, login: str, password: str):
        self._host = host
        self._login = login
        self._password = password

    def get_host(self) -> str:
        return self._host

    def get_headers(self) -> dict[str, str]:
        cred = "{}:{}".format(self._login, self._password)
        return {"Authorization": "Basic {}".format(standard_b64encode(cred.encode()).decode())}


class EineStageLogHelper:
    @classmethod
    def iterator(cls, records):
        for record in records:
            yield cls.format(record)

    @staticmethod
    def format(record):
        return "{time} | {type} | {message}".format(
            time=format_time(record["date"]), type=record["type"], message=record["message"]
        )


class EineClient:
    def __init__(self, provider: EineProvider):
        self._provider = provider

    def get_profiles(self):
        scheme = DictScheme({"_id": String(), "created_at": Float(), "created_by": String(), "name": String()})
        profiles = self._api_request_pager("GET", "/profiles/", scheme=scheme, timeout=LOW_LATENCY_NETWORK_TIMEOUT)
        return [profile["name"] for profile in profiles]

    def get_tags(self, inv):
        return self._api_request(
            "GET",
            "/computers/inventory/{inv}/tags".format(inv=inv),
            allow_errors=[requests.codes.not_found],
            scheme=List(String()),
        )

    def update_tags(self, inv, add=None, remove=None):
        update = {}

        if add:
            update["add"] = add

        if remove:
            update["remove"] = remove

        if not update:
            return

        self._api_request(
            "PATCH",
            "/computers/inventory/{inv}/tags".format(inv=inv),
            data=update,
            allow_errors=[requests.codes.not_found],
        )

    def get_host_status(self, inv, profile=False, location=False) -> EineHostStatus:
        """Returns current host status.

        Attention: Host gets "queued" status on adding, so it not always means that host is profiling now - only when it
        has "queued" status and profile is not null.
        """

        request_fields = ["inventory", "hostname", "in_use", "macs", "einstellung._id"]
        einstellung_scheme = DictScheme({"_id": String()}, optional=True)

        if profile:
            request_fields += [
                "einstellung.status",
                "einstellung.profile_name",
                "einstellung.current_stage",
                "einstellung.assigned_at",
                "einstellung.updated_at",
                "einstellung.message",
                "einstellung.stages.status",
                "einstellung.stages.stage",
                "einstellung.tags_local",
                "props.otrs_ticket",
            ]
            einstellung_scheme = ApiDictScheme(
                {
                    "_id": String(),
                    "profile_name": String(),
                    "current_stage": Integer(optional=True),
                    "assigned_at": Float(),
                    "updated_at": Float(),
                    "message": String(optional=True),
                    "status": Integer(min=0, max=len(EineProfileStatus.ALL) - 1),
                    "tags_local": List(String(), optional=True),
                    "stages": List(
                        DictScheme(
                            {
                                "status": Integer(min=0, max=len(EineStageState.ALL) - 1),
                                "stage": String(),
                            }
                        )
                    ),
                },
                optional=True,
                drop_none=True,
            )
        if location:
            request_fields += ["rack", "props.switch", "props.last_active_mac"]

        property_scheme = DictScheme(
            {
                "timestamp": Float(),
                "value": String(),
            },
            optional=True,
        )

        scheme = ApiDictScheme(
            {
                "inventory": StringToInteger(choices=[inv]),
                "hostname": String(optional=True),
                "in_use": Bool(),
                "macs": List(String()),
                "rack": String(optional=True),
                "props": DictScheme(
                    {
                        "otrs_ticket": property_scheme,
                        "switch": property_scheme,
                        "last_active_mac": property_scheme,
                    },
                    optional=True,
                ),
                "einstellung": einstellung_scheme,
            },
            drop_none=True,
        )
        try:
            host = self._api_request(
                "GET",
                "/computers/inventory/{inv}/".format(inv=inv),
                scheme=scheme,
                allow_errors=[requests.codes.not_found],
                params={"fields": ",".join(request_fields)},
            )
        except EinePersistentError as e:
            if e.status_code == http.client.NOT_FOUND:
                raise e.host_does_not_exist(inv)
            raise

        return EineHostStatus(host)

    def get_profile_log(self, inv):
        host_info = self.get_host_status(inv)

        if not host_info.has_profile():
            return ""

        scheme = List(
            ApiDictScheme(
                {
                    "_id": String(),
                    "einstellung_id": String(choices=[host_info.profile_id()]),
                    "date": Float(),
                    "type": String(),
                    "message": String(),
                    "created_at": Float(),
                    "code": String(optional=True),
                    "values": DictScheme(
                        {}, ignore_unknown=True, optional=True
                    ),  # is this field important? what's there?
                    "message_id": Integer(optional=True),
                },
                drop_none=True,
            )
        )

        response = self._api_request("GET", "/logs/{}/".format(host_info.profile_id()), scheme=scheme)
        records = EineStageLogHelper.iterator(response)
        return "\n".join(records)

    def assign_profile(self, inv, profile, local_tags=None, cc_logins=None, assigned_for=None):
        assigned_for = _verify_assigned_for_argument(assigned_for)
        self._api_request(
            "POST",
            "/computers/inventory/{inv}/einstellungs/".format(inv=inv),
            data=drop_none(
                {
                    "action": "assign",
                    "profile": profile,
                    "tags_local": local_tags,
                    "cc": cc_logins,
                    "assigned_for": assigned_for,
                }
            ),
            allow_errors=[requests.codes.not_found],
        )

    def get_profile_stat(self):
        """Returns information of last known time of successful host profiling."""
        host_filters = {
            "einstellung.profile_name": EINE_PROFILES_WITH_DC_SUPPORT,
            "einstellung.status": EineProfileStatus.COMPLETED,
        }

        hosts_iterator = self._eine_host_info_iterator(profile=True, filters=host_filters)
        return {h.inventory(): int(h.profile_updated_timestamp()) for h in hosts_iterator}

    def set_host_location(self, inv, switch, port):
        self._api_request(
            "PUT",
            "/computers/inventory/{inv}/props/switch/".format(inv=inv),
            data={"value": "{switch} {port}".format(switch=switch, port=port)},
            expected_status_code=requests.codes.created,
        )

    def get_network_map(self):
        netmap = {}

        for host in gevent_idle_iter(self._eine_host_info_iterator()):
            inv = host.inventory()

            network_configuration = host.switch(), host.active_mac()
            if any(network_configuration):
                netmap[inv] = network_configuration

        return netmap

    def _eine_host_info_iterator(self, profile=False, filters=None):
        request_fields = ["inventory", "hostname", "in_use", "macs", "rack", "props.switch", "props.last_active_mac"]
        if profile:
            request_fields += ["einstellung.status", "einstellung.profile_name", "einstellung.updated_at"]

        params = dict(filters or {}, fields=",".join(request_fields))

        property_scheme = DictScheme(
            {
                "timestamp": Float(),
                "value": String(),
            },
            optional=True,
        )

        scheme = ApiDictScheme(
            {
                "inventory": StringToInteger(),
                "hostname": String(optional=True),
                "in_use": Bool(),
                "macs": List(String()),
                "rack": String(optional=True),
                "props": DictScheme(
                    {
                        "switch": property_scheme,
                        "last_active_mac": property_scheme,
                    },
                    optional=True,
                ),
                "einstellung": DictScheme(
                    {
                        "profile_name": String(),
                        "updated_at": Float(),
                        "status": Integer(min=0, max=len(EineProfileStatus.ALL) - 1),
                    },
                    optional=True,
                ),
            },
            drop_none=True,
        )

        for host_info in self._api_request_pager("GET", "/computers/", params=params, scheme=scheme):
            yield EineHostStatus(host_info)

    def _api_request_pager(self, method, path, params=None, data=None, allow_errors=(), scheme=None, **kwargs):
        """Take request parameters and return an iterator over all response items across all pages.
        Scheme is a scheme of a single item.
        """
        # If you need to change page size or want ro forbid paging at all, add `per_page` or `nopager` parameters.
        # `nopager` parameter will eliminate "paginate" dict from response.
        MAX_ALLOWED_PAGE_SIZE = 1000

        if params is None:
            params = {}

        page_number = 1
        has_next = True
        while has_next:
            params["page"] = page_number
            params["per_page"] = MAX_ALLOWED_PAGE_SIZE
            pager_scheme = DictScheme(
                {
                    "paginate": DictScheme(
                        {
                            "count": Integer(),
                            "has_next": Bool(),
                            "has_previous": Bool(),
                            "pages": Integer(),
                            "per_page": Integer(),
                        }
                    ),
                    "result": GeventIdleList(scheme),
                }
            )

            page = self._api_request(
                method,
                path,
                params=params,
                data=data,
                allow_errors=allow_errors,
                pager=True,
                scheme=pager_scheme,
                **kwargs
            )
            has_next = page["paginate"]["has_next"]
            page_number += 1

            yield from gevent_idle_iter(page["result"])

    def _api_request(
        self,
        method,
        path,
        params=None,
        data=None,
        allow_errors=(),
        scheme=None,
        pager=False,
        expected_status_code=None,
        **kwargs
    ):
        url = "https://{}/api/v2{}".format(self._provider.get_host(), path)

        if scheme is not None and not pager:  # pager sends full scheme with paging attrs.
            scheme = DictScheme({"result": scheme})

        if expected_status_code is None:
            expected_status_code = requests.codes.ok

        # Eine v2 API doesn't support JSON API: all requests must be sent as application/x-www-form-urlencoded.
        # Do some postprocessing to make it more like JSON API.
        if data is not None:
            data = data.copy()
            for key, value in list(data.items()):
                if isinstance(value, (tuple, list)):
                    if value:
                        data[key] = ",".join(map(str, value))
                    else:
                        del data[key]

        try:
            response, result = walle.clients.utils.json_request(
                "eine",
                method,
                url,
                params=params,
                form_data=data,
                headers=self._provider.get_headers(),
                success_codes=(expected_status_code,),
                allow_errors=util.misc.ALL_ERRORS,
                scheme=scheme,
                error_scheme=ApiDictScheme({"description": String()}),
                with_response=True,
                **kwargs
            )
        except RequestException as e:
            raise EineInternalError("{}", e, url=url, params=params)

        if response.status_code == expected_status_code:
            return (
                result if pager else result["result"]
            )  # send all attrs to pager, there are some paging related attrs.
        elif response.status_code in allow_errors:
            raise EinePersistentError(
                strip_api_error(result["description"]), status_code=response.status_code, url=url, params=params
            )
        else:
            error = strip_api_error(result["description"])
            raise EineInternalError(
                "Server returned an error ({} {}): {}",
                response.status_code,
                response.reason,
                error,
                status_code=response.status_code,
                url=url,
                params=params,
                error=error,
            )


def get_yandex_internal_provider() -> EineProvider:
    return YandexInternalEineProvider(
        config.get_value("eine.host", "localhost"), config.get_value("eine.access_token", "")
    )


def get_yandex_box_provider_if_exists(box_id: str) -> tp.Optional[EineProvider]:
    params = config.get_value("eine.boxes.{}".format(box_id), None)
    if params is None:
        return None
    return YandexBoxEineProvider(
        host=params.get("host", "localhost"), login=params.get("login", ""), password=params.get("password", "")
    )


def get_eine_provider(box_id: tp.Optional[str] = None) -> EineProvider:
    if box_id is not None:
        provider = get_yandex_box_provider_if_exists(box_id)
        if provider is not None:
            return provider
    return get_yandex_internal_provider()


def get_client(provider: EineProvider) -> EineClient:
    return EineClient(provider)


def get_eine_profiles(provider: EineProvider):
    @db_cache.cached("eine-profiles-of-{}".format(provider.get_host()), 5 * constants.MINUTE_SECONDS)
    def cache_getter():
        log.debug("Updating Einstellung profile names cache for {}...".format(provider.get_host()))
        return get_client(provider).get_profiles()

    return cache_getter()


def _verify_assigned_for_argument(arg):
    if arg:
        arg = arg.rstrip("@")
        arg = ISSUER_ROBOT_WALLE if arg == ISSUER_WALLE else arg
    return arg
