from requests import RequestException

from object_validator import List, DictScheme, String, Integer
from walle.clients.utils import json_request
from walle.errors import RecoverableError
from walle.hosts import Host
from walle.trypo_radix import TRYPOCompatibleRadix
from walle.util.validation import AnyScheme, Network, TrypoNetwork, TrypoNetworkRange, IPAddress

RT_DATACENTER_RE = r"_DC_([A-Z]+)_NETS_"


class HBFInternalError(RecoverableError):
    def __init__(self, message, *args, **kwargs):
        super().__init__("Error in communication with HBF: " + message, *args, **kwargs)


def get_hbf_drills():
    network_scheme = AnyScheme([Network(), TrypoNetwork(), TrypoNetworkRange()])
    scheme = List(
        DictScheme(
            {
                "project": String(),
                "project_ips": List(network_scheme),
                "location": String(regex=RT_DATACENTER_RE),
                "location_ips": List(network_scheme),
                "begin": Integer(),
                "duration": Integer(),
                "exclude": List(String()),
                "exclude_ips": List(AnyScheme([IPAddress(), Network(), TrypoNetwork(), TrypoNetworkRange()])),
            }
        )
    )

    url = "https://hbf.yandex.net/drills-walle/v1"
    try:
        resp = json_request("hbf", "GET", url, scheme=scheme)
    except RequestException as e:
        raise HBFInternalError("{}", e, url=url)

    return resp


def get_macro_networks(macro):
    scheme = List(AnyScheme([IPAddress(), Network(), TrypoNetwork(), TrypoNetworkRange()]))

    url = "https://hbf.yandex.net/macros/{}".format(macro)
    try:
        resp = json_request("hbf", "GET", url, scheme=scheme)
    except RequestException as e:
        raise HBFInternalError("{}", e, url=url)

    return resp


def get_hosts_of_macro(macro, dc=None):
    radix = TRYPOCompatibleRadix()
    for net in get_macro_networks(macro):
        radix.add(net)

    query = {"ips__exists": True}
    if dc is not None:
        query["location__short_datacenter_name"] = dc

    matching_hosts = {}
    for host in Host.objects(**query).only("name", "ips"):
        for ip in host.ips:
            node = radix.search_best(ip)
            if node is not None:
                matching_hosts[host.name] = "{} matches {}".format(ip, radix.node_repr(node))
                break

    return matching_hosts
