import functools

import dataclasses
import enum
import logging
import typing as tp

import grpc
import yc_as_client
from yandex.cloud.priv.iam.v1 import (
    user_account_service_pb2,
    user_account_service_pb2_grpc,
    service_account_service_pb2,
    service_account_service_pb2_grpc,
)

from sepelib.core import config

GIZMO_RESOURCE = "gizmo"

FOLDER_RESOURCE_TYPE = "resource-manager.folder"
GIZMO_RESOURCE_TYPE = "iam.gizmo"

IAM_AUTH_FILTER_LIMIT = 20


log = logging.getLogger(__name__)


class IamAuthError(Exception):
    """Authentification or authorization error from IAM"""


class TooComplexAuthError(Exception):
    """Authorization is too complex, relax filter request"""


class BasePermission(str, enum.Enum):
    """For typing"""


class ProjectsPermissions(BasePermission):
    create = "wall-e.projects.create"
    update = "wall-e.projects.update"
    get = "wall-e.projects.get"
    delete = "wall-e.projects.delete"


class AutomationPlotsPermissions(BasePermission):
    update = "wall-e.automation-plots.update"
    get = "wall-e.automation-plots.get"
    delete = "wall-e.automation-plots.delete"


class HostsPermissions(BasePermission):
    update = "wall-e.hosts.update"
    get = "wall-e.hosts.get"


class ApiPermissions(BasePermission):
    juggler = "wall-e.juggler.push"
    agent = "wall-e.agent.push"


class FederatedUserAccountSubject(yc_as_client.entities.UserAccountSubject):
    def __init__(self, id, federation_id=""):
        self.id = id
        self.federation_id = federation_id

    def __repr__(self):
        return f"<FederatedUserAccount: id {self.id}, federation_id {self.federation_id}>"

    def __eq__(self, other):
        return (
            isinstance(other, FederatedUserAccountSubject)
            and other.id == self.id
            and other.federation_id == self.federation_id
        )

    @classmethod
    def _from_grpc_message(cls, grpc_subject):
        return cls(
            id=grpc_subject.id,
            federation_id=grpc_subject.federation_id,
        )


# XXX(rocco66): for federation user info and SA name fetching
yc_as_client.entities.UserAccountSubject = FederatedUserAccountSubject


@dataclasses.dataclass(frozen=True)
class IamAction:
    permission: BasePermission
    resource: str

    @property
    def resource_type(self) -> str:
        if isinstance(self.permission, ApiPermissions) or self.permission == ProjectsPermissions.create:
            return GIZMO_RESOURCE_TYPE
        else:
            return FOLDER_RESOURCE_TYPE

    def __str__(self):
        return f"IamAction(permission: {self.permission}, {self.resource_type}: {self.resource})"


Subject = tp.Union[yc_as_client.entities.UserAccountSubject, yc_as_client.entities.ServiceAccountSubject]


def _check_subject_type(subject):
    if not isinstance(subject, (yc_as_client.entities.UserAccountSubject, yc_as_client.entities.ServiceAccountSubject)):
        raise IamAuthError(f"wrong IAM subject type {subject}")


class AsClient:
    def __init__(self, iam_as_endpoint):
        channel = grpc.secure_channel(
            iam_as_endpoint,
            grpc.ssl_channel_credentials(),
            options=tuple(),
        )
        self._iam_access_service_client = yc_as_client.YCAccessServiceClient(channel)

    def authenticate(self, iam_token, request_id) -> Subject:
        try:
            result = self._iam_access_service_client.authenticate(iam_token=iam_token, request_id=request_id)
        except (
            yc_as_client.exceptions.BadRequestException,
            yc_as_client.exceptions.UnauthenticatedException,
        ) as exc:
            raise IamAuthError(str(exc))
        _check_subject_type(result)
        return result

    def authorize(self, iam_token, actions: [IamAction], request_id) -> Subject:
        # TODO(rocco66): remove filter after service X will be created
        actual_actions = {a for a in actions if not isinstance(a.permission, HostsPermissions)}
        if len(actual_actions) > IAM_AUTH_FILTER_LIMIT:
            raise TooComplexAuthError(
                f"Too complex filter for log: {len(actual_actions)} (limit is {IAM_AUTH_FILTER_LIMIT})"
            )

        # TODO(rocco66): BulkAuthorizeRequest
        for action_index, action in enumerate(actual_actions):
            try:
                result = self._iam_access_service_client.authorize(
                    permission=action.permission,
                    resource_path=yc_as_client.entities.Resource(id=action.resource, type=action.resource_type),
                    iam_token=iam_token,
                    request_id=request_id,
                )
            except (
                yc_as_client.exceptions.BadRequestException,
                yc_as_client.exceptions.UnauthenticatedException,
            ) as exc:
                raise IamAuthError(f"{str(exc)} for {action}")
            except yc_as_client.exceptions.PermissionDeniedException as exc:
                error_msg = f"{str(exc)} for {action} subject {self.authenticate(iam_token, request_id)} request-id {request_id}"
                log.info(error_msg)
                raise IamAuthError(error_msg)
            else:
                if action_index == len(actions) - 1:
                    _check_subject_type(result)
                    return result


def get_name_by_token(iam_token, subject):
    if isinstance(subject, yc_as_client.entities.UserAccountSubject):
        return _cached_get_user_login(iam_token, subject.id)
    elif isinstance(subject, yc_as_client.entities.ServiceAccountSubject):
        return _cached_get_sa_name(iam_token, subject.id)
    else:
        raise IamAuthError(f"wrong IAM subject type {subject}")


@functools.lru_cache(maxsize=1000)
def _cached_get_user_login(iam_token, subject_id):
    client = IamClient(config.get_value("iam.iam_endpoint"))
    # TODO assert federative user subject?
    return client.get_user_login(iam_token, subject_id)


@functools.lru_cache(maxsize=1000)
def _cached_get_sa_name(iam_token, subject_id):
    client = IamClient(config.get_value("iam.iam_endpoint"))
    try:
        name = client.get_sa_name(iam_token, subject_id)
    except Exception:
        # NOTE(rocco66): it is better to have `@123:sa-name` instead of `@123`, but iam_token might not have
        # permissions for that
        return f"@{subject_id}"
    return f"@{subject_id}:{name}"


class IamClient:
    def __init__(self, iam_endpoint):
        channel = grpc.secure_channel(
            iam_endpoint,
            grpc.ssl_channel_credentials(),
            options=tuple(),
        )
        self._user_account_service = user_account_service_pb2_grpc.UserAccountServiceStub(channel)
        self._service_account_service = service_account_service_pb2_grpc.ServiceAccountServiceStub(channel)

    def get_user_login(self, iam_token, subject_id):
        user_account = self._user_account_service.Get(
            user_account_service_pb2.GetUserAccountRequest(user_account_id=subject_id),
            metadata=(('authorization', f"Bearer {iam_token}"),),
        )
        name_id = user_account.saml_user_account.name_id
        # NOTE(rocco66): name_id is 'rocco66@yandex-team.ru', we should return 'rocco66@'
        return name_id[: name_id.find("@") + 1]

    def get_sa_name(self, iam_token, subject_id):
        service_account = self._service_account_service.Get(
            service_account_service_pb2.GetServiceAccountRequest(service_account_id=subject_id),
            metadata=(('authorization', f"Bearer {iam_token}"),),
        )
        return service_account.name
