"""Provides a client to host inventory service.

Note: at this time there is no such service, but there are a few services that already provide some information that
will be stored in inventory service later.
"""

import logging
from collections import namedtuple

from sepelib.core.exceptions import LogicalError
from walle import constants as walle_constants
from walle.clients import bot, deploy, eine
from walle.constants import NetworkTarget
from walle.errors import InvalidDeployConfiguration, NoInformationError, InvalidHostConfiguration
from walle.hosts import get_host_human_id

log = logging.getLogger(__name__)


def check_deploy_configuration(
    provisioner, config, project_box_id, tags=None, need_certificate=False, network=None, deploy_config_policy=None
):
    """Checks the specified deploy config for existence.

    :raises InvalidConfigNameError
    """

    if provisioner == walle_constants.PROVISIONER_LUI:
        if tags is not None:
            raise InvalidDeployConfiguration("Can't use Einstellung tags with LUI.")

        if config not in deploy.get_deploy_configs(deploy.get_deploy_provider(project_box_id)):
            raise InvalidDeployConfiguration("Invalid deploy config name: {}.", config)

    elif provisioner == walle_constants.PROVISIONER_EINE:
        if config not in get_eine_profiles():
            raise InvalidDeployConfiguration("Invalid Einstellung deploy profile: {}.", config)

        if need_certificate:
            raise InvalidHostConfiguration("Can only install host certificates with LUI.")

        if network and network != NetworkTarget.SERVICE:
            raise InvalidHostConfiguration("Can not deploy host in project's network with Eine, try use LUI.")

        if deploy_config_policy is not None:
            raise InvalidDeployConfiguration("Deploy policies are currently used only with LUI")

    else:
        raise LogicalError()


def get_eine_profiles():
    return sorted(set(eine.get_eine_profiles(eine.get_eine_provider(box_id=None))))


def get_lui_configs():
    return sorted(set(deploy.get_deploy_configs(deploy.get_deploy_provider(box_id=None))))


BotHostInfo = namedtuple("BotHostInfo", "inv,name,ipmi_mac,macs,location,bot_project_id,platform")


def get_host_info_and_check_status(inv_or_name, supress_exceptions=False):
    """Gets information from BOT for the specified host and checks its OEBS status.

    Attention: returned name may be None if host is not acquired yet by the host owner.

    :raises NoInformationError, InvalidHostConfiguration
    :rtype BotHostInfo
    """

    info = bot.get_host_info(inv_or_name, with_macs=True, with_platform=True)
    if info is None:
        raise NoInformationError("{} is not registered in BOT.", inv_or_name)

    inv = info["inv"]
    name = info.get("name")
    ipmi_mac = info.get("ipmi_mac")
    macs = info["macs"]
    platform = info["platform"]

    # We should check the status to prevent adding an excepted host to Wall-E (which Wall-E will invalidate on next
    # synchronization with Bot database).
    #
    # See https://st.yandex-team.ru/SEPE-9156
    oebs_status = info["oebs_status"]
    if bot.is_excepted_status(oebs_status) and not supress_exceptions:
        raise InvalidHostConfiguration(
            "{} has '{}' OEBS status which means that it's excepted from production.",
            get_host_human_id(inv, name),
            oebs_status,
            host_id=get_host_human_id(inv, name),
            oebs_status=oebs_status,
        )

    if not macs and not supress_exceptions:
        raise InvalidHostConfiguration("{} doesn't have any registered MAC addresses.", get_host_human_id(inv, name))

    return BotHostInfo(inv, name, ipmi_mac, macs, info["location"], info.get("bot_project_id"), platform)
