"""Provides hardware management tools."""

import http.client
import logging
import re
import time
from abc import abstractmethod, ABC
from base64 import standard_b64encode
from typing import Optional

from requests import RequestException

import object_validator
import walle.clients.utils
from object_validator import String, Bool, DictScheme
from sepelib.core import config
from walle import boxes
from walle.clients.dns.local_dns_resolver import LocalDNSResolver
from walle.clients.utils import retry
from walle.errors import RecoverableError, UserRecoverableError
from walle.util.validation import ApiDictScheme

log = logging.getLogger(__name__)


class InternalError(RecoverableError):
    """Raised when some internal error occurred."""


class HostHwError(RecoverableError):
    """Raised on error during communication with host's service processor."""


class BrokenIpmiCommandError(HostHwError):
    """
    Some IPMI commands may not work on some nodes. The exception is intended to workaround the errors when it's
    possible and not create admin requests for IPMI error.
    """


class IpmiHostMissingError(RecoverableError):
    """
    Sometimes IPMI host vanishes from DNS which requires administrator attention.

    This means that DHCP server hasn't given an IP address and corresponded DNS record to the service processor which
    may be caused by:
    * The machine is not connected to IPMI switch.
    * Broken cable.
    * The machine was installed to a queue without IPMI support.
    """

    def __init__(self, ipmi_fqdn, **kwargs):
        super().__init__("IPMI host {} is missing.", ipmi_fqdn, ipmi_fqdn=ipmi_fqdn, **kwargs)
        self.ipmi_fqdn = ipmi_fqdn


class IPMIProxyTemporaryError(RecoverableError):
    """Indicates IPMI Proxy server error. Retry may help."""

    pass


class IPMIProxyPermanentError(UserRecoverableError):
    """Indicates IPMI Proxy permanent error. Fail the task, nothing may help."""

    pass


class IpmiProvider(ABC):
    @abstractmethod
    def get_url(self) -> str:
        raise NotImplementedError

    @abstractmethod
    def get_auth_headers(self) -> dict[str, str]:
        raise NotImplementedError

    @abstractmethod
    def handle_not_found_error(self, response, error):
        raise NotImplementedError


class YandexInternalIpmiProvider(IpmiProvider):
    def __init__(self, proxy_host: str, token: str, ipmi_mac: str):
        self._proxy_host = proxy_host
        self._token = token
        self._ipmi_mac = ipmi_mac

    def _get_ipmi_fqdn(self):
        ipmi_host = self._ipmi_mac.replace(":", "-")
        ipmi_fqdn = ipmi_host + ".ipmi.yandex.net"
        return ipmi_fqdn

    def get_url(self) -> str:
        ipmi_fqdn = self._get_ipmi_fqdn()

        local_dns_resolver = LocalDNSResolver()
        ipmi_ips = local_dns_resolver.get_aaaa(ipmi_fqdn) or local_dns_resolver.get_a(ipmi_fqdn)
        if not ipmi_ips:
            raise IpmiHostMissingError(ipmi_fqdn)

        return "https://{}/ipmiproxy/api/v1.0/{}".format(self._proxy_host, ipmi_fqdn)

    def get_auth_headers(self) -> dict[str, str]:
        return {"Authorization": "OAuth {}".format(self._token)}

    def handle_not_found_error(self, response, error):
        raise IpmiHostMissingError(
            self._get_ipmi_fqdn(), response=response, status_code=http.client.NOT_FOUND, error=error
        )


class YandexBoxIpmiProvider(IpmiProvider):
    def __init__(self, proxy_host: str, login: str, password: str, inv: int):
        self._proxy_host = proxy_host
        self._login = login
        self._password = password
        self._inv = inv

    def get_url(self) -> str:
        return "https://{}/ipmiproxy/api/v1.0/{}".format(self._proxy_host, self._inv)

    def get_auth_headers(self) -> dict[str, str]:
        cred = "{}:{}".format(self._login, self._password)
        return {"Authorization": "Basic {}".format(standard_b64encode(cred.encode()).decode())}

    def handle_not_found_error(self, response, error):
        pass


class IpmiProxyClient:
    """Uses IPMI Proxy service as IPMI backend."""

    BMC_RESET_COLD = "cold"
    BMC_RESET_WARM = "warm"

    _IPMI_COMMAND_RESPONSE_SCHEME = ApiDictScheme(
        {
            "message": String(),
            "host": String(),
            "success": Bool(choices=[True]),
        }
    )

    _IPMI_INFO_RESPONSE_SCHEME = ApiDictScheme(
        {"host": String(), "data": DictScheme({"System Power": String(choices=("on", "off"))}, ignore_unknown=True)}
    )

    _IPMI_ERROR_RESPONSE_SCHEME = ApiDictScheme(
        {
            "message": String(),
        }
    )

    _max_tries = 3
    _try_interval = 5
    _bootdev_apply_timeout = 5

    def __init__(self, provider: IpmiProvider, human_id: Optional[str] = None):
        self._provider = provider
        self._human_id = human_id or ""

    def is_power_on(self):
        return self.power_status() == "on"

    def power_status(self):
        return self._call_new("/chassis/status")["data"]["System Power"]

    def power_on(self, pxe=False):
        log.info("Power on %s%s.", self._human_id, " (PXE)" if pxe else "")

        if pxe:
            self._call_new("/chassis/bootdev/pxe", "POST", data={"disable_bootdev_timeout": 1})
            # Supermicro X9 does not run power/on after bootdev/pxe, need to let it rest for a bit.
            # Think this is weird? See https://st.yandex-team.ru/WALLESUPPORT-2 then.
            log.info("%s: Wait for bootdev/pxe to apply...", self._human_id)
            time.sleep(self._bootdev_apply_timeout)

        self._call_new("/chassis/power/on", "POST")

    def reset(self):
        log.info("Reset %s.", self._human_id)
        self._call_new("/chassis/power/reset", "POST")

    def power_off(self):
        log.info("Power off %s.", self._human_id)
        self._call_new("/chassis/power/off", "POST")

    def soft_power_off(self):
        log.info("Power off %s (soft).", self._human_id)
        self._call_new("/chassis/power/soft", "POST")

    def bmc_reset(self, reset_type=BMC_RESET_COLD):
        log.info("Executing bmc reset %s command for %s.", reset_type, self._human_id)
        self._call_new("/bmc/reset/{}".format(reset_type), "POST")

    def raw_command(self, command, timeout=10):
        ipmitool_command = "raw {}".format(command)
        return self._call_ipmitool(ipmitool_command, timeout)

    def _call_ipmitool(self, command, timeout=10):
        log.debug("Executing ipmitool command: %s, on %s with timeout: %s", command, self._human_id, timeout)

        payload = {
            'timeout': timeout,
            'cmd': command,
        }
        return self._call_new("/command", method="POST", data=payload)

    def _call_new(self, path, method="GET", data=None, **kwargs):
        log.debug("IPMI Proxy command: %s for %s...", path, self._human_id)
        result = self.json_request(path, method, data=data, **kwargs)
        log.info(
            "IPMI Proxy command for %s success: %s", self._human_id, result.get("message") or "[no message provided]"
        )

        return result

    @retry(max_tries=_max_tries, interval=_try_interval, exceptions=(HostHwError,), skip=(BrokenIpmiCommandError,))
    def json_request(self, path, method="GET", scheme=None, data=None, **kwargs):
        base_url = self._provider.get_url()
        auth_headers = self._provider.get_auth_headers()

        url = "{base_url}/{path}".format(base_url=base_url, path=path.lstrip("/"))
        if scheme is None:
            scheme = self._default_scheme(method)

        try:
            result = walle.clients.utils.json_request(
                "ipmiproxy", method, url, data=data, headers=auth_headers, scheme=scheme, **kwargs
            )
        except RequestException as e:
            # Use e.message here: unicode(RequestException("some non-ascii text")) raises UnicodeEncodeError
            if e.response is not None:
                self._raise_for_status(e.response, str(e))
            raise IPMIProxyTemporaryError(str(e), url=url, data=data, **kwargs)
        except OSError as e:
            raise IPMIProxyTemporaryError(str(e), url=url, data=data, **kwargs)

        return result

    def _default_scheme(self, method):
        if method == "GET":
            return self._IPMI_INFO_RESPONSE_SCHEME
        if method == "POST":
            return self._IPMI_COMMAND_RESPONSE_SCHEME
        return None

    def _raise_for_status(self, response, error):
        code = response.status_code

        if code >= 500:  # server error, retry may help
            raise IPMIProxyTemporaryError(
                "Failed to access IPMI Proxy {}: {} {}\n{}",
                response.url,
                code,
                response.reason,
                response.text,
                response=response,
                status_code=code,
                error=error,
            )

        message = self._validate_error_response(response, error)
        if code == http.client.NOT_FOUND:
            log.info("Got NOT FOUND error from IPMI Proxy: %s", message)
            self._provider.handle_not_found_error(response, error)

        if code == http.client.UNPROCESSABLE_ENTITY:
            if re.match("IPMI error: .* Invalid data field in request", message):
                raise BrokenIpmiCommandError(message, response=response, status_code=code, error=error)
            else:
                raise HostHwError(message, response=response, status_code=code, error=error)

        raise InternalError(message, response=response, status_code=code, error=error)

    def _validate_error_response(self, response, error_message):
        result = response.json()
        try:
            result = object_validator.validate("result", result, self._IPMI_ERROR_RESPONSE_SCHEME)
        except object_validator.ValidationError as error:
            log.error("%s: Got an invalid JSON response: %r. %s", response.url, result, error)
            message = error_message
        else:
            message = result["message"].rstrip("\n")
        return message


def get_yandex_internal_provider(ipmi_mac: str) -> IpmiProvider:
    return YandexInternalIpmiProvider(
        proxy_host=config.get_value("ipmiproxy.host", "localhost"),
        token=config.get_value("ipmiproxy.access_token", ""),
        ipmi_mac=ipmi_mac,
    )


def get_yandex_box_provider(box_id: str, inv: int) -> IpmiProvider:
    params = config.get_value(f"{boxes.IPMIPROXY_BOXES_SECTION}.{box_id}")
    return YandexBoxIpmiProvider(
        proxy_host=params.get("host", "localhost"),
        login=params.get("login", ""),
        password=params.get("password", ""),
        inv=inv,
    )


def get_client(provider: IpmiProvider, human_id: Optional[str] = None) -> IpmiProxyClient:
    """Returns hardware management client for the specified host."""
    return IpmiProxyClient(provider, human_id)
