"""Client methods for netmon API."""
import json
import logging
from typing import Optional

import fastjsonschema
from requests import ConnectionError, Timeout

from walle.clients.utils import json_request as make_json_request, retry, HttpServerError
from walle.util.jsonschema import Array, Number, Object, String, OneOfPartial
from walle.util.misc import drop_none

log = logging.getLogger(__name__)

_ALIVE_REPORT_SCHEMA = Object(
    properties={
        "generated": Number(),
        "switches": Array(
            items=Object(
                properties={
                    "name": String(),
                    "queue": String(),
                    "dc": String(),
                    "alive": Number(),
                    "connectivity": OneOfPartial(
                        parent=Array(items=Number(minimum=0, maximum=1)),
                        params=[{"minItems": 0, "maxItems": 0}, {"minItems": 4, "maxItems": 4}],
                    ),
                },
                required=["name", "queue", "dc", "alive", "connectivity"],
            )
        ),
        "queues": Array(
            items=Object(
                properties={
                    "name": String(),
                    "dc": String(),
                    "alive": Number(),
                    "connectivity": OneOfPartial(
                        parent=Array(items=Number(minimum=0, maximum=1)),
                        params=[{"minItems": 0, "maxItems": 0}, {"minItems": 4, "maxItems": 4}],
                    ),
                },
                required=["name", "dc", "alive", "connectivity"],
            )
        ),
        "datacenters": Array(
            items=Object(
                properties={
                    "name": String(),
                    "alive": Number(),
                    "connectivity": OneOfPartial(
                        parent=Array(items=Number(minimum=0, maximum=1)),
                        params=[{"minItems": 0, "maxItems": 0}, {"minItems": 4, "maxItems": 4}],
                    ),
                },
                required=["name", "alive", "connectivity"],
            )
        ),
    },
    required=["generated", "switches", "queues", "datacenters"],
)

_ALIVE_REPORT_VALIDATOR = fastjsonschema.compile(_ALIVE_REPORT_SCHEMA.get_schema())


class NetmonClient:
    def __init__(
        self,
        service: str,
        host: str,
        expression: Optional[str] = None,
        network: Optional[str] = None,
        protocol: Optional[str] = None,
    ):
        self.service = service
        self.host = host
        self.expression = expression
        self.network = network
        self.protocol = protocol

    def get_alive_metrics(self, level):
        """Get connectivity info (alive metric).
        Netmon uses different time windows for different size of objects: 2 min for dc, 9 min for queue, 30 min for switch.
        We use configurable value called `level` to address that.
        Each response contains all levels but they all calculated according to the selected time window.
        """
        url_path = "{level}/alive".format(level=level)
        data = drop_none(dict(expression=self.expression, network=self.network, protocol=self.protocol))
        return self.json_request(url_path, data=data, validator=_ALIVE_REPORT_VALIDATOR)

    def get_seen_hosts(self):
        url_path = "seen_hosts"
        data = drop_none(dict(expression=self.expression, network=self.network, protocol=self.protocol))
        return self.json_request(url_path, data)["hosts"]

    @retry(interval=1, backoff=2, exceptions=(HttpServerError, ConnectionError, Timeout))
    def json_request(self, path, data, validator=None):
        headers = {"Accept-Encoding": "deflate, gzip"}
        url = "https://{host}/api/v1/{path}".format(host=self.host, path=path)

        log.debug("url: [%s], data: [%s]", url, json.dumps(data))
        from requests import HTTPError

        try:
            return make_json_request(self.service, "POST", url, data=data, validator=validator, headers=headers)
        except HTTPError as e:
            if e.response:
                log.error(
                    "Failed to fetch data from netmon. Error code is %s, error data is %s",
                    e.response.status_code,
                    e.response.text,
                )

            raise
