from abc import ABCMeta, abstractmethod

from sepelib.core import config
from walle.clients.racktables import RacktablesError, shorten_switch_name, _shorten_port_name


class InvalidSwitchPortError(RacktablesError):
    pass


class NetworkClient(metaclass=ABCMeta):
    @classmethod
    def shorten_switch_port_name(cls, switch, port, trust_switch_name=True):
        """Converts full switch/port names to short names used in RackTables.

        :param trust_switch_name: There is no up-to-date list of all switches in our company. NOC may connect nodes to
        experimental switches that aren't supported by RackTables, delete switch from their database before it'll be
        actually demounted and so on. So if we got switch/port from a reliable source we may ignore unknown switch error
        and accept our version of shortened switch/port.
        """

        short_switch = shorten_switch_name(switch)
        short_port = _shorten_port_name(port)
        switches = cls.get_switch_ports()

        try:
            switch_ports = switches[short_switch]
        except KeyError:
            if not trust_switch_name and short_switch not in config.get_value("racktables.experimental_switches"):
                raise InvalidSwitchPortError("{} ({}) switch is unknown.", short_switch, switch)
        else:
            if short_port not in switch_ports:
                raise InvalidSwitchPortError("{} switch doesn't have {} ({}) port.", switch, short_port, port)

        return short_switch, short_port

    @classmethod
    def is_interconnect_switch(cls, switch):
        """Take a switch name. Return true if switch have the "interconnect" flag."""
        return switch in cls._interconnect_switch_set()

    @staticmethod
    @abstractmethod
    def get_switch_ports():
        raise NotImplementedError

    @staticmethod
    @abstractmethod
    def _interconnect_switch_set():
        raise NotImplementedError
