"""RackTables client.

See definitive guide to L3 networks - https://clubs.at.yandex-team.ru/sysadmin/10327
"""

import dataclasses
import json
import logging
import re
import typing as tp

from cachetools.func import ttl_cache
from requests.exceptions import RequestException

from object_validator import DictScheme, Integer, List, Dict
from sepelib.core import constants
from walle import constants as walle_constants
from walle.clients import bot, racktables
from walle.clients.network.network_client import NetworkClient, _shorten_port_name
from walle.clients.utils import iter_csv_response
from walle.errors import NoInformationError, InvalidHostConfiguration
from walle.util import db_cache
from walle.util.cache import cached_with_exceptions
from walle.util.gevent_tools import gevent_idle_iter
from walle.util.validation import String

log = logging.getLogger(__name__)


RACKTABLE_INFINIBAND_INFO_PATH = "/export/IB-sw-to-host-link.txt"


class RacktablesClient(NetworkClient):
    @staticmethod
    @cached_with_exceptions(value_ttl=30 * constants.MINUTE_SECONDS, error_ttl=1)
    def get_switch_ports():
        """Returns a switch -> ports mapping which contains all known switch/ports."""

        log.info("Receiving switch -> ports mapping...")
        response = racktables._raw_request_to_racktables("/export/switchports.txt", stream=True, limited=False)

        switches = {}
        port_name_cache = {}

        try:
            for switch, port in iter_csv_response(response, fields_number=2, ignore_extra_fields=True):
                # These port names can be in longer format. Shorten them for uniformity (we use shot name everywhere else).
                # it should work in requests in both ways https://st.yandex-team.ru/NOCREQUESTS-7240
                # speed up shortening by using the fact that all switches have same port names (ports: 300k, names: 5k)
                try:
                    short_port_name = port_name_cache[port]
                except KeyError:
                    short_port_name = port_name_cache[port] = _shorten_port_name(port)

                switches.setdefault(switch, []).append(short_port_name)
        except RequestException as e:
            racktables._handle_error(e)

        log.info("switch -> ports mapping has been successfully received.")

        return switches

    @staticmethod
    @ttl_cache(maxsize=1, ttl=5 * constants.MINUTE_SECONDS)
    def _interconnect_switch_set():
        # We use this list heavily during netmap loading. We use this list often during `wait_for_active_mac` stage.
        # * we don't want to fetch it from the database for every host in netmap,
        #   so we need to cache it
        # * we don't want to keep this in memory after we've finished netmap syncronization,
        #   so we need a short-term cache
        # * still we don't want to fetch this list from ractables every time we run `wait_for_active_mac` stage
        #   (this list does not change very often), so we want a long-term cache too.
        # So we cache this list into our database and fetch it from there,
        # and hold a short-term cache in memory during heavy usage periods.
        # We can not save a `set` object into the database, so we need a separate method for the long-term cache.
        return set(_get_interconnect_switch_list())

    @staticmethod
    def get_fb_vlans(bb_vlan, location):
        """Determines host's fastbone VLANs by backbone VLAN.

        :raises InvalidHostConfiguration
        """

        vlan_map = get_fb_vlans_map()

        try:
            rt_location = bot.get_rt_location(location)
        except NoInformationError:
            # Per-location fastbones is a very rare hackaround and may be deprecated in the future, so don't give up if we
            # don't know some location.
            pass
        else:
            try:
                return vlan_map[bb_vlan, rt_location]
            except KeyError:
                pass

        try:
            return vlan_map[bb_vlan, None]
        except KeyError:
            raise InvalidHostConfiguration("Backbone VLAN {} doesn't exists in RackTables fastbone VLAN map.", bb_vlan)


@cached_with_exceptions(value_ttl=constants.MINUTE_SECONDS, error_ttl=1)
def get_fb_vlans_map():
    raw_vlan_map = racktables.json_request(
        "/export/fastbone-vlan-map.json",
        limited=False,
        scheme=Dict(String(), List(Integer(min=walle_constants.VLAN_ID_MIN, max=walle_constants.VLAN_ID_MAX))),
    )

    vlan_spec_re = re.compile(
        r"""
        ^(?P<vlan>\d+)
        (?:-(?P<dc>[A-Z]+))?$
    """,
        re.VERBOSE,
    )

    vlan_map = {}

    for vlan_spec, fb_vlans in gevent_idle_iter(raw_vlan_map.items()):
        match = vlan_spec_re.search(vlan_spec)

        try:
            if match is None:
                raise ValueError

            bb_vlan, dc = int(match.group("vlan")), match.group("dc")
            if bb_vlan < walle_constants.VLAN_ID_MIN or bb_vlan > walle_constants.VLAN_ID_MAX:
                raise ValueError
        except ValueError:
            log.error("Got an invalid VLAN specification from fastbone VLAN map: %r.", vlan_spec)
            raise racktables.RacktablesError("Got an invalid data from RackTables.")

        vlan_map[(bb_vlan, dc)] = sorted(set(fb_vlans))

    return vlan_map


@db_cache.cached("interconnect_switch_list", constants.DAY_SECONDS)
def _get_interconnect_switch_list():
    try:
        response = racktables.request(
            "/export/allobjects.php", params={"text": "{interconnect}", "noresolve": "true"}, limited=True
        )
        return list(map(racktables.shorten_switch_name, response.split()))
    except RequestException as e:
        racktables._handle_error(e)


def get_mac_status(mac_address):
    """Check MAC address status. Useful tim find which one is the current active mac address."""
    log.info("Checking status of %s mac address...", mac_address)
    scheme = DictScheme(
        {
            "MAC": String(),
            "Switch": String(optional=True, regex="^[a-z0-9-]+$"),
            "Port": String(optional=True),
            "Port_timestamp": Integer(optional=True),
            "ips": List(
                DictScheme({"IP": String(), "Network": String(), "Location": String(), "VLAN": String()}), optional=True
            ),
        }
    )

    mac_info = racktables.json_request(
        "/export/get-l123.php", params={"q": mac_address, "json": 1}, scheme=scheme, limited=True
    )
    if "Switch" in mac_info and RacktablesClient().is_interconnect_switch(mac_info["Switch"]):
        # No active mac info for MAC address
        return None

    expected_keys = ("MAC", "Switch", "Port_timestamp")
    if all(key in mac_info for key in expected_keys) and mac_info["MAC"].lower() == mac_address.lower():
        return mac_info

    return None


def get_owned_vlans(username):
    info = racktables.request("/export/vlans-owned-by.php", params={"user": username}, limited=True)
    if info:
        return [int(vlan) for vlan in info.strip().split("\n")]
    else:
        return []


@dataclasses.dataclass
class InfinibandInfo:
    host: str
    cluster_tag: str
    ports: tp.Set[str]


def get_hosts_infiniband_info() -> dict[str, InfinibandInfo]:
    info = racktables.request(RACKTABLE_INFINIBAND_INFO_PATH, limited=False)
    res = {}
    for line in info.strip().split("\n"):
        if line:
            parsed = json.loads(line)
            name = parsed["neighbor_name"]
            cluster_tag = parsed["cluster_tag"]
            port = parsed["neighbor_port"]
            if infiniband_host_info := res.get(name):
                assert infiniband_host_info.cluster_tag == cluster_tag
                infiniband_host_info.ports.add(port)
            else:
                res[name] = InfinibandInfo(name, cluster_tag, {port})
    if not res:
        raise racktables.InternalRacktablesError(f"{RACKTABLE_INFINIBAND_INFO_PATH} is empty")
    return res
