"""OAuth client."""

import logging

import requests
from requests.exceptions import RequestException

import walle.clients.utils
import walle.util.misc
from object_validator import String
from sepelib.core import config
from sepelib.core.exceptions import Error
from walle.clients.utils import get_json_response
from walle.errors import RecoverableError
from walle.util.validation import ApiDictScheme

log = logging.getLogger(__name__)


class OauthError(RecoverableError):
    pass


def get_authorization_url():
    """Returns a URL where user can get an authorization code."""

    return _url("authorize") + "?response_type=code&client_id=" + config.get_value("oauth.client_id")


def get_token(authorization_code):
    """Obtains OAuth access token by an authorization code."""

    return _request(
        "token",
        {
            "code": authorization_code,
            "grant_type": "authorization_code",
            "client_id": config.get_value("oauth.client_id"),
            "client_secret": config.get_value("oauth.client_secret"),
        },
        ApiDictScheme({"access_token": String()}),
    )["access_token"]


def _request(method, params, scheme):
    url = _url(method)

    try:
        response = walle.clients.utils.request("oauth", "POST", url, data=params, check_status=False)

        if response.status_code == requests.codes.bad:
            result = get_json_response(
                response,
                check_status=False,
                scheme=ApiDictScheme(
                    {
                        "error": String(),
                        "error_description": String(),
                    }
                ),
            )

            raise OauthError("Authorization failed: {}.", result["error_description"])

        return get_json_response(response, scheme=scheme)
    except RequestException as e:
        raise Error("Error in communication with {}: {}", url, e)


def _url(method):
    return "https://{host}/{method}".format(host=config.get_value("oauth.host"), method=method)
