"""Client for approval gathering service."""
import uuid

import requests

from sepelib.core import config
from walle.errors import RecoverableError

STAFF_IFRAME_TEMPLATE = (
    "{{{{iframe src=\"https://ok.yandex-team.ru/approvements/{uuid}?_embedded=1\""
    " frameborder=0 width=100% height=400px scrolling=no}}}}"
)


class OKError(RecoverableError):
    pass


class OKConnectionError(OKError):
    pass


class OKBadRequest(OKConnectionError):
    pass


class OKCommunicationError(OKError):
    pass


class OKApprovementNotFound(OKError):
    pass


class ApprovementStatus:
    """https://wiki.yandex-team.ru/Intranet/OK/private-api/"""

    REJECTED: str = "rejected"
    IN_PROGRESS: str = "in_progress"
    CLOSED: str = "closed"
    SUSPENDED: str = "suspended"


class ApprovementResolution:
    """https://wiki.yandex-team.ru/Intranet/OK/private-api/"""

    APPROVED: str = "approved"
    DECLINED: str = "declined"
    NO_RESOLUTION: str = ""


class Approvement:
    def __init__(
        self,
        approvement_id,
        text,
        stages,
        uid=None,
        ticket_key=None,
        is_approved=False,
        uuid=None,
        status=None,
        resolution=None,
    ):
        self.id = approvement_id
        self.text = text
        self.stages = stages
        self.uid = uid
        self.ticket_key = ticket_key
        self.is_approved = is_approved
        self.uuid = uuid
        self.status = status
        self.resolution = resolution

    def is_login_approved(self, login):
        return next((True for stage in self.stages if stage["approver"] == login and stage["approved_by"]), False)

    @classmethod
    def from_dict(cls, approvement_dict):
        return cls(
            approvement_id=approvement_dict["id"],
            text=approvement_dict["text"],
            stages=[
                {"approver": stage["approver"], "approved_by": stage["approved_by"]}
                for stage in approvement_dict["stages"]
            ],
            is_approved=approvement_dict["resolution"] == "approved",
            status=approvement_dict["status"],
            resolution=approvement_dict["resolution"],
            # https://wiki.yandex-team.ru/Intranet/OK/private-api/#sozdanieizapusksoglasovanija
            # 'uid' is used here when creating new approvenent, but is not mentioned in API specs, subject to remove.
            uid=approvement_dict.get("uid"),
            ticket_key=approvement_dict.get("object_id"),
            uuid=approvement_dict.get("uuid"),
        )

    @property
    def iframe(self):
        return STAFF_IFRAME_TEMPLATE.format(uuid=self.uuid)


class OkClient:
    PRODUCTION_BASE_URL = "https://ok.yandex-team.ru/api"
    TESTING_BASE_URL = PRODUCTION_BASE_URL

    def __init__(self, base_url, token):
        self._base_url = base_url
        self._token = token
        self._session = requests.Session()

    def _make_request(self, method, path, uuid, **kwargs):
        headers = kwargs.pop('headers', None) or {}
        headers["Authorization"] = "OAuth {}".format(self._token)
        headers["Content-Type"] = "application/json"
        path = "{}{}/".format(self._base_url, path)

        try:
            response = self._session.request(method, path, headers=headers, **kwargs)
        except requests.RequestException as e:
            raise OKConnectionError(
                "Error while connecting to OK server: {}.".format(e), method=method, path=path, **kwargs
            )

        if not response.ok:
            if response.status_code == requests.codes.not_found:
                raise OKApprovementNotFound("Approvement with id {} does not exist.".format(uuid), uuid=uuid)
            elif response.status_code == requests.codes.bad_request:
                raise OKBadRequest("Server responded with 400 Bad Request: " + response.text)
            else:
                raise OKConnectionError(
                    f"Error while communicating with OK: ID={uuid}, HTTP code {response.status_code}, message: '{response.text}'."
                )

        if response.status_code in (requests.codes.ok, requests.codes.created):
            try:
                return response.json()
            except ValueError:
                raise OKConnectionError(
                    "Got an invalid response from OK server: {}".format(response.text),
                    method=method,
                    path=path,
                    **kwargs,
                )

    def create_approvement(self, approvers, ticket_key, text, groups=None, author=None):
        uid = self._generate_new_uid()
        params = {
            "object_id": ticket_key,
            "uid": uid,
            "text": text,
            "is_parallel": True,
            "stages": [{"need_all": False, "stages": [{"approver": approver} for approver in approvers]}],
        }
        if groups is not None:
            params["groups"] = [group for group in groups]
        if author is not None:
            params["author"] = author
        result = self._make_request("POST", "/approvements", uid, json=params)
        # OK API does not return uid and ticket, fill from request
        result.update({"ticket_key": ticket_key, "uid": uid})

        return Approvement.from_dict(result)

    def edit_approvers(self, uuid, approvers):
        params = {"stages": [{"need_all": False, "stages": [{"approver": approver} for approver in approvers]}]}
        result = self._make_request("PUT", "/approvements/{}/edit".format(uuid), uuid, json=params)

        return Approvement.from_dict(result)

    def close_approvement(self, uuid):
        return self._make_request("POST", "/approvements/{}/close".format(uuid), uuid, json={})

    def get_approvement(self, uuid):
        return Approvement.from_dict(self._make_request("GET", "/approvements/{}".format(uuid), uuid))

    @staticmethod
    def _generate_new_uid():
        return uuid.uuid4().hex


def get_client():
    api = config.get_value("ok.api")
    token = config.get_value("ok.access_token")
    url = getattr(OkClient, api.upper() + "_BASE_URL")
    return OkClient(url, token)
