"""Qloud client.

Production (External)(qloud-ext): https://qloud-ext.yandex-team.ru
Prestable (Production)(qloud): https://qloud.yandex-team.ru
Testing (Testing)(qloud-test): https://qloud-test.yandex-team.ru
Host resolver:
"""


import logging

import requests

import walle.clients.utils
from sepelib.core import config
from walle.errors import RecoverableError, FixableError

log = logging.getLogger(__name__)


class QloudConnectionError(RecoverableError):
    pass


class QloudHostInfoError(FixableError):
    pass


class QloudHostNotFound(RecoverableError):
    pass


class CantResolveInstallationApi(FixableError):
    pass


class CantResolveProjectFromSegment(FixableError):
    pass


class QloudHostStates:
    DOWN = "DOWN"
    SUSPECTED = "SUSPECTED"
    INITIAL = "INITIAL"
    MAINTENANCE = "MAINTENANCE"
    PREPARE_MAINTENANCE = "PREPARE_MAINTENANCE"
    PROBATION = "PROBATION"
    UNRECOGNIZED = "UNRECOGNIZED"
    UP = "UP"
    ALL_STATES = {DOWN, SUSPECTED, INITIAL, MAINTENANCE, PREPARE_MAINTENANCE, PROBATION, UP}


def split_segment(segment):
    dot_index = segment.index('.')
    return segment[0:dot_index], segment[dot_index + 1 :]


class ResolverHostInfo:
    def __init__(self, name, data):
        self.name = name
        if data:
            self.installation = data.get("installation", None)
            self.segment = data.get("segment", None)
        else:
            self.installation = None
            self.segment = None


class QloudHost:
    def __init__(self, installation=None, data=None):
        self.segment = '{installation}.{segment}'.format(installation=installation, segment=data.get("segment", ""))
        self.state = data.get("state", None)
        self.installation = installation
        self.slots_count = len(data.get("slots"))
        self.name = data.get("fqdn", None)

        # Calculating is qloud knows enough about host
        memory = data.get("memoryBytes", None)
        cpu = data.get("cpu", None)
        dc = data.get("dataCenter", None)
        noc_dc = data.get("nocDC", None)
        line = data.get("line", None)
        rack = data.get("rack", None)
        ssd = 0
        hdd = 0
        for disk in data.get("disks", []):
            if 'allocationMountPoint' in disk and len(disk['allocationMountPoint']) > 0:
                if disk["type"].upper() == "SSD":
                    ssd += float(disk["size"]) / 1024.0 / 1024.0 / 1024.0
                elif disk["type"].upper() == "HDD":
                    hdd += float(disk["size"]) / 1024.0 / 1024.0 / 1024.0
                else:
                    raise QloudHostInfoError("Weird disk {} on host {}".format(disk["type"].upper(), self.name))
        # This field is used to determine if host is ready for service allocations
        self.is_data_filled = (
            noc_dc is not None
            and dc is not None
            and line is not None
            and rack is not None
            and cpu is not None
            and cpu != 0
            and memory is not None
            and memory != 0
            and (ssd is not None and ssd != 0 or hdd is not None and hdd != 0)
        )


class QloudClient:
    def __init__(self, token, api_map, host_resolver_api):
        self._api_map = api_map
        self._host_resolver_url = host_resolver_api
        self._token = token

    def _make_qloud_request(self, installation, method, path, params=None, **kwargs):
        try:
            (response, result) = walle.clients.utils.json_request(
                "qloud",
                method,
                "https://{}/api/v1{}".format(self._api_map[installation], path),
                headers={"Authorization": "OAuth {auth}".format(auth=self._token)},
                params=params,
                success_codes=(
                    requests.codes.ok,
                    requests.codes.created,
                    requests.codes.accepted,
                    requests.codes.no_content,
                ),
                with_response=True,
                allow_errors=(404,),
                **kwargs
            )
            if response.status_code == 404:
                return None
            return result
        except requests.RequestException as e:
            raise QloudConnectionError("Error while connecting to Qloud installation {}: {}.".format(installation, e))

    def _make_resolver_request(self, method, path, **kwargs):
        try:
            (response, result) = walle.clients.utils.json_request(
                "qloud",
                method,
                "http://{}/v1{}".format(self._host_resolver_url, path),
                success_codes=(requests.codes.ok,),
                allow_errors=(404,),
                with_response=True,
                **kwargs
            )
            if response.status_code == 404:
                return None
            return result
        except requests.RequestException as e:
            raise QloudConnectionError("Error while connecting to Qloud Host Resolver: {}.".format(e))

    def _get_installation_api(self, installation):
        if installation in self._api_map:
            return self._api_map[installation]
        raise CantResolveInstallationApi("Can't resolve qloud installation api: {}".format(installation))

    def update_qloud_host_metadata(self, host):
        return self._make_qloud_request(
            installation=host.installation, method="POST", path="/admin/hostupdate", data=[host.name]
        )

    def update_host_metadata(self, hostname):
        host = self.find_host(hostname)
        if host:
            return self.update_qloud_host_metadata(host)

    def add_host(self, hostname, segment):
        installation, qloud_segment = split_segment(segment)
        result = self._make_qloud_request(
            installation=installation,
            method="POST",
            path="/hosts",
            params={"boxFqdn": hostname, "hardwareSegment": qloud_segment},
        )
        return result

    def remove_qloud_host(self, host):
        result = self._make_qloud_request(
            installation=host.installation, method="DELETE", path="/hosts/{}".format(host.name)
        )
        return result

    def _get_host(self, installation, hostname):
        result = self._make_qloud_request(installation, "GET", "/hosts/search", params={"fqdn": hostname})
        if result is None or len(result) != 1:
            return None
        return QloudHost(installation, result[0])

    def find_host_installation_segment(self, hostname):
        try:
            host_info = ResolverHostInfo(hostname, self._make_resolver_request("GET", "/host/{}".format(hostname)))
            return host_info.installation, host_info.segment
        except (QloudConnectionError, ValueError, AttributeError):
            log.exception("Qloud Resolver error")
            return None, None

    def find_host(self, hostname):
        installation, segment = self.find_host_installation_segment(hostname)
        if installation:
            return self._get_host(installation, hostname)
        for installation in sorted(self._api_map.keys()):
            host = self._get_host(installation, hostname)
            if host:
                return host
        return None

    def set_qloud_host_state(self, host, state, comment=None):
        if not comment:
            comment = "State changed by wall-e from {} to {}".format(host.state, state)
        if host.state == state:
            return None
        result = self._make_qloud_request(
            installation=host.installation,
            method="PUT",
            path="/hosts/{name}".format(name=host.name),
            params={"state": state, "comment": comment},
        )
        host.state = state
        return result

    def get_host_state(self, hostname):
        host = self.find_host(hostname)
        if not host:
            raise QloudHostNotFound("Qloud host {} not found".format(hostname))
        return host.state

    def set_host_state(self, hostname, state, comment=None):
        host = self.find_host(hostname)
        if not host:
            raise QloudHostNotFound("Qloud host {} not found".format(hostname))
        return self.set_qloud_host_state(host, state, comment)


def get_project_tag():
    return config.get_value("qloud.project_tag")


def get_segment_project(segment):
    segment_project_map = config.get_value("qloud.segment_project_map")
    # Special project for special segments
    if segment in segment_project_map:
        return segment_project_map[segment]

    # Default project for installation
    installation, installation_segment = split_segment(segment)
    if installation in segment_project_map:
        return segment_project_map[installation]
    raise CantResolveProjectFromSegment("Can't resolve project from segment: {}".format(segment))


def get_client():
    token = config.get_value("qloud.access_token")
    api_map = config.get_value("qloud.api_map")
    resolver_api = config.get_value("qloud.host_resolver_api")
    client = QloudClient(token, api_map, resolver_api)
    return client
