"""RackTables client.

See definitive guide to L3 networks - https://clubs.at.yandex-team.ru/sysadmin/10327
"""

import logging
import re
from collections import defaultdict, namedtuple
from ipaddress import IPv6Network, ip_network, IPv6Interface, IPv4Network, IPv4Address
from xml.etree import ElementTree

import requests
import six
from cachetools.func import ttl_cache
from gevent.lock import Semaphore
from requests.exceptions import RequestException

import walle.clients.utils
import walle.hosts
from sepelib.core import config, constants
from sepelib.core.exceptions import Error
from walle import constants as walle_constants
from walle.clients.utils import iter_csv_response, strip_api_error
from walle.errors import RecoverableError
from walle.util.cache import cached_with_exceptions
from walle.util.gevent_tools import gevent_idle_iter

log = logging.getLogger(__name__)

_NETMAP_INACCURACY = constants.HOUR_SECONDS
"""Netmap timestamps have ~ one hour inaccuracy (the timestamp may be bigger of the actual value)."""

_L3_PER_PORT_VLANS = (688, 788)
"""These VLANs are expected to have /57 networks for switch because they have /64 network per port."""

L3_SWITCH_VLAN_NETWORK_SIZE = 2
L3_SWITCH_VLAN_NETWORK_TTL = constants.MINUTE_SECONDS

SWITCH_DOMAINS = [".yndx.net", ".netinfra.cloud.yandex.net"]


class RacktablesError(RecoverableError):
    pass


class InvalidSwitchPortError(RacktablesError):
    pass


class PersistentRacktablesError(RacktablesError):
    def __init__(self, error):
        super().__init__("RackTables returned an error: {}", strip_api_error(error))


class MtnNotSupportedForSwitchError(PersistentRacktablesError):
    pass


class InternalRacktablesError(RacktablesError):
    def __init__(self, message, *args):
        super().__init__("Error in communication with RackTables: " + message, *args)


RacktablesSwitchInfo = namedtuple("RacktablesSwitchInfo", ["switch", "port", "int_mac", "timestamp"])
L3SegmentInfo = namedtuple("L3SegmentInfo", ["network", "fb_vlans"])
VlanArgs = namedtuple("VlanArgs", ["raw_request", "path", "service_name"])


@cached_with_exceptions(value_ttl=constants.MINUTE_SECONDS, error_ttl=1)
def get_l3_segments_map():
    l3_segments_map = {}

    file_name = "l3-segments.xml"
    response = request("/export/" + file_name, limited=False)

    try:
        root = ElementTree.fromstring(six.ensure_str(response, "utf-8"))
        if root.tag != "data":
            raise Error("Invalid root element: {}.", root.tag)

        for network in gevent_idle_iter(root):
            if network.tag != "network":
                raise Error("Invalid element instead of expected network element: {}.", network.tag)

            try:
                network_name = network.attrib["match"]
                if not network_name:
                    raise ValueError

                bb_vlan = _parse_vlan_id(network.attrib["vlan"])
            except (ValueError, KeyError):
                raise Error("Invalid network element: {}.", network.attrib)

            datacenter = network.find("datacenter")
            if datacenter is None:
                raise Error("Network {} doesn't have a datacenter element.", network.attrib)

            dc = datacenter.text
            if not dc:
                raise Error("Datacenter {} of {} network has an empty name.", datacenter.attrib, network.attrib)

            fb_vlans = set()
            for fastbone in network.findall("fastbone"):
                try:
                    fb_vlan = _parse_vlan_id(fastbone.attrib["vlan"])
                except (ValueError, KeyError):
                    raise Error("Invalid fastbone element: {}.", network.attrib)

                fb_vlans.add(fb_vlan)

            l3_segment = L3SegmentInfo(network=network_name, fb_vlans=tuple(sorted(fb_vlans)))
            l3_segments_map.setdefault((dc, bb_vlan), []).append(l3_segment)
    except Exception as e:
        log.error("Failed to parse RackTables': %s", e)
        raise RacktablesError("Failed to parse RackTables' {}.", file_name)

    return l3_segments_map


def get_vlan_networks(switch, vlan, vlan_scheme=None):
    """Determines IPv6 network by switch + VLAN.
    Return list of networks assigned to the switch+vlan pair.

    Attention: This doesn't work for switches witch have one network per port.

    """
    # Don't try to optimize and skip l3 by switch name as datacenters migrating away from l2
    # and we don't want to update maps every time they drift.
    l3_networks = _get_l3_switch_vlan_networks(vlan_scheme)
    networks = l3_networks.get(switch, vlan)

    if not networks:
        l2_networks = _get_network_layout()
        networks = l2_networks.get(switch, vlan)

    return networks or None


def is_l3_switch(switch):
    """Determine whether given switch is located in l3 segment or not
    by checking it's appearance in the l3 networks map.

    NB: may return false negative for switch without any known l3 tors.
    """
    return any(
        _get_l3_switch_vlan_networks(vlan_scheme).have_switch(switch)
        for vlan_scheme in [walle_constants.VLAN_SCHEME_SEARCH, walle_constants.VLAN_SCHEME_CLOUD]
    )


def is_vlan_available(switch, vlan):
    """Check network layout and return whether requested vlan is available for switch's domain or not."""
    return _get_network_layout().is_vlan_available(switch, vlan)


class _L3NetworkMap:
    """Represents direct switch+vlan -> l3-network mapping with some additional useful data.
    Exposes similar interface to _L2NetworkMap, try to keep up.
    """

    def __init__(self, switches, network_map):
        self._switches = switches
        self._network_map = network_map

    def get(self, switch, vlan):
        return self._network_map[(switch, vlan)]  # defaultdict, returns empty list by default

    def have_switch(self, switch):
        return switch in self._switches


class _L2NetworkMap:
    """Make this map little bit more usable.
    The user have a switch name and vlan number while this map maps vlans to datacenters.
    We incapsulate this fact and transparently map switches to datacenters here.

    Exposes similar interface to _L3NetworkMap, try to keep up.
    """

    def __init__(self, switch_dc_map, dc_vlan_map, network_map):
        self._network_map = network_map
        self._dc_vlan_map = dc_vlan_map
        self._switch_dc_map = switch_dc_map

    def get(self, switch, vlan):
        dc = self.get_dc(switch)
        return self._network_map[(dc, vlan)]  # defaultdict, returns empty list by default

    def get_dc(self, switch):
        try:
            return self._switch_dc_map[switch]
        except KeyError:
            return None

    def have_switch(self, switch):
        return switch in self._switch_dc_map

    def is_vlan_available(self, switch, vlan):
        dc = self.get_dc(switch)
        return vlan in self._dc_vlan_map[dc]  # defaultdict, returns empty list by default


@ttl_cache(L3_SWITCH_VLAN_NETWORK_SIZE, L3_SWITCH_VLAN_NETWORK_TTL)
def _get_l3_switch_vlan_networks(vlan_scheme):
    """Returns switch+VLAN -> L3 networks mapping.

    Выжимка из переписки с alan@:

    Прежде чем VLAN'ом можно будет шелкать, он должен быть разрешен. VLANы разрешаются не для каждого свитча, а сразу
    для всего ДЦ. Это происходит при заказе сети. Выгрузки имеющихся VLAN'ов нет.

    Для L3-сетей:

    Если пользователь просит переключить в VLAN один из портов, то, если данный свитч переключается в этот VLAN впервые,
    после переключения порта для пары switch+VLAN будет аллоцированна /64 сеть. Сеть закрепляется за switch+VLAN
    навсегда и попадает в https://racktables.yandex.net/export/l3-tors.txt.

    Отдаваться по RA сеть начнет через 2-5 минут после заказа переключения и через 1-3 минуты после появления флага
    synced в https://racktables.yandex.net/export/get-vlanconfig-by-port.php. Если во время выделения сети происходит
    какая-то ошибка, то это просто увеличивает время ее выделения, но не отменяет - как только проблему починят руками,
    сеть будет выделена.

    https://racktables.yandex.net/export/l3-tors.txt обновляется раз в 10 минут.

    Получить список VLAN'ов которые сконфигурированы для L3 в принципе при желании можно из
    https://racktables.yandex.net/export/l3-segments.xml (по паре VLAN+ДЦ).
    """

    log.info("Receiving L3 networks mapping...")
    vlan_info = map_vlan_name_to_l3_networks(vlan_scheme)
    response = vlan_info.raw_request(vlan_info.path, limited=False, stream=True)

    networks = defaultdict(list)
    switches = set()

    try:
        for switch, vlan, tor_interface in iter_csv_response(response, fields_number=3, ignore_extra_fields=False):
            try:
                if not switch:
                    raise ValueError

                vlan = int(vlan)
                if vlan < walle_constants.VLAN_ID_MIN or vlan > walle_constants.VLAN_ID_MAX:
                    raise ValueError

                # interface is basically a network address combined with it's default gateway e.g 2a02:6b8:c0c:d80::1/64
                network_obj = IPv6Interface(tor_interface).network

                # We expect to have only /64 networks here.
                # But we need to properly check network availability for _L3_PER_PORT_VLANS, which have /57 networks,
                # so we just store first /64 subnet for them.
                # We need to make some adjustments if we ever need to assign DNS addresses for such networks.
                if network_obj.prefixlen == 57 and vlan in _L3_PER_PORT_VLANS:
                    network = next(network_obj.subnets(new_prefix=64)).compressed
                elif network_obj.prefixlen == 64:
                    network = network_obj.compressed
                else:
                    raise ValueError  # we expect here only /64 networks.
            except ValueError:
                log.error(
                    "Got an unexpected data from %r: switch=%r, vlan=%r, tor_ip=%r.",
                    vlan_info.service_name,
                    switch,
                    vlan,
                    tor_interface,
                )
                raise RacktablesError("Got an invalid data from {}.".format(vlan_info.service_name))

            map_key = (switch, vlan)
            networks[map_key].append(network)
            switches.add(switch)
    except RequestException as e:
        _handle_error(e)

    log.info("L3 networks mapping has been successfully received.")

    return _L3NetworkMap(switches, networks)


@cached_with_exceptions(value_ttl=5 * constants.MINUTE_SECONDS, error_ttl=1)
def _get_network_layout():
    """Return a dict-like object that contains the switch+vlan -> L2-network map.
    The underlying file updates every hour.
    """
    try:
        switch_dc_map, networks = _fetch_l2_dc_switch_vlan_networks()

        if not switch_dc_map:
            raise Error("Invalid export data from Racktables: net-layout.xml does not contain any switches.")

        dc_vlan_map, network_map = _build_l2_dc_networks_map(networks)

        if not network_map:
            raise Error("Invalid export data from Racktables: net-layout.xml does not contain any networks.")

    except Exception as e:
        log.warning("Failed to receive l2 networks from RackTables' net-layout.xml: %s", e)
        raise RacktablesError("Failed to receive l2 networks from RackTables' net-layout.xml.")

    return _L2NetworkMap(switch_dc_map, dc_vlan_map, network_map)


def _fetch_l2_dc_switch_vlan_networks():
    log.info("Receiving L2 network mapping...")
    response = request("/export/net-layout.xml", limited=False)

    root = ElementTree.fromstring(six.ensure_str(response, "utf-8"))

    if root.tag != "layout":
        raise Error("Invalid root element: {}.", root.tag)

    switch_dc_map = _get_l2_switch_dc_map(root.iter("switch"))
    networks = _get_l2_vlan_network_map(root.iter("network"))

    log.info("L2 network mapping has been successfully received.")

    return switch_dc_map, networks


def _get_l2_switch_dc_map(switches):
    switch_dc_map = {}

    for switch in switches:
        try:
            name = switch.attrib["name"]
            domain_id = int(switch.attrib["domain_id"])
        except KeyError as e:
            log.warning("Got invalid 'switch' tag: %s/%s", switch.attrib, switch.text.strip())
            raise Error("Invalid 'switch' tag in /layout/switches: attribute '{}' is missing.", e)

        if name in switch_dc_map:
            raise Error("Invalid network layout data: switch {} has multiple locations.", name)

        switch_dc_map[name] = domain_id
    return switch_dc_map


def _get_l2_vlan_network_map(networks):
    for network in networks:
        try:
            net_address_str = str(network.attrib["addr"])
        except KeyError as e:
            log.warning("Got invalid 'network' tag: %s/%s", network.attrib, network.text.strip())
            raise Error("Invalid 'network' tag in /layout/networks: attribute '{}' is missing.", e)

        for vlan_tag in network.iter("vlan"):
            try:
                vlan = _parse_vlan_id(vlan_tag.attrib["vlan_id"])
                domain_id = int(vlan_tag.attrib["domain_id"])
            except KeyError as e:
                log.warning("Got invalid 'vlan' tag: %s/%s", vlan_tag.attrib, vlan_tag.text.strip())
                raise Error("Invalid 'vlan' tag in /layout/networks/network: attribute '{}' is missing.", e)
            except ValueError as e:
                log.warning("Got invalid 'vlan' tag: %s/%s", vlan_tag.attrib, vlan_tag.text.strip())
                raise Error("Invalid 'vlan' tag in /layout/networks/network: {}", e)

            yield domain_id, vlan, net_address_str


def _build_l2_dc_networks_map(networks):
    """Run trough the network tags and collect them into two structures:
    * dc, vlan -> network mapping containing only IPv6 /64 networks
    * dc -> vlan mapping containing all vlans available for domain
    """
    networks_map = defaultdict(list)
    available_vlans = defaultdict(list)
    deprecated_networks = {IPv6Network(str(net)) for net in config.get_value("racktables.deprecated_networks")}

    for dc, vlan, net_address_str in networks:
        # Here we are adding all vlans to the map, thus allowing the use of IPv4-only vlans in vlan schemes.
        # Still, there is no support for IPv4-only hosts and no dns support for IPv4-only vlans.
        available_vlans[dc].append(vlan)

        network = ip_network(str(net_address_str))
        if network.version == 4:
            # IPv4 network. We don't need them.
            continue

        if network.prefixlen != 64:
            # aggregated network, it shall split into /64 networks once vlan is assigned to a switch in the domain.
            # but we have to ignore it now.
            continue

        map_key = (dc, vlan)
        # Deprecated networks go after actual networks.
        # Assuming we have only one "actual" network for data center in any given VLAN.
        if network in deprecated_networks:
            networks_map[map_key].append(network.compressed)
        else:
            networks_map[map_key].insert(0, network.compressed)

    return available_vlans, networks_map


def get_port_vlan_status(switch, port):
    response = request("/export/get-vlanconfig-by-port.php", params={"switch": switch, "port": port}, limited=True)

    try:
        result = response.strip().split("\n")
        if len(result) != 2 or result[1] not in ("yes", "no"):
            raise ValueError

        vlans, native_vlan = _parse_vlan_specification(result[0])
    except ValueError:
        log.error("Got an invalid response from RackTables: %r.", response)
        raise RacktablesError("Got an invalid response from RackTables.")

    return vlans, native_vlan, result[1] == "yes"


def _parse_vlan_specification(vlan_spec):
    # Из общения с alan@:
    #
    # access - подается только один VLAN без тега.
    # trunk - может быть много VLAN, из них какой-то может быть без тега.
    #
    # trunk с единственным VLAN *без тега* функционально неотличим от access.
    #
    # Если у нас trunk, и все VLAN'ы тегированные, то свитч будет дропать все нетегированные пакеты.
    #
    # API поддерживает диапазоны VLAN'ов, но по факту на портах конечных машин больших диапазонов не бывает, так что
    # при отправке запросов на смену VLAN'ов можно их не схлопывать.
    #
    # На оборудовании встречаются и другие типы, но RT их не поддерживает и в нем могут быть только access и trunk. Но
    # если надо один VLAN без тега, лучше делать access, а не trunk. Визуально оно смотрится лучше и в RT, и на
    # оборудовании.

    vlan_spec = re.sub(r"\s", "", vlan_spec)
    mode, vlan_list = vlan_spec[:1], vlan_spec[1:]

    if mode not in "AT":
        raise ValueError

    plus_count = vlan_list.count("+")

    if plus_count > 1:
        raise ValueError
    elif plus_count:
        native, vlan_list = vlan_list.split("+")
        native = _parse_vlan_id(native) if native else None
        vlans = _parse_vlan_list(vlan_list)
    else:
        native = _parse_vlan_id(vlan_list) if vlan_list else None
        vlans = []

    if native is not None:
        vlans.append(native)
    vlans = sorted(set(vlans))

    if mode == "A" and (native is None or len(vlans) > 1):
        raise ValueError

    return vlans, native


def _parse_vlan_list(vlan_list):
    vlans = []

    if not vlan_list:
        return vlans

    for vlan in vlan_list.split(","):
        range_count = vlan.count("-")
        if range_count > 1:
            raise ValueError

        if range_count:
            vlan_range = vlan.split("-")
            start_vlan, end_vlan = _parse_vlan_id(vlan_range[0]), _parse_vlan_id(vlan_range[1])
            if start_vlan > end_vlan:
                raise ValueError

            vlans.extend(range(start_vlan, end_vlan + 1))
        else:
            vlans.append(_parse_vlan_id(vlan))

    return vlans


def _parse_vlan_id(vlan):
    return _check_vlan_id(int(vlan))


def _check_vlan_id(vlan):
    if vlan < walle_constants.VLAN_ID_MIN or vlan > walle_constants.VLAN_ID_MAX:
        raise ValueError

    return vlan


def switch_vlans(switch, port, vlans, native_vlan=None):
    params = {"switchname": switch, "portname": port}
    params.update(_get_update_vlan_query(vlans, native_vlan))

    request("/export/vlanrequest.php", params=params, limited=True)


def _get_update_vlan_query(vlans, native_vlan=None):
    query_string = {}

    allowed = set(vlans)

    if native_vlan is not None:
        query_string["native"] = native_vlan
        allowed.discard(native_vlan)

    query_string["mode"] = "access" if native_vlan is not None and not allowed else "trunk"
    if allowed:
        query_string["allowed[]"] = sorted(allowed)

    return query_string


def get_port_project_status(switch, port):
    """Take switch and port names and return project id associated with them.
    Return a tuple containing project id and "synced" flag.

    Project id might be None if no project id is associated with port.
    In that case sync flag is always True due to lack of information from racktables.
    """
    response = request("/export/get-projectid-by-port.php", params={"switch": switch, "port": port}, limited=True)
    project_id, synced = None, None

    try:
        result = response.strip().split("\n")
        if len(result) == 1 and result[0] == "None":
            synced = True

        elif len(result) == 2 and result[0] in ("yes", "no"):
            project_id, macro_name = _parse_mtn_project_specification(result[1])
            synced = result[0] == "yes"
        else:
            raise ValueError

    except ValueError:
        log.error("Got an invalid response from RackTables: %r.", response)
        raise RacktablesError("Got an invalid response from RackTables.")

    return project_id, synced


def _parse_mtn_project_specification(spec):
    prj, macro_name = spec.split()
    prj = hex(int(prj, 16)).lower()  # trigger value checking

    if not macro_name.startswith("_") or not macro_name.endswith("_"):
        raise ValueError  # we've got some very suspicious macro name

    return prj, macro_name


def switch_mtn_project(switch, port, mtn_project_id):
    params = {"switchname": switch, "portname": port, "set_project": mtn_project_id}
    request("/export/vlanrequest.php", params=params, limited=True)


def delete_mtn_project(switch, port, mtn_project_id):
    params = {"switchname": switch, "portname": port, "delete_project": mtn_project_id}
    request("/export/vlanrequest.php", params=params, limited=True)


@cached_with_exceptions(value_ttl=constants.HOUR_SECONDS, error_ttl=1)
def get_hbf_projects():
    """Return dictionary mapping hbf project ids to their macro names."""
    log.info("Receiving hbf projects list...")
    try:
        response = _raw_request_to_racktables("/export/vm-projects.txt", limited=False)
        data = {int(prj, 16): macro for macro, prj in (line.strip().split("\t") for line in response.text.splitlines())}
    except RequestException as e:
        raise _handle_error(e)

    except (AttributeError, ValueError):
        log.error("Got invalid data from racktables for vm-projects.txt export file.")
        raise RacktablesError("Got invalid data from racktables for vm-projects.txt export file.")

    return data


def is_nat64_network(address):
    """Take an IPv4 address and return True if it belongs to nat64 networks."""
    address = IPv4Address(address)
    return any(address in network for network in _get_v4_to_v6_networks())


@cached_with_exceptions(value_ttl=constants.HOUR_SECONDS, error_ttl=1)
def _get_v4_to_v6_networks():
    """Return a list of IPv4 networks that used for IPv6-to-IPv4 tunnels.
    Wall-E shall ignore IP-addresses from these networks in the "search" VLAN-scheme.
    """
    log.info("Receiving list of nat/tunnel IPv4 addresses...")
    address_list = []

    for report_type in ("nat64", "tun64"):
        try:
            response = _raw_request_to_racktables(
                "/export/networklist.php", params={"report": report_type}, limited=False
            )
            address_list.extend(IPv4Network(str(net)) for net in response.text.splitlines())
        except RequestException as e:
            raise _handle_error(e)

        except ValueError as e:
            log.error("Got invalid data from racktables for networklist.php?report=%s: %s.", report_type, str(e))
            raise RacktablesError("Got invalid data for networklist.php ({} networks).", report_type)

    return address_list


def shorten_switch_name(switch):
    switch = switch.lower()
    for switch_domain in SWITCH_DOMAINS:
        if switch.endswith(switch_domain):
            return switch[: -len(switch_domain)]
    return switch


def _shorten_port_name(port):
    """Converts a full port name to short port name used in RackTables.

    The function is inspired by
    https://github.yandex-team.ru/YC2/neutron-yandex/blob/aaa5748010d2c6f35f084912960ab7dcec377e25/neutron_yandex/agent/utils.py#L51
    which is a port of
    https://github.com/RackTables/racktables/blob/4d98ddad0103466810fd82d13bfcd58a9046d67e/wwwroot/inc/remote.php#L785
    from PHP to Python with the following changes:
    * Do not shorten GExxx to gixxx

    See the details here - https://st.yandex-team.ru/WALLE-513.
    """

    match = re.match(r"^eth-trunk(\d+)$", port, flags=re.I)
    if match:
        return "Eth-Trunk{}".format(match.group(1))

    port = re.sub(r"^(?:[Ee]thernet|Eth)(.+)$", r"e\1", port)
    port = re.sub(r"^FastEthernet(.+)$", r"fa\1", port)
    port = re.sub(r"^GigabitEthernet\s*(.+)$", r"gi\1", port)
    port = re.sub(r"^TenGigabitEthernet(.+)$", r"te\1", port)
    port = re.sub(r"^port-channel(.+)$", r"po\1", port, flags=re.I)
    port = re.sub(r"^(?:XGigabitEthernet|XGE)(.+)$", r"xg\1", port)
    port = re.sub(r"^LongReachEthernet(.+)$", r"lo\1", port)
    port = re.sub(r"^ManagementEthernet\s(.+)$", r"ma\1", port)
    port = re.sub(r"^Et(\d.*)$", r"e\1", port)
    port = re.sub(r"^TenGigE(.*)$", r"te\1", port)
    port = re.sub(r"^Mg(?:mtEth)?(.*)$", r"mg\1", port)
    port = re.sub(r"^BE(\d+)$", r"bundle-ether\1", port)

    port = port.lower()
    port = re.sub(r"^(e|fa|gi|te|po|xg|lo|ma)\s+(\d.*)", r"\1\2", port)

    return port


def request(path, limited, params=None, stream=False, timeout=None):
    response = _raw_request_to_racktables(path, limited, params=params, stream=stream, timeout=timeout)
    return _check_response(response).text


def json_request(path, limited, params=None, scheme=None, timeout=None):
    try:
        response = _raw_request_to_racktables(path, limited, params=params, timeout=timeout)
        return walle.clients.utils.get_json_response(response, check_content_type=False, scheme=scheme)
    except RequestException as e:
        _handle_error(e)


racktables_semaphore = Semaphore(value=2)


def _raw_request_to_racktables(path, limited, params=None, stream=False, timeout=None):
    # RackTables has rate limits for its *.php API handlers:
    # * 2 concurrent requests to all API handlers from one host
    #
    # 429 HTTP status code is returned when either of the limits is exceeded.

    if limited:
        if timeout is None:
            timeout = 3

        racktables_semaphore.acquire(timeout=15)

    try:
        headers = {"Authorization": "OAuth {}".format(config.get_value("racktables.access_token", ""))}
        return walle.clients.utils.request(
            "racktables",
            "GET",
            _url_to_racktables(path),
            params=params,
            headers=headers,
            stream=stream,
            error_from_contents=True,
            timeout=timeout,
        )
    except RequestException as e:
        _handle_error(e)
    finally:
        if limited:
            racktables_semaphore.release()


def _url_to_racktables(path):
    return "https://" + config.get_value("racktables.host") + path


def _check_response(response):
    error_prefix = "ERROR: "
    body = response.text.strip()

    if body.startswith(error_prefix):
        error = body[len(error_prefix) :]
        if error.startswith("no project id vlans on switchport"):
            raise MtnNotSupportedForSwitchError(error)
        raise PersistentRacktablesError(error)

    return response


def _handle_error(error):
    response = error.response

    if response is not None:
        body = response.text.strip()

        if response.status_code == requests.codes.forbidden:
            if body:
                error_message = strip_api_error(body)
            else:
                error_message = "{} {}.".format(response.status_code, response.reason)

            raise PersistentRacktablesError(error_message)
        elif response.status_code == requests.codes.bad_request:
            # Request validation errors may look like:
            # "Argument error: Argument 'portname' of value &#039;&#039; is invalid (parameter is missing).
            #  Usage:..."
            #
            # But we should treat some errors as persistent. Examples:
            # "An ERROR occured processing this directive.
            #  Object "sas1-s274" does not exist or is not a VLAN switch."
            # "An ERROR occured processing this directive.\nMalformed request:
            #  VLAN604 being added does not exist in the VLAN domain of the switch"
            error_prefix = "An ERROR occured processing this directive."
            if body.startswith(error_prefix):
                error_message = body[len(error_prefix) :]
                raise PersistentRacktablesError(error_message)

        _check_response(response)

    raise InternalRacktablesError("{}", error)


def map_vlan_name_to_l3_networks(vlan_name):
    if vlan_name == walle_constants.VLAN_SCHEME_CLOUD:
        return VlanArgs(_raw_request_to_cloud, "/netbox", "Cloud")
    else:
        return VlanArgs(_raw_request_to_racktables, "/export/l3-tors.txt", 'Racktables')


def _raw_request_to_cloud(path, limited=None, params=None, stream=False, timeout=None):
    try:
        return walle.clients.utils.request(
            "cloud", "GET", _url_to_cloud(path), params=params, stream=stream, error_from_contents=True, timeout=timeout
        )
    except RequestException as e:
        _handle_error(e)


def _url_to_cloud(path):
    return "https://" + config.get_value("cloud.host") + path
