"""Startrek client."""

import logging

from sepelib.core import config
from sepelib.yandex.startrek import (
    StartrekClient,
    Relationship,
    StartrekError,
    StartrekConnectionError,
    StartrekRequestError,
)
from walle.errors import RecoverableError

log = logging.getLogger(__name__)

MAX_TICKET_PAGE_SIZE = 100


class TicketStatus:
    OPEN: str = "open"
    CLOSED: str = "closed"


class StartrekClientError(RecoverableError):
    pass


class StartrekClientConnectionError(StartrekClientError):
    pass


class StartrekClientRequestError(StartrekClientError):
    def __init__(self, response, message, **kwargs):
        super(RecoverableError, self).__init__(message, **kwargs)
        self.response = response


class WrappedStartrekClient(StartrekClient):
    def _http_request(self, method, url, *args, **kwargs):
        headers = kwargs.pop("headers", {})
        headers.update({"Accept-Language": "en-US"})
        try:
            return super()._http_request(method, url, headers=headers, *args, **kwargs)
        except StartrekConnectionError as e:
            raise StartrekClientConnectionError(str(e), method=method, url=url, **kwargs)
        except StartrekRequestError as e:
            raise StartrekClientRequestError(e.response, str(e), method=method, url=url, **kwargs)
        except StartrekError as e:
            raise StartrekClientError(str(e), method=method, url=url, **kwargs)


def get_client():
    api = config.get_value("startrek.api")
    token = config.get_value("startrek.access_token")
    url = getattr(StartrekClient, api.upper() + "_BASE_URL")
    return WrappedStartrekClient(url, token)


def get_ticket_url_by_id(ticket_id):
    return "https://st.yandex-team.ru/{}".format(ticket_id)


def get_tickets_by_query(query):
    client = get_client()
    issues = []
    limit = MAX_TICKET_PAGE_SIZE
    page = 1

    while True:
        new_page = client.get_issues(query=query, limit=limit, page=page)
        if not new_page:
            break
        issues += new_page
        page += 1

    return issues


def get_ticket_relationships(ticket_id, client):
    try:
        return client.get_relationships(ticket_id)
    except StartrekClientError:
        return []


def link_tickets(ticket_id, tickets_to_link, relation=Relationship.RELATES, silent=False):
    """Get list of tickets and mark them all as related for the given one ticket.
    Return list of tickets that was successfully marked as related.
    If silent=True, don't raise authorization and other workflow errors (connection-level errors still be raised).
    """
    client = get_client()
    success_tickets = []
    already_linked_tickets = [rel["object"]["key"] for rel in get_ticket_relationships(ticket_id, client)]

    tickets_to_link = set(tickets_to_link) - set(already_linked_tickets)
    for related_ticket_id in tickets_to_link:
        try:
            client.add_relationship(ticket_id, relationship=relation, other_issue_id=related_ticket_id)
        except StartrekClientError as e:
            if silent:
                log.error("Failed to link ticket %s to %s: %s", related_ticket_id, ticket_id, e)
            else:
                raise
        else:
            success_tickets.append(related_ticket_id)

    return success_tickets
